/**
 * <copyright>
 *
 * Copyright (c) 2010, 2013 E.D.Willink and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     E.D.Willink - initial API and implementation
 *     E.D.Willink (CEA LIST) - Bug 388493, 399252, 399378
 *
 * </copyright>
 */
package org.eclipse.ocl.examples.pivot.uml;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.ocl.examples.domain.ids.TypeId;
import org.eclipse.ocl.examples.pivot.Annotation;
import org.eclipse.ocl.examples.pivot.Comment;
import org.eclipse.ocl.examples.pivot.ConnectionPointReference;
import org.eclipse.ocl.examples.pivot.Constraint;
import org.eclipse.ocl.examples.pivot.DataType;
import org.eclipse.ocl.examples.pivot.DynamicElement;
import org.eclipse.ocl.examples.pivot.DynamicType;
import org.eclipse.ocl.examples.pivot.Element;
import org.eclipse.ocl.examples.pivot.Enumeration;
import org.eclipse.ocl.examples.pivot.EnumerationLiteral;
import org.eclipse.ocl.examples.pivot.FinalState;
import org.eclipse.ocl.examples.pivot.NamedElement;
import org.eclipse.ocl.examples.pivot.Namespace;
import org.eclipse.ocl.examples.pivot.OpaqueExpression;
import org.eclipse.ocl.examples.pivot.Operation;
import org.eclipse.ocl.examples.pivot.Parameter;
import org.eclipse.ocl.examples.pivot.PivotFactory;
import org.eclipse.ocl.examples.pivot.PivotPackage;
import org.eclipse.ocl.examples.pivot.PrimitiveType;
import org.eclipse.ocl.examples.pivot.Profile;
import org.eclipse.ocl.examples.pivot.Property;
import org.eclipse.ocl.examples.pivot.Pseudostate;
import org.eclipse.ocl.examples.pivot.Region;
import org.eclipse.ocl.examples.pivot.State;
import org.eclipse.ocl.examples.pivot.StateMachine;
import org.eclipse.ocl.examples.pivot.Stereotype;
import org.eclipse.ocl.examples.pivot.TemplateSignature;
import org.eclipse.ocl.examples.pivot.TemplateableElement;
import org.eclipse.ocl.examples.pivot.Transition;
import org.eclipse.ocl.examples.pivot.Type;
import org.eclipse.ocl.examples.pivot.TypeTemplateParameter;
import org.eclipse.ocl.examples.pivot.TypedMultiplicityElement;
import org.eclipse.ocl.examples.pivot.ecore.Ecore2PivotDeclarationSwitch;
import org.eclipse.ocl.examples.pivot.manager.MetaModelManager;
import org.eclipse.ocl.examples.pivot.manager.TypeServer;
import org.eclipse.uml2.uml.LiteralSpecification;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLSwitch;

public class UML2PivotDeclarationSwitch extends UMLSwitch<Object>
{
	protected final Ecore2PivotDeclarationSwitch ecoreSwitch;
	protected final UML2Pivot converter;
	protected final MetaModelManager metaModelManager;
	
	public UML2PivotDeclarationSwitch(UML2Pivot converter) {
		this.converter = converter;
		this.ecoreSwitch = new Ecore2PivotDeclarationSwitch(converter);
		this.metaModelManager = converter.getMetaModelManager();
	}
	
	@Override
	public Object caseAssociation(org.eclipse.uml2.uml.Association umlAssociation) {
		assert umlAssociation != null;
		@SuppressWarnings("null") @NonNull List<org.eclipse.uml2.uml.Property> memberEnds = umlAssociation.getMemberEnds();
		converter.addProperties(memberEnds, null);
		Property firstPivotElement = null;
		for (org.eclipse.uml2.uml.Property umlProperty : memberEnds) {
			Property pivotElement = (Property) doSwitch(umlProperty);
//			converter.copyProperty(pivotElement, umlProperty, null);
//			pivotElement.setImplicit(umlProperty.getAssociation() != null);
//			converter.queueReference(umlAssociation);				// For opposite installation
			if (firstPivotElement == null) {
				firstPivotElement = pivotElement;
			}
			else {
				firstPivotElement.setOpposite(pivotElement);
				pivotElement.setOpposite(firstPivotElement);
			}
		}
		return this;
	}

	@Override
	public org.eclipse.ocl.examples.pivot.Class caseClass(org.eclipse.uml2.uml.Class umlClass) {
		assert umlClass != null;
		org.eclipse.ocl.examples.pivot.Class pivotElement = converter.refreshNamedElement(org.eclipse.ocl.examples.pivot.Class.class, PivotPackage.Literals.CLASS, umlClass);
		copyClass(pivotElement, umlClass);
		return pivotElement;
	}

	@Override
	public Object caseClassifierTemplateParameter(org.eclipse.uml2.uml.ClassifierTemplateParameter umlTemplateParameter) {
		assert umlTemplateParameter != null;
		@SuppressWarnings("null") @NonNull org.eclipse.uml2.uml.Class umlParameterClass = (org.eclipse.uml2.uml.Class) umlTemplateParameter.getParameteredElement();
		org.eclipse.ocl.examples.pivot.Class pivotElement = converter.refreshNamedElement(org.eclipse.ocl.examples.pivot.Class.class, PivotPackage.Literals.CLASS, umlParameterClass);
//		TypeTemplateParameter pivotTemplateParameter = converter.refreshNamedElement(org.eclipse.ocl.examples.pivot.Class.class, PivotPackage.Literals.CLASS, umlTemplateParameter);
//		setOriginalMapping(pivotElement, umlTemplateParameter);
//		String name = umlTemplateParameter.getName();
//		pivotElement.setName(name);
		TypeTemplateParameter typeTemplateParameter = (TypeTemplateParameter) pivotElement.getTemplateParameter();
		if (typeTemplateParameter == null) {
			typeTemplateParameter = PivotFactory.eINSTANCE.createTypeTemplateParameter();
			typeTemplateParameter.setOwnedParameteredElement(pivotElement);
			converter.setOriginalMapping(typeTemplateParameter, umlTemplateParameter);
		}
		converter.setOriginalMapping(pivotElement, umlParameterClass);
//		List<EGenericType> eBounds = umlTemplateParameter.getEBounds();
//		if (!eBounds.isEmpty()) {
//			doSwitchAll(eBounds);
//			converter.queueReference(umlTemplateParameter);
//		}
		return typeTemplateParameter;
	}

	@Override
	public Comment caseComment(org.eclipse.uml2.uml.Comment umlComment) {
		assert umlComment != null;
		Comment pivotElement = converter.refreshElement(Comment.class, PivotPackage.Literals.COMMENT, umlComment);
		pivotElement.setBody(umlComment.getBody());
		copyComments(pivotElement, umlComment);
		return pivotElement;
	}

	@Override
	public ConnectionPointReference caseConnectionPointReference(org.eclipse.uml2.uml.ConnectionPointReference umlConnectionPointReference) {
		assert umlConnectionPointReference != null;
		ConnectionPointReference pivotElement = converter.refreshNamedElement(ConnectionPointReference.class, PivotPackage.Literals.CONNECTION_POINT_REFERENCE, umlConnectionPointReference);
		copyNamedElement(pivotElement, umlConnectionPointReference);
		return pivotElement;
	}

	@Override
	public Constraint caseConstraint(org.eclipse.uml2.uml.Constraint umlConstraint) {
		assert umlConstraint != null;
		Constraint pivotElement = converter.refreshNamedElement(Constraint.class, PivotPackage.Literals.CONSTRAINT, umlConstraint);
		Object pivotSpecification = doSwitch(umlConstraint.getSpecification());
		pivotElement.setSpecification((OpaqueExpression) pivotSpecification);
		copyNamedElement(pivotElement, umlConstraint);
		if (!umlConstraint.getConstrainedElements().isEmpty()) {
			converter.queueReference(umlConstraint);	// Defer
		}
		else {
			pivotElement.getConstrainedElement().clear();
		}
		return pivotElement;
	}

	@Override
	public DataType caseDataType(org.eclipse.uml2.uml.DataType umlDataType) {
		assert umlDataType != null;
		DataType pivotElement = converter.refreshNamedElement(DataType.class, PivotPackage.Literals.DATA_TYPE, umlDataType);
		copyDataTypeOrEnum(pivotElement, umlDataType);
		@SuppressWarnings("null") @NonNull List<org.eclipse.uml2.uml.Property> umlAttributes = umlDataType.getAttributes();
		doSwitchAll(umlAttributes);
		converter.addProperties(umlAttributes, null);
		pivotElement.getSuperClass().add(metaModelManager.getOclAnyType());
		return pivotElement;
	}

	@Override
	public Enumeration caseEnumeration(org.eclipse.uml2.uml.Enumeration umlEnumeration) {
		assert umlEnumeration != null;
		Enumeration pivotElement = converter.refreshNamedElement(Enumeration.class, PivotPackage.Literals.ENUMERATION, umlEnumeration);
		copyDataTypeOrEnum(pivotElement, umlEnumeration);
		doSwitchAll(pivotElement.getOwnedLiteral(), umlEnumeration.getOwnedLiterals(), null);
//		pivotElement.getSuperClass().add(metaModelManager.getOclAnyType());
		pivotElement.getSuperClass().add(metaModelManager.getEnumerationType());
		return pivotElement;
	}

	@Override
	public EnumerationLiteral caseEnumerationLiteral(org.eclipse.uml2.uml.EnumerationLiteral umlEnumLiteral) {
		assert umlEnumLiteral != null;
		EnumerationLiteral pivotElement = converter.refreshNamedElement(EnumerationLiteral.class,
			PivotPackage.Literals.ENUMERATION_LITERAL, umlEnumLiteral);
		copyNamedElement(pivotElement, umlEnumLiteral);
//		if (eEnumLiteral.eIsSet(EcorePackage.Literals.EENUM_LITERAL__VALUE)) {
//			pivotElement.setValue(BigInteger.valueOf(eEnumLiteral.getValue()));
//		}
//		else {
//			pivotElement.eUnset(PivotPackage.Literals.ENUMERATION_LITERAL__VALUE);
//		}
//			String literal = basicGet(eObject, EcorePackage.Literals.EENUM_LITERAL__LITERAL, String.class);
//			Enumerator instance = eEnumLiteral.getInstance();
//			if (literal != null) {
/*				AnnotationCS csAnnotation = PivotFactory.eINSTANCE.createAnnotationCS();
				csAnnotation.setIdSource(EcorePackage.eNS_URI);
				DetailCS csDetail = PivotFactory.eINSTANCE.createDetailCS();
				csDetail.setIdName("literal");
				copyDetailLines(csDetail.getValue(), literal);
				csAnnotation.getDetails().add(csDetail);
				pivotElement.getAnnotations().add(csAnnotation); */
//			}
		return pivotElement;
	}

	@Override
	public FinalState caseFinalState(org.eclipse.uml2.uml.FinalState umlState) {
		assert umlState != null;
		FinalState pivotElement = converter.refreshNamedElement(FinalState.class, PivotPackage.Literals.FINAL_STATE, umlState);
		copyState(pivotElement, umlState);
		return pivotElement;
	}

	@Override
	public Object caseInstanceValue(org.eclipse.uml2.uml.InstanceValue umlInstanceValue) {
		assert umlInstanceValue != null;
		OpaqueExpression pivotElement = converter.refreshNamedElement(OpaqueExpression.class, PivotPackage.Literals.OPAQUE_EXPRESSION, umlInstanceValue);
		pivotElement.getBody().clear();
		pivotElement.getLanguage().clear();
		String umlBody = umlInstanceValue.getQualifiedName();	// FIXME escaping
		pivotElement.getBody().add(umlBody);
		copyNamedElement(pivotElement, umlInstanceValue);
		return pivotElement;
	}

	@Override
	public org.eclipse.ocl.examples.pivot.Class caseInterface(org.eclipse.uml2.uml.Interface umlInterface) {
		assert umlInterface != null;
		org.eclipse.ocl.examples.pivot.Class pivotElement = converter.refreshNamedElement(org.eclipse.ocl.examples.pivot.Class.class, PivotPackage.Literals.CLASS, umlInterface);
		pivotElement.setIsInterface(true);			
		copyClassOrInterface(pivotElement, umlInterface);
		return pivotElement;
	}

	@Override
	public Object caseLiteralSpecification(LiteralSpecification umlLiteral) {
		assert umlLiteral != null;
		OpaqueExpression pivotElement = converter.refreshNamedElement(OpaqueExpression.class, PivotPackage.Literals.OPAQUE_EXPRESSION, umlLiteral);
		pivotElement.getBody().clear();
		pivotElement.getLanguage().clear();
		String umlBody = umlLiteral.toString();
		String s = umlBody.trim();		// Trim a leading 'result=' to convert UML BodyCondition to Pivot BodyExpression
		if (s.startsWith("result")) {
			s = s.substring(6);
			s = s.trim();
			if (s.startsWith("=")) {
				s = s.substring(1);
				umlBody = s.trim();
			}
		}
		pivotElement.getBody().add(umlBody);
		copyNamedElement(pivotElement, umlLiteral);
		return pivotElement;
	}

	@Override
	public OpaqueExpression caseOpaqueExpression(org.eclipse.uml2.uml.OpaqueExpression umlExpression) {
		assert umlExpression != null;
		OpaqueExpression pivotElement = converter.refreshNamedElement(OpaqueExpression.class, PivotPackage.Literals.OPAQUE_EXPRESSION, umlExpression);
		pivotElement.getBody().clear();
		pivotElement.getLanguage().clear();
		List<String> umlBodies = umlExpression.getBodies();
		List<String> umlLanguages = umlExpression.getLanguages();
		for (int i = 0; i < umlBodies.size(); i++) {
			String umlLanguage = null;
			if (i < umlLanguages.size()) {		// languages are optional, with defaults implementation defined ==> OCL
				umlLanguage = umlLanguages.get(i);
				pivotElement.getLanguage().add(umlLanguage);
			}
			String umlBody = umlBodies.get(i);
			if ((umlLanguage == null) || umlLanguage.equals("OCL")) {
				String s = umlBody.trim();		// Trim a leading 'result=' to convert UML BodyCondition to Pivot BodyExpression
				if (s.startsWith("result")) {
					s = s.substring(6);
					s = s.trim();
					if (s.startsWith("=")) {
						s = s.substring(1);
						umlBody = s.trim();
					}
				}
			}
			pivotElement.getBody().add(umlBody);
		}
		copyNamedElement(pivotElement, umlExpression);
		return pivotElement;
	}

	@Override
	public Operation caseOperation(org.eclipse.uml2.uml.Operation umlOperation) {
		assert umlOperation != null;
		Operation pivotElement = converter.refreshNamedElement(Operation.class, PivotPackage.Literals.OPERATION, umlOperation);
//		converter.copyTypedElement(pivotElement, umlOperation, excludedAnnotations);
		List<org.eclipse.uml2.uml.Constraint> preconditions = umlOperation.getPreconditions();
		org.eclipse.uml2.uml.Constraint bodyCondition = umlOperation.getBodyCondition();
		List<org.eclipse.uml2.uml.Constraint> postconditions = umlOperation.getPostconditions();
		doSwitchAll(pivotElement.getPrecondition(), preconditions, null);
		doSwitchAll(pivotElement.getPostcondition(), postconditions, null);
		Constraint constraint = bodyCondition != null ? (Constraint) doSwitch(bodyCondition) : null;
		pivotElement.setBodyExpression(constraint != null ? constraint.getSpecification() : null);
		List<org.eclipse.uml2.uml.Constraint> exclusions;
		if ((preconditions.size() > 0) || (bodyCondition != null) || (postconditions.size() > 0)) {
			exclusions = new ArrayList<org.eclipse.uml2.uml.Constraint>();
			exclusions.addAll(preconditions);
			if (bodyCondition != null) {
				exclusions.add(bodyCondition);
			}
			exclusions.addAll(postconditions);
		}
		else {
			exclusions = Collections.emptyList();
		}
		copyNamespace(pivotElement, umlOperation, exclusions);
//		converter.copyMultiplicityElement(pivotElement, umlOperation);
		for (org.eclipse.uml2.uml.Parameter umlParameter : umlOperation.getOwnedParameters()) {
			org.eclipse.uml2.uml.ParameterDirectionKind direction = umlParameter.getDirection();
			if (direction == org.eclipse.uml2.uml.ParameterDirectionKind.IN_LITERAL) {
				Parameter pivotObject = (Parameter) doSwitch(umlParameter);
				if (pivotObject != null) {
					pivotElement.getOwnedParameter().add(pivotObject);
				}
			}
		}
		copyTemplateSignature(pivotElement, umlOperation.getOwnedTemplateSignature());
//		doSwitchAll(umlOperation.getEGenericExceptions());
		converter.queueReference(umlOperation);				// For exceptions
		return pivotElement;
	}

	@Override
	public org.eclipse.ocl.examples.pivot.Package casePackage(org.eclipse.uml2.uml.Package umlPackage) {
		assert umlPackage != null;
		org.eclipse.ocl.examples.pivot.Package pivotElement = converter.refreshNamedElement(org.eclipse.ocl.examples.pivot.Package.class, PivotPackage.Literals.PACKAGE, umlPackage);
		copyPackage(pivotElement, umlPackage);
		return pivotElement;
	}

	@Override
	public Parameter caseParameter(org.eclipse.uml2.uml.Parameter eObject) {
		assert eObject != null;
		Parameter pivotElement = converter.refreshNamedElement(Parameter.class, PivotPackage.Literals.PARAMETER, eObject);
		copyTypedMultiplicityElement(pivotElement, eObject, null);
//		converter.copyMultiplicityElement(pivotElement, eObject2);
		return pivotElement;
	}

	@Override
	public PrimitiveType casePrimitiveType(org.eclipse.uml2.uml.PrimitiveType umlPrimitiveType) {
		assert umlPrimitiveType != null;
		PrimitiveType primaryElement = null;
		String name = umlPrimitiveType.getName();
		if (TypeId.BOOLEAN_NAME.equals(name)) {
			primaryElement = metaModelManager.getBooleanType();
		}
		else if (TypeId.INTEGER_NAME.equals(name)) {
			primaryElement = metaModelManager.getIntegerType();
		}
		else if (TypeId.REAL_NAME.equals(name)) {
			primaryElement = metaModelManager.getRealType();
		}
		else if (TypeId.STRING_NAME.equals(name)) {
			primaryElement = metaModelManager.getStringType();
		}
		else if (TypeId.UNLIMITED_NATURAL_NAME.equals(name)) {
			primaryElement = metaModelManager.getUnlimitedNaturalType();
		}
//		if (pivotElement != null) {
//			converter.addCreated(umlPrimitiveType, pivotElement);
//		}
		PrimitiveType pivotElement = converter.refreshNamedElement(PrimitiveType.class, PivotPackage.Literals.PRIMITIVE_TYPE, umlPrimitiveType);
		if (primaryElement != null) {
			@SuppressWarnings("unused")TypeServer typeServer1 = metaModelManager.getTypeServer(primaryElement);
			@SuppressWarnings("unused")TypeServer typeServer2 = metaModelManager.getTypeServer(pivotElement);
			pivotElement.setBehavioralType(primaryElement);
		}
		copyClassifier(pivotElement, umlPrimitiveType);
		String instanceClassName = null;
		org.eclipse.uml2.uml.Stereotype ecoreStereotype = umlPrimitiveType.getAppliedStereotype("Ecore::EDataType");
		if (ecoreStereotype != null) {
			Object object = umlPrimitiveType.getValue(ecoreStereotype, "instanceClassName");
			if (object instanceof String) {
				instanceClassName = (String) object;
			}
		}
		pivotElement.setInstanceClassName(instanceClassName);
		return pivotElement;
	}

	@Override
	public Object caseProfile(org.eclipse.uml2.uml.Profile umlProfile) {
		assert umlProfile != null;
		Profile pivotElement = converter.refreshNamedElement(Profile.class, PivotPackage.Literals.PROFILE, umlProfile);
		copyPackage(pivotElement, umlProfile);
		return pivotElement;
	}

	@Override
	public Property caseProperty(org.eclipse.uml2.uml.Property umlProperty) {
		assert umlProperty != null;
		Property pivotElement = converter.refreshNamedElement(Property.class, PivotPackage.Literals.PROPERTY, umlProperty);
		copyProperty(pivotElement, umlProperty, null);
		// NB MDT/UML2's base_XXX/extension_YYY are spurious composites
		pivotElement.setIsComposite((umlProperty.getClass_() != null) && umlProperty.isComposite());			
		pivotElement.setImplicit(umlProperty.getClass_() == null);
//		pivotElement.setIsID(umlProperty.isID());			
//		pivotElement.setIsResolveProxies(umlProperty.isResolveProxies());			
		converter.queueReference(umlProperty);	// Defer
		org.eclipse.uml2.uml.ValueSpecification defaultValue = umlProperty.getDefaultValue();
		pivotElement.setDefaultExpression((OpaqueExpression) (defaultValue != null ? doSwitch(defaultValue) : null));
		return pivotElement;
	}

	@Override
	public Pseudostate casePseudostate(org.eclipse.uml2.uml.Pseudostate umlPseudostate) {
		assert umlPseudostate != null;
		Pseudostate pivotElement = converter.refreshNamedElement(Pseudostate.class, PivotPackage.Literals.PSEUDOSTATE, umlPseudostate);
		copyNamedElement(pivotElement, umlPseudostate);
		return pivotElement;
	}

	@Override
	public Region caseRegion(org.eclipse.uml2.uml.Region umlRegion) {
		assert umlRegion != null;
		Region pivotElement = converter.refreshNamedElement(Region.class, PivotPackage.Literals.REGION, umlRegion);
		copyNamespace(pivotElement, umlRegion, null);
		doSwitchAll(pivotElement.getSubvertex(), umlRegion.getSubvertices(), null);
		doSwitchAll(pivotElement.getTransition(), umlRegion.getTransitions(), null);
		return pivotElement;
	}

	@Override
	public State caseState(org.eclipse.uml2.uml.State umlState) {
		assert umlState != null;
		State pivotElement = converter.refreshNamedElement(State.class, PivotPackage.Literals.STATE, umlState);
		copyState(pivotElement, umlState);
		return pivotElement;
	}

	@Override
	public StateMachine caseStateMachine(org.eclipse.uml2.uml.StateMachine umlStateMachine) {
		assert umlStateMachine != null;
		StateMachine pivotElement = converter.refreshNamedElement(StateMachine.class, PivotPackage.Literals.STATE_MACHINE, umlStateMachine);
		copyClass(pivotElement, umlStateMachine);
		doSwitchAll(pivotElement.getRegion(), umlStateMachine.getRegions(), null);
		return pivotElement;
	}

	@Override
	public Stereotype caseStereotype(org.eclipse.uml2.uml.Stereotype umlStereotype) {
		assert umlStereotype != null;
		Stereotype pivotElement = converter.refreshNamedElement(Stereotype.class, PivotPackage.Literals.STEREOTYPE, umlStereotype);
		copyClass(pivotElement, umlStereotype);
		return pivotElement;
	}

	@Override
	public Transition caseTransition(org.eclipse.uml2.uml.Transition umlTransition) {
		assert umlTransition != null;
		Transition pivotElement = converter.refreshNamedElement(Transition.class, PivotPackage.Literals.TRANSITION, umlTransition);
		copyNamespace(pivotElement, umlTransition, null);
		converter.queueReference(umlTransition);	// Defer
		return pivotElement;
	}

	@Override
	public Object caseValueSpecification(org.eclipse.uml2.uml.ValueSpecification umlValueSpecification) {
		assert umlValueSpecification != null;
		OpaqueExpression pivotElement = converter.refreshNamedElement(OpaqueExpression.class, PivotPackage.Literals.OPAQUE_EXPRESSION, umlValueSpecification);
		pivotElement.getBody().clear();
		pivotElement.getLanguage().clear();
		String umlBody = "<<Unsupported " + umlValueSpecification.getClass().getSimpleName() + ">>";
		pivotElement.getBody().add(umlBody);
		copyNamedElement(pivotElement, umlValueSpecification);
		return pivotElement;
	}

	protected void copyAnnotatedElement(@NonNull NamedElement pivotElement,
			@NonNull EModelElement umlElement, @Nullable List<EAnnotation> excludedAnnotations) {
		List<Annotation> pivotAnnotations = pivotElement.getOwnedAnnotation();
		for (EAnnotation eAnnotation : umlElement.getEAnnotations()) {
			if ((excludedAnnotations == null) || !excludedAnnotations.contains(eAnnotation)) {
				Annotation pivotAnnotation = (Annotation) doSwitch(eAnnotation);
				if (pivotAnnotation != null) {
					pivotAnnotations.add(pivotAnnotation);
				}
			}
		}
	}

	protected void copyClassOrInterface(@NonNull org.eclipse.ocl.examples.pivot.Class pivotElement, @NonNull org.eclipse.uml2.uml.Classifier umlClassifier) {
		copyClassifier(pivotElement, umlClassifier);
		pivotElement.setIsAbstract(umlClassifier.isAbstract());			
		String instanceClassName = null;
		org.eclipse.uml2.uml.Stereotype ecoreStereotype = umlClassifier.getAppliedStereotype("Ecore::EClass");
		if (ecoreStereotype != null) {
			Object object = umlClassifier.getValue(ecoreStereotype, "instanceClassName");
			if (object instanceof String) {
				instanceClassName = (String) object;
			}
		}
		pivotElement.setInstanceClassName(instanceClassName);
//		doSwitchAll(umlClass.getSuperClasses());
		doSwitchAll(pivotElement.getOwnedOperation(), umlClassifier.getOperations(), null);
		@SuppressWarnings("null") @NonNull List<org.eclipse.uml2.uml.Property> umlAttributes = umlClassifier.getAttributes();
		converter.addProperties(umlAttributes, new UML2Pivot.Predicate<org.eclipse.uml2.uml.Property>()
		{
			public boolean filter(@NonNull org.eclipse.uml2.uml.Property element) {
				if (element.getAssociation() == null) {
					doSwitch(element);
				}
				return element.getAssociation() == null;
			}
		});
//		doSwitchAll(pivotElement.getOwnedAttribute(), umlClassifier.getAttributes());
		converter.queueReference(umlClassifier);				// For superclasses
	}

	protected void copyClass(@NonNull org.eclipse.ocl.examples.pivot.Class pivotElement, @NonNull org.eclipse.uml2.uml.Class umlClass) {
		pivotElement.setIsInterface(false);			
		copyClassOrInterface(pivotElement, umlClass);
		for (org.eclipse.uml2.uml.Classifier umlType : umlClass.getNestedClassifiers()) {
//			doSwitch(umlType);
			Type pivotObject = (Type) doSwitch(umlType);
			if (pivotObject != null) {
// WIP				metaModelManager.addOrphanClass(pivotObject);
			}
		}
//		doSwitchAll(pivotElement.getOwnedType(), umlClassifier.getOwnedTypes(), null);
		doSwitchAll(pivotElement.getOwnedBehavior(), umlClass.getOwnedBehaviors(), null);
	}

	protected void copyClassifier(@NonNull org.eclipse.ocl.examples.pivot.Class pivotElement, @NonNull org.eclipse.uml2.uml.Classifier umlClassifier) {
		List<org.eclipse.uml2.uml.Constraint> invariants = umlClassifier.getOwnedRules();
		doSwitchAll(pivotElement.getOwnedInvariant(), invariants, null);
		copyNamespace(pivotElement, umlClassifier, invariants);
		copyTemplateSignature(pivotElement, umlClassifier.getOwnedTemplateSignature());
	}

	protected void copyComments(@NonNull Element pivotElement, @NonNull org.eclipse.uml2.uml.Element umlElement) {
		doSwitchAll(pivotElement.getOwnedComment(), umlElement.getOwnedComments(), null);
	}

	protected void copyConstraints(@NonNull Namespace pivotElement, @NonNull org.eclipse.uml2.uml.Namespace umlNamespace,
			@Nullable List<org.eclipse.uml2.uml.Constraint> exclusions) {	
		List<org.eclipse.uml2.uml.Constraint> ownedRules = umlNamespace.getOwnedRules();
		if ((exclusions != null) && (exclusions.size() > 0)) {
			ownedRules = new ArrayList<org.eclipse.uml2.uml.Constraint>(ownedRules);
			ownedRules.removeAll(exclusions);
		}
		doSwitchAll(pivotElement.getOwnedRule(), ownedRules, null);
	}

	protected void copyDataTypeOrEnum(@NonNull DataType pivotElement, @NonNull org.eclipse.uml2.uml.DataType umlDataType) {
		copyClassifier(pivotElement, umlDataType);
		String instanceClassName = null;
		org.eclipse.uml2.uml.Stereotype ecoreStereotype = umlDataType.getAppliedStereotype("Ecore::EDataType");
		if (ecoreStereotype != null) {
			Object object = umlDataType.getValue(ecoreStereotype, "instanceClassName");
			if (object instanceof String) {
				instanceClassName = (String) object;
			}
		}
		pivotElement.setInstanceClassName(instanceClassName);
//		pivotElement.setIsSerializable(umlDataType.isSerializable());
	}

/*		public void copyDetailLines(List<String> lines, String value) {
			String[] splitLines = value.split("\n");
			for (int i = 0; i < splitLines.length-1; i++) {
				lines.add(splitLines[i] + '\n');
			}
			if (splitLines.length > 0) {
				lines.add(splitLines[splitLines.length-1]);
			}
		} */

	protected void copyModelElement(@NonNull Element pivotElement, @NonNull org.eclipse.uml2.uml.Element umlElement) {
		converter.setOriginalMapping(pivotElement, umlElement);
	}

	protected void copyNamedElement(@NonNull NamedElement pivotElement, @NonNull org.eclipse.uml2.uml.NamedElement umlNamedElement) {
		copyModelElement(pivotElement, umlNamedElement);
		String name = umlNamedElement.getName();
		pivotElement.setName(name);
		copyAnnotatedElement(pivotElement, umlNamedElement, null);
		copyComments(pivotElement, umlNamedElement);
	}
	
	protected void copyNamespace(@NonNull Namespace pivotElement, @NonNull org.eclipse.uml2.uml.Namespace umlNamespace,
			@Nullable List<org.eclipse.uml2.uml.Constraint> exclusions) {
		copyNamedElement(pivotElement, umlNamespace);
		copyConstraints(pivotElement, umlNamespace, exclusions);
	}

	protected void copyPackage(@NonNull org.eclipse.ocl.examples.pivot.Package pivotElement, @NonNull org.eclipse.uml2.uml.Package umlPackage) {
//		EAnnotation eAnnotation = umlPackage.getEAnnotation(EcorePackage.eNS_URI);
//		List<EAnnotation> exclusions = eAnnotation == null ? Collections.<EAnnotation>emptyList() : Collections.singletonList(eAnnotation);
		copyNamespace(pivotElement, umlPackage, null);
		String nsPrefix = null;
		org.eclipse.uml2.uml.Stereotype ecoreStereotype = umlPackage.getAppliedStereotype("Ecore::EPackage");
		if (ecoreStereotype != null) {
			Object object = umlPackage.getValue(ecoreStereotype, "nsPrefix");
			if (object instanceof String) {
				nsPrefix = (String) object;
			}
		}
		pivotElement.setNsPrefix(nsPrefix);
//		org.eclipse.uml2.uml.PrimitiveType umlStringType = getUMLPrimitiveType(umlPackage, "String");
//		Object object = ecoreStereotype != null ? ecoreStereotype.getAttribute("nsPrefix", null) : null;
//		if (umlPackage.eIsSet(EcorePackage.Literals.EPACKAGE__NS_PREFIX)) {
//			pivotElement.setNsPrefix(umlPackage.getNsPrefix());
//		}
//		if (umlPackage.eIsSet(EcorePackage.Literals.EPACKAGE__NS_URI)) {
		String nsURI = umlPackage.getURI();
		if ((nsURI == null) &&  (ecoreStereotype != null)) {
			Object object = umlPackage.getValue(ecoreStereotype, "nsURI");
			if (object instanceof String) {
				nsPrefix = (String) object;
			}
		}
		pivotElement.setNsURI(nsURI);
//		}
		doSwitchAll(pivotElement.getNestedPackage(), umlPackage.getNestedPackages(), null);
		doSwitchAll(pivotElement.getOwnedType(), umlPackage.getOwnedTypes(), new UML2Pivot.Predicate<org.eclipse.uml2.uml.Type>()
		{
			public boolean filter(@NonNull org.eclipse.uml2.uml.Type element) {
				return !(element instanceof org.eclipse.uml2.uml.Association);
			}
		});
		for (org.eclipse.uml2.uml.Type type : umlPackage.getOwnedTypes()) {
			if (type instanceof org.eclipse.uml2.uml.Association) {
				doSwitch(type);
			}
		}
		List<org.eclipse.uml2.uml.Package> importedPackages = umlPackage.getImportedPackages();
		if (!importedPackages.isEmpty()) {
			converter.addImportedPackages(importedPackages);
			converter.queueReference(umlPackage);	// Defer
		}
		else {
			pivotElement.getImportedPackage().clear();
		}
	}


	protected void copyProperty(@NonNull Property pivotElement, @NonNull org.eclipse.uml2.uml.Property umlProperty, List<EAnnotation> excludedAnnotations) {
		copyTypedMultiplicityElement(pivotElement, umlProperty, excludedAnnotations);
//		converter.copyMultiplicityElement(pivotElement, umlProperty);
		pivotElement.setIsReadOnly(umlProperty.isReadOnly());			
		pivotElement.setIsDerived(umlProperty.isDerived());			
//		pivotElement.setIsTransient(umlProperty.isTransient());			
//		pivotElement.setIsUnsettable(umlProperty.isUnsettable());			
//		pivotElement.setIsVolatile(umlProperty.isVolatile());			
//		if (umlProperty.eIsSet(EcorePackage.Literals.ESTRUCTURAL_FEATURE__DEFAULT_VALUE_LITERAL)) {
//			pivotElement.setDefault(eObject.getDefaultValueLiteral());
//		}
//		else {
//			pivotElement.eUnset(PivotPackage.Literals.PROPERTY__DEFAULT);
//		}
	}

	protected void copyState(@NonNull State pivotElement, @NonNull org.eclipse.uml2.uml.State umlState) {
		copyNamespace(pivotElement, umlState, null);
		doSwitchAll(pivotElement.getRegion(), umlState.getRegions(), null);
	}

	protected void copyTemplateSignature(TemplateableElement pivotElement, org.eclipse.uml2.uml.TemplateSignature umlTemplateSignature) {
		if (umlTemplateSignature != null) {
			List<org.eclipse.uml2.uml.TemplateParameter> umlTemplateParameters = umlTemplateSignature.getOwnedParameters();
			if (!umlTemplateParameters.isEmpty()) {
				TemplateSignature pivotTemplateSignature = PivotFactory.eINSTANCE.createTemplateSignature();
				pivotElement.setOwnedTemplateSignature(pivotTemplateSignature);
				doSwitchAll(pivotTemplateSignature.getOwnedParameter(), umlTemplateParameters, null);
			}
		}
	}

	protected void copyTypedMultiplicityElement(@NonNull TypedMultiplicityElement pivotElement, @NonNull org.eclipse.uml2.uml.TypedElement umlTypedElement, List<EAnnotation> excludedAnnotations) {
		copyNamedElement(pivotElement, umlTypedElement);
		int lower = ((org.eclipse.uml2.uml.MultiplicityElement)umlTypedElement).getLower();
		int upper = ((org.eclipse.uml2.uml.MultiplicityElement)umlTypedElement).getUpper();
		pivotElement.setIsRequired((upper == 1) && (lower == 1));
		org.eclipse.uml2.uml.Type umlType = umlTypedElement.getType();
		if (umlType != null) {
			converter.queueReference(umlTypedElement);
			Resource umlResource = umlType.eResource();
			if (umlResource != null) {
				converter.addImportedResource(umlResource);
			}
		}
	}

	@Override
	public Element defaultCase(EObject umlObject) {
		DynamicElement pivotElement;
		if (umlObject instanceof org.eclipse.uml2.uml.Type) {
			pivotElement = converter.refreshElement(DynamicType.class, PivotPackage.Literals.DYNAMIC_TYPE, umlObject);
			((DynamicType)pivotElement).setName(((org.eclipse.uml2.uml.Type)umlObject).getName());
		}
		else if (umlObject instanceof org.eclipse.uml2.uml.Element) {
			pivotElement = converter.refreshElement(DynamicElement.class, PivotPackage.Literals.DYNAMIC_ELEMENT, umlObject);
		}
		else {
			converter.error("Unsupported " + umlObject.eClass().getName() + " for UML2PivotDeclarationSwitch");
			return null;
		}
		EClass umlMetaClass = umlObject.eClass();
		Type metaType = metaModelManager.getPivotOfEcore(Type.class, umlMetaClass);
		pivotElement.setMetaType(metaType);
		return pivotElement;
	}

	public Object doInPackageSwitch(@NonNull EObject eObject) {
		int classifierID = eObject.eClass().getClassifierID();
		return doSwitch(classifierID, eObject);
	}

	@Override
	public Object doSwitch(EObject eObject) {
		EClass eClass = eObject.eClass();
		EPackage ePackage = eClass.getEPackage();
		if (ePackage == UMLPackage.eINSTANCE) {
			return doInPackageSwitch(eObject);
		}
		else if (ePackage == EcorePackage.eINSTANCE) {
			return ecoreSwitch.doInPackageSwitch(eObject);
		}
		else {
			converter.addStereotypeApplication(eObject);
			return null;
		}
	}

	public <T extends Element, V extends EObject> void doSwitchAll(/*@NonNull*/ Collection<T> pivotObjects, /*@NonNull*/ List<V> eObjects, @Nullable UML2Pivot.Predicate<V> predicate) {
		assert pivotObjects != null;
		assert eObjects != null;
		eObjects.size();
		for (V eObject : eObjects) {
			if ((eObject != null) && ((predicate == null) || predicate.filter(eObject))) {
				@SuppressWarnings("unchecked")
				T pivotObject = (T) doSwitch(eObject);
				if (pivotObject != null) {
					pivotObjects.add(pivotObject);
				}
			}
		}
	}

	public <T extends Element> void doSwitchAll(@NonNull List<? extends EObject> eObjects) {
		for (EObject eObject : eObjects) {
			doSwitch(eObject);
		}
	}
}
