/*******************************************************************************
 * Copyright (c) 2008, 2009 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.acceleo.engine.internal.evaluation;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.eclipse.acceleo.common.IAcceleoConstants;
import org.eclipse.acceleo.engine.AcceleoEngineMessages;
import org.eclipse.acceleo.engine.AcceleoEnginePlugin;
import org.eclipse.acceleo.engine.AcceleoEvaluationException;
import org.eclipse.acceleo.engine.event.AcceleoTextGenerationEvent;
import org.eclipse.acceleo.engine.event.IAcceleoTextGenerationListener;
import org.eclipse.acceleo.model.mtl.Block;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.Monitor;
import org.eclipse.emf.ecore.EObject;

/**
 * This will hold all necessary variables for the evaluation of an Acceleo module.
 * 
 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
 */
public final class AcceleoEvaluationContext {
	/** This will hold the system specific line separator ("\n" for unix, "\r\n" for dos, "\r" for mac, ...). */
	protected static final String LINE_SEPARATOR = System.getProperty("line.separator"); //$NON-NLS-1$

	/** Default size to be used for new buffers. */
	private static final int DEFAULT_BUFFER_SIZE = 1024;

	/** This is the tag we will look for to determine if a file has to be passed through JMerge. */
	private static final String JMERGE_TAG = "@generated"; //$NON-NLS-1$

	/** This will be populated with the list of tasks currently executing fot the creation of lost files. */
	private static final List<Future<Object>> LOST_CREATION_TASKS = new ArrayList<Future<Object>>();

	/** This pool will be used for the lost file creators. */
	private static final ExecutorService LOST_CREATORS_POOL = Executors.newFixedThreadPool(Runtime
			.getRuntime().availableProcessors());

	/** Holds the generation preview in the form of mappings filePath => fileContent. */
	protected final Map<String, Writer> generationPreview = new HashMap<String, Writer>();

	/** If <code>true</code>, no file will be generated by this context. */
	protected final boolean previewMode;

	/** This will be initialized with this generation's progress monitor. */
	private final Monitor progressMonitor;

	/**
	 * Blocks' init sections might change existing variables which will need to be restored afterwards. This
	 * will keep the altered variables values.
	 */
	private final LinkedList<Map<String, Object>> blockVariables = new LinkedList<Map<String, Object>>();

	/** References the file which is to be used as the root for all generated files. */
	private final File generationRoot;

	/** The state of his boolean will be changed while reading files prior to generation. */
	private boolean hasJMergeTag;

	/**
	 * This will hold the list of all listeners registered for notification on text generation from this
	 * engine.
	 */
	private final List<IAcceleoTextGenerationListener> listeners = new ArrayList<IAcceleoTextGenerationListener>(
			3);

	/** This will keep a reference to all user code blocks of a given File. */
	private final Map<Writer, Map<String, String>> userCodeBlocks = new HashMap<Writer, Map<String, String>>();

	/** This will hold the buffer stack. */
	private final LinkedList<Writer> writers = new LinkedList<Writer>();

	/**
	 * Instantiates an evaluation context given the root of the to-be-generated files.
	 * 
	 * @param root
	 *            Root of all files that will be generated.
	 * @param listeners
	 *            The list of all listeners that are to be notified for text generation from this context.
	 * @param preview
	 *            Tells this evaluation context it shouldn't generate any file.
	 * @param monitor
	 *            This will be used as the progress monitor for the generation.
	 */
	public AcceleoEvaluationContext(File root, List<IAcceleoTextGenerationListener> listeners,
			boolean preview, Monitor monitor) {
		generationRoot = root;
		previewMode = preview;
		this.listeners.addAll(listeners);
		if (monitor != null) {
			progressMonitor = monitor;
		} else {
			progressMonitor = new BasicMonitor();
		}
		flatten();
	}

	/**
	 * Allows clients to await for the lost file creation to end.
	 * 
	 * @throws InterruptedException
	 *             This will be thrown if the lost files creation is interrupted somehow.
	 */
	public static void awaitCompletion() throws InterruptedException {
		for (Future<Object> task : new ArrayList<Future<Object>>(LOST_CREATION_TASKS)) {
			while (!task.isDone() && !task.isCancelled()) {
				try {
					task.get();
				} catch (ExecutionException e) {
					// LostFileWriters cannot throw exceptions
				}
			}
			LOST_CREATION_TASKS.remove(task);
		}
	}

	/**
	 * Appends the given string to the last buffer of the context stack. This will notify all text generation
	 * listeners along the way.
	 * 
	 * @param string
	 *            String that is to be appended to the current buffer.
	 * @param sourceBlock
	 *            The block for which this text has been generated.
	 * @param source
	 *            The Object for which was generated this text.
	 * @param fireEvent
	 *            Tells us whether we should fire generation events.
	 * @throws AcceleoEvaluationException
	 *             Thrown if we cannot append to the current buffer.
	 */
	public void append(String string, Block sourceBlock, EObject source, boolean fireEvent)
			throws AcceleoEvaluationException {
		try {
			final Writer currentWriter = writers.getLast();
			currentWriter.append(string);
			if (fireEvent) {
				fireTextGenerated(new AcceleoTextGenerationEvent(string, sourceBlock, source));
			}
		} catch (final IOException e) {
			throw new AcceleoEvaluationException(AcceleoEngineMessages
					.getString("AcceleoEvaluationContext.AppendError"), e); //$NON-NLS-1$
		}
	}

	/**
	 * Closes the last writer of the stack and returns its result if it was a StringWriter. The empty String
	 * will be returned for FileWriters.
	 * 
	 * @return Result held by the last writer of the stack.
	 * @throws AcceleoEvaluationException
	 *             This will be thrown if the last writer of the stack cannot be flushed and closed.
	 */
	public String closeContext() throws AcceleoEvaluationException {
		final Writer last = writers.removeLast();
		final String result;
		try {
			if (last instanceof AcceleoWriterDecorator) {
				final String filePath = ((AcceleoWriterDecorator)last).getTargetPath();
				// Did we lose user code?
				final Map<String, String> lostCode = userCodeBlocks.remove(last);
				if (lostCode.size() > 0) {
					createLostFile(filePath, lostCode);
				}

				// Save the file
				last.close();
				if (previewMode) {
					generationPreview.put(filePath, last);
				}
				result = ""; //$NON-NLS-1$
			} else if (last instanceof OutputStreamWriter) {
				last.close();
				result = ""; //$NON-NLS-1$
			} else {
				// others are plain StringWriters. Close has no effect on those.
				result = last.toString();
			}
			return result;
		} catch (final IOException e) {
			throw new AcceleoEvaluationException(AcceleoEngineMessages
					.getString("AcceleoEvaluationContext.WriteError"), e); //$NON-NLS-1$
		}
	}

	/**
	 * This will be used to dispose of all created buffers and caches.
	 * 
	 * @throws AcceleoEvaluationException
	 *             Thrown if the disposal of the old writers fails.
	 */
	public void dispose() throws AcceleoEvaluationException {
		AcceleoEvaluationException exception = null;
		try {
			try {
				awaitCompletion();
			} catch (InterruptedException e) {
				exception = new AcceleoEvaluationException(AcceleoEngineMessages
						.getString("AcceleoEvaluationContext.CleanUpError"), e); //$NON-NLS-1$
			}
			try {
				for (final Writer writer : writers) {
					writer.close();
				}
			} catch (final IOException e) {
				exception = new AcceleoEvaluationException(AcceleoEngineMessages
						.getString("AcceleoEvaluationContext.CleanUpError"), e); //$NON-NLS-1$
			}
		} finally {
			LOST_CREATION_TASKS.clear();
			generationPreview.clear();
			blockVariables.clear();
			listeners.clear();
			userCodeBlocks.clear();
			writers.clear();
		}
		if (exception != null) {
			throw exception;
		}
	}

	/**
	 * Returns the preview of the generation handled by this context.
	 * 
	 * @return The generation preview.
	 */
	public Map<String, Writer> getGenerationPreview() {
		return new HashMap<String, Writer>(generationPreview);
	}

	/**
	 * This will return the last variables that were added to the stack so that they can be restored in the
	 * evaluation environment. Note that calling this method removes the returned variables from the stack.
	 * 
	 * @return The variables that were last saved.
	 */
	public Map<String, Object> getLastVariablesValues() {
		return blockVariables.removeLast();
	}

	/**
	 * This will return the current progress monitor.
	 * 
	 * @return The current progress monitor.
	 */
	public Monitor getProgressMonitor() {
		return progressMonitor;
	}

	/**
	 * This will return the content of the protected area associated with the given marker in the current
	 * context.
	 * 
	 * @param marker
	 *            Marker of the sought protected area content.
	 * @return Content of the protected area associated with the given marker. <code>null</code> if no content
	 *         can be found.
	 */
	public String getProtectedAreaContent(String marker) {
		// Seeks out the last opened file writer
		Writer writer = null;
		for (int i = writers.size() - 1; i >= 0; i--) {
			writer = writers.get(i);
			if (writer instanceof AcceleoWriterDecorator) {
				break;
			}
			writer = null;
		}

		final Map<String, String> areas = userCodeBlocks.get(writer);
		if (areas != null) {
			return areas.remove(marker);
		}
		return null;
	}

	/**
	 * Creates a new writer and appends it to the end of the stack.
	 * 
	 * @throws AcceleoEvaluationException
	 *             Thrown if the precedent buffer cannot be flushed.
	 */
	public void openNested() throws AcceleoEvaluationException {
		try {
			if (writers.size() > 0) {
				writers.getLast().flush();
			}
		} catch (final IOException e) {
			throw new AcceleoEvaluationException(AcceleoEngineMessages
					.getString("AcceleoEvaluationContext.FlushError"), e); //$NON-NLS-1$
		}
		writers.add(new StringWriter(DEFAULT_BUFFER_SIZE));
	}

	/**
	 * Create a new writer directed at the given {@link OutputStream}. This is mainly used for fileBlocks with
	 * "stdout" URI.
	 * 
	 * @param stream
	 *            Stream to which writing will be directed.
	 */
	public void openNested(OutputStream stream) {
		try {
			if (writers.size() > 0) {
				writers.getLast().flush();
			}
		} catch (final IOException e) {
			throw new AcceleoEvaluationException(AcceleoEngineMessages
					.getString("AcceleoEvaluationContext.FlushError"), e); //$NON-NLS-1$
		}
		writers.add(new OutputStreamWriter(new AcceleoFilterOutputStream(stream)));
	}

	/**
	 * Create a new writer for the file located at the given path under <tt>generationRoot</tt> and appends it
	 * to the end of the stack.
	 * <p>
	 * &quot;file&quot; schemes are handled as absolute paths and will ignore the <tt>generationRoot</tt>.
	 * </p>
	 * 
	 * @param filePath
	 *            Path of the file around which we need a FileWriter. The file will be created under the
	 *            generationRoot if needed.
	 * @param fileBlock
	 *            The file block which asked for this context. Only used for generation events.
	 * @param source
	 *            The source EObject for this file block. Only used for generation events.
	 * @param appendMode
	 *            If <code>false</code>, the file will be replaced by a new one.
	 * @throws AcceleoEvaluationException
	 *             Thrown if the file cannot be created.
	 */
	public void openNested(String filePath, Block fileBlock, EObject source, boolean appendMode)
			throws AcceleoEvaluationException {
		final File generatedFile;
		if (filePath.startsWith("file:")) { //$NON-NLS-1$
			generatedFile = new File(filePath);
		} else {
			generatedFile = new File(generationRoot, filePath);
		}
		fireFilePathComputed(new AcceleoTextGenerationEvent(generatedFile.getPath(), fileBlock, source));
		if (!previewMode && !generatedFile.getParentFile().exists()) {
			if (!generatedFile.getParentFile().mkdirs()) {
				throw new AcceleoEvaluationException(AcceleoEngineMessages.getString(
						"AcceleoEvaluationContext.FolderCreationError", generatedFile.getParentFile())); //$NON-NLS-1$
			}
		}
		try {
			if (writers.size() > 0) {
				writers.getLast().flush();
			}
			final Map<String, String> savedCodeBlocks = new HashMap<String, String>();
			if (generatedFile.exists()) {
				savedCodeBlocks.putAll(saveProtectedAreas(generatedFile));
			}
			if (generationPreview.containsKey(generatedFile.getPath())) {
				savedCodeBlocks.putAll(saveProtectedAreas(generationPreview.get(generatedFile.getPath())
						.toString()));
			}
			// We checked for JMerge tags when saving protected areas. we'll use this information here.
			final Writer writer;
			if (!previewMode && (!hasJMergeTag || appendMode)) {
				writer = new AcceleoWriterDecorator(generatedFile, appendMode);
				// Append a line separator to the new file so as to avoid writing on an existing line
				if (appendMode) {
					writer.append(LINE_SEPARATOR);
				}
			} else {
				if (appendMode && generationPreview.containsKey(generatedFile.getPath())) {
					writer = generationPreview.get(generatedFile.getPath());
					writer.append(System.getProperty("line.separator")); //$NON-NLS-1$
				} else if (!appendMode && hasJMergeTag) {
					if (!previewMode) {
						writer = new AcceleoWriterDecorator(generatedFile.getPath(), hasJMergeTag);
					} else {
						writer = generationPreview.get(generatedFile.getPath());
						((AcceleoWriterDecorator)writer).reinit();
					}
				} else {
					writer = new AcceleoWriterDecorator(generatedFile.getPath());
				}
				generationPreview.put(generatedFile.getPath(), writer);
				// reset the jmerge state for the following file blocks
				hasJMergeTag = false;
			}
			userCodeBlocks.put(writer, savedCodeBlocks);
			writers.add(writer);
		} catch (final IOException e) {
			throw new AcceleoEvaluationException(AcceleoEngineMessages.getString(
					"AcceleoEvaluationContext.FileCreationError", generatedFile.getPath()), e); //$NON-NLS-1$
		}
	}

	/**
	 * This will save the given variables in the stack.
	 * 
	 * @param vars
	 *            Variables which values will need to be restored after evaluation.
	 */
	public void saveVariableValues(Map<String, Object> vars) {
		blockVariables.add(vars);
	}

	/**
	 * Creates a .lost file that will contain the lost user code from file located at
	 * <code>originalPath</code>. As it doesn't need any more computation, the lost file creation will take
	 * place in a separate process.
	 * 
	 * @param originalPath
	 *            Absolute path of the file which user code has been fully or partially lost.
	 * @param lostAreas
	 *            Protected areas which markers couldn't be matched with markers from the template file.
	 */
	private void createLostFile(final String originalPath, final Map<String, String> lostAreas) {
		final Callable<Object> fileCreator = new LostFileWriter(originalPath, lostAreas);
		LOST_CREATION_TASKS.add(LOST_CREATORS_POOL.submit(fileCreator));
	}

	/**
	 * Notifies all listeners that a file is going to be created.
	 * 
	 * @param event
	 *            The generation event that is to be sent to registered listeners.
	 */
	private void fireFilePathComputed(AcceleoTextGenerationEvent event) {
		for (IAcceleoTextGenerationListener listener : listeners) {
			listener.filePathComputed(event);
		}
	}

	/**
	 * Notifies all listeners that text has been generated.
	 * 
	 * @param event
	 *            The generation event that is to be sent to registered listeners.
	 */
	private void fireTextGenerated(AcceleoTextGenerationEvent event) {
		for (IAcceleoTextGenerationListener listener : listeners) {
			listener.textGenerated(event);
		}
	}

	/**
	 * Used internally to remove ended tasks from the cache.
	 */
	private void flatten() {
		for (Future<Object> task : new ArrayList<Future<Object>>(LOST_CREATION_TASKS)) {
			if (task.isDone() || task.isCancelled()) {
				LOST_CREATION_TASKS.remove(task);
			}
		}
	}

	/**
	 * This will return the list of protected areas the given file contains.
	 * 
	 * @param reader
	 *            Reader which content is to be searched through for protected areas.
	 * @return The list of saved protected areas.
	 * @throws IOException
	 *             Thrown if we cannot read through the provided reader.
	 */
	private Map<String, String> internalSaveProtectedAreas(BufferedReader reader) throws IOException {
		final Map<String, String> protectedAreas = new HashMap<String, String>();
		final String usercodeStart = AcceleoEngineMessages.getString("usercode.start"); //$NON-NLS-1$
		final String usercodeEnd = AcceleoEngineMessages.getString("usercode.end"); //$NON-NLS-1$
		String line = reader.readLine();
		while (line != null) {
			if (!hasJMergeTag && line.contains(JMERGE_TAG)) {
				hasJMergeTag = true;
			}
			if (line.contains(usercodeStart)) {
				final String marker = line.substring(line.indexOf(usercodeStart) + usercodeStart.length())
						.trim();
				final StringBuffer areaContent = new StringBuffer(DEFAULT_BUFFER_SIZE);
				// Append a line separator before the protected area if need be
				if (line.indexOf(usercodeStart) - LINE_SEPARATOR.length() > 0) {
					final String previous = line.substring(line.indexOf(usercodeStart)
							- LINE_SEPARATOR.length(), line.indexOf(usercodeStart));
					if (LINE_SEPARATOR.equals(previous)) {
						areaContent.append(LINE_SEPARATOR);
					}
				}
				// Everything preceding the start of user code doesn't need to be saved
				areaContent.append(line.substring(line.indexOf(usercodeStart)));
				line = reader.readLine();
				while (line != null) {
					areaContent.append(LINE_SEPARATOR);
					if (!hasJMergeTag && line.contains(JMERGE_TAG)) {
						hasJMergeTag = true;
					}
					// Everything following the end of use code marker doesn't need to be saved
					if (line.contains(usercodeEnd)) {
						final int endOffset = line.indexOf(usercodeEnd) + usercodeEnd.length();
						areaContent.append(line.substring(0, endOffset));
						// Append a line separator after the protected area if need be
						if (endOffset + LINE_SEPARATOR.length() <= line.length()
								&& LINE_SEPARATOR.equals(line.substring(endOffset, endOffset
										+ LINE_SEPARATOR.length()))) {
							areaContent.append(LINE_SEPARATOR);
						}
						break;
					}
					areaContent.append(line);
					line = reader.readLine();
				}
				protectedAreas.put(marker, areaContent.toString());
			}
			line = reader.readLine();
		}
		return protectedAreas;
	}

	/**
	 * This will return the list of protected areas the given file contains. <b>Note</b> that we will use this
	 * occasion to look for {@value #JMERGE_TAG} throughout the file.
	 * 
	 * @param file
	 *            File which protected areas are to be saved.
	 * @return The list of saved protected areas.
	 * @throws IOException
	 *             Thrown if we cannot read through <tt>file</tt>.
	 */
	private Map<String, String> saveProtectedAreas(File file) throws IOException {
		Map<String, String> protectedAreas = new HashMap<String, String>();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new FileReader(file));
			protectedAreas = internalSaveProtectedAreas(reader);
		} catch (final FileNotFoundException e) {
			// cannot be thrown here, we were called after testing that the file indeed existed.
		} finally {
			if (reader != null) {
				reader.close();
			}
		}
		return protectedAreas;
	}

	/**
	 * This will return the list of protected areas the given String contains. <b>Note</b> that we will use
	 * this occasion to look for {@value #JMERGE_TAG} throughout the file.
	 * 
	 * @param buffer
	 *            String (file content) which protected areas are to be saved.
	 * @return The list of saved protected areas.
	 */
	private Map<String, String> saveProtectedAreas(String buffer) {
		Map<String, String> protectedAreas = new HashMap<String, String>();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new StringReader(buffer));
			protectedAreas = internalSaveProtectedAreas(reader);
		} catch (IOException e) {
			// Cannot happen here
		} finally {
			if (!previewMode && reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					// This should never happen with a String Reader
				}
			}
		}
		return protectedAreas;
	}

	/**
	 * This implementation of a FilterOutputStream will avoid closing the standard output if it is the
	 * underlying stream.
	 * 
	 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
	 */
	private final class AcceleoFilterOutputStream extends FilterOutputStream {
		/**
		 * Constructs an output stream redirecting all calls to the given {@link OutputStream}.
		 * 
		 * @param out
		 *            The decorated output stream.
		 */
		public AcceleoFilterOutputStream(OutputStream out) {
			super(out);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.FilterOutputStream#close()
		 */
		@Override
		public void close() throws IOException {
			try {
				flush();
			} catch (IOException e) {
				// Ignored exception
			}
			if (out != System.out) {
				out.close();
			}
		}
	}

	/**
	 * This implementation of a Writer will be wrapped around a FileWriter and keep a reference to its target
	 * file's absolute path. If we are in preview mode, this will simply delegate to a StringWriter. This
	 * buffer will mostly be used to support JMerge tags in generation targets.
	 * 
	 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
	 */
	private final class AcceleoWriterDecorator extends Writer {
		/** The buffer to which all calls will be delegated. */
		private Writer delegate;

		/** This will be set to true if the delegate is a file writer. */
		private final boolean isFile;

		/**
		 * If we are in preview mode and the user has JMerge tags in his file, this will be initialized with
		 * the old contents of this writer has it will be overriden by the generation.
		 */
		private String oldContent;

		/**
		 * If this is set to <code>true</code>, closing this buffer will first attempt to merge the previous
		 * file content with the to-be-generated content.
		 */
		private boolean shouldMerge;

		/** Keeps a reference to the target file's absolute path. */
		private final String targetPath;

		/**
		 * Constructs a buffered file writer around the given file.
		 * 
		 * @param target
		 *            File in which this writer will append text.
		 * @param appendMode
		 *            Tells us wether the former content of the file should be deleted.
		 * @throws IOException
		 *             Thrown if the target file doesn't exist and cannot be created.
		 */
		public AcceleoWriterDecorator(File target, boolean appendMode) throws IOException {
			delegate = new BufferedWriter(new FileWriter(target, appendMode));
			if (appendMode) {
				((BufferedWriter)delegate).newLine();
			}
			targetPath = target.getAbsolutePath();
			isFile = true;
			shouldMerge = false;
		}

		/**
		 * Constructs a buffered file writer around the given file.
		 * 
		 * @param filePath
		 *            Path of the file this writer will contain the content of.
		 */
		public AcceleoWriterDecorator(String filePath) {
			delegate = new StringWriter(DEFAULT_BUFFER_SIZE);
			targetPath = filePath;
			isFile = false;
			shouldMerge = false;
		}

		/**
		 * Constructs a buffered file writer around the given file.
		 * 
		 * @param filePath
		 *            Path of the file this writer will contain the content of.
		 * @param merge
		 *            If <code>true</code>, we'll use JMerge to merge the file content before overwriting it.
		 */
		public AcceleoWriterDecorator(String filePath, boolean merge) {
			delegate = new StringWriter(DEFAULT_BUFFER_SIZE);
			targetPath = filePath;
			isFile = true;
			shouldMerge = merge;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.BufferedWriter#close()
		 */
		@Override
		public void close() throws IOException {
			if (isFile) {
				if (!shouldMerge) {
					delegate.close();
				} else {
					// The decorated writer is a StringWriter. Closing has no effect on it
					flush();
					try {
						Class.forName("org.eclipse.emf.codegen.merge.java.JMerger"); //$NON-NLS-1$
						final Writer newDelegate = JMergeUtil.mergeFileContent(new File(targetPath),
								toString(), previewMode, oldContent);
						if (newDelegate != null) {
							delegate = newDelegate;
						}
					} catch (ClassNotFoundException e) {
						/*
						 * shouldn't happen. This would mean we are in eclipse yet org.eclipse.emf.codegen
						 * cannot be found as a dependency of the generator plugin. This shouldn't happen
						 * since it is a reexported dependency of the engine.
						 */
					}
				}
			} else {
				// closing has no effect on StringWriters, yet it does on BufferedWriters
				flush();
			}
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.Writer#flush()
		 */
		@Override
		public void flush() throws IOException {
			delegate.flush();
		}

		/**
		 * Returns the target file's path.
		 * 
		 * @return The target file's path.
		 */
		public String getTargetPath() {
			return targetPath;
		}

		/**
		 * This will be used in preview mode to reinitialize the delegate writer.
		 */
		public void reinit() {
			oldContent = toString();
			shouldMerge = true;
			delegate = new StringWriter(DEFAULT_BUFFER_SIZE);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return delegate.toString();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.Writer#write(char[], int, int)
		 */
		@Override
		public void write(char[] cbuf, int off, int len) throws IOException {
			delegate.write(cbuf, off, len);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.Writer#write(int)
		 */
		@Override
		public void write(int c) throws IOException {
			delegate.write(c);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.io.Writer#write(java.lang.String, int, int)
		 */
		@Override
		public void write(String str, int off, int len) throws IOException {
			delegate.write(str, off, len);
		}
	}

	/**
	 * This will be used to create log files.
	 * 
	 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
	 */
	private final class LostFileWriter implements Callable<Object> {
		/** Lost protected areas. */
		private final Map<String, String> lostAreas;

		/** Path to the file which protected areas have been lost. */
		private final String originalPath;

		/**
		 * Instantiate a writer given the path to the original file (will be suffixed by &quot;.lost&quot;)
		 * and a map containing the lost protected areas.
		 * 
		 * @param originalPath
		 *            Path to the file in which protected areas have been lost.
		 * @param lostAreas
		 *            Map containing the lost protected areas of this file.
		 */
		LostFileWriter(String originalPath, Map<String, String> lostAreas) {
			this.originalPath = originalPath;
			this.lostAreas = lostAreas;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.lang.Runnable#run()
		 */
		public Object call() {
			StringBuilder lostContent = new StringBuilder();
			for (final String lostAreaContent : lostAreas.values()) {
				lostContent.append(lostAreaContent);
				lostContent.append(LINE_SEPARATOR);
			}
			Writer writer = null;
			try {
				final File lostFile = new File(originalPath
						.concat(IAcceleoConstants.ACCELEO_LOST_FILE_EXTENSION));
				if (!previewMode) {
					writer = new BufferedWriter(new FileWriter(lostFile, true));
				} else {
					writer = new StringWriter(DEFAULT_BUFFER_SIZE);
					if (lostFile.exists() && lostFile.canRead()) {
						final BufferedReader lostFileReader = new BufferedReader(new FileReader(lostFile));
						String line = lostFileReader.readLine();
						while (line != null) {
							writer.append(line);
							line = lostFileReader.readLine();
						}
					}
				}
				writer.append(LINE_SEPARATOR).append(Calendar.getInstance().getTime().toString()).append(
						LINE_SEPARATOR);
				writer
						.append("================================================================================"); //$NON-NLS-1$
				writer.append(LINE_SEPARATOR);
				writer.append(lostContent);
			} catch (final IOException e) {
				final String errorMessage = AcceleoEngineMessages.getString(
						"AcceleoEvaluationContext.LostContent", originalPath, lostContent); //$NON-NLS-1$
				AcceleoEnginePlugin.log(errorMessage, false);
			} finally {
				if (writer != null) {
					try {
						if (!previewMode) {
							writer.close();
						} else {
							writer.flush();
							generationPreview.put(originalPath
									.concat(IAcceleoConstants.ACCELEO_LOST_FILE_EXTENSION), writer);
						}
					} catch (IOException e) {
						AcceleoEnginePlugin.log(e, false);
					}
				}
			}
			// This has no explicit result. Only used to await termination
			return null;
		}
	}
}
