/**
 ********************************************************************************
 * Copyright (c) 2019, 2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters.common.converter;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is responsible for converting the namespace of the AMALTHEA model.
 * It is implemented as a component factory that gets created for every
 * input/output version in ModelMigration.
 */
@Component(factory = ServiceConstants.NAMESPACE_CONVERTER_FACTORY)
public class NamespaceConverter extends AbstractConverter {

	public static final Logger LOGGER = LoggerFactory.getLogger(NamespaceConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> fileName_documentsMap, List<ICache> caches) {

		LOGGER.info("Migration from {} to {} : Executing Namespace converter for model file : {}",
				getInputModelVersion(), getOutputModelVersion(), targetFile.getName());

		Document document = fileName_documentsMap.get(targetFile);
		if (document == null) {
			return;
		}

		Element rootElement = document.getRootElement();

		ModelVersion inputVersion = ModelVersion.getModelVersion(getInputModelVersion());
		ModelVersion outputVersion = ModelVersion.getModelVersion(getOutputModelVersion());

		if (inputVersion != null && outputVersion != null) {
			HelperUtil.updateRootElement_NameSpaces(
					rootElement,
					inputVersion,
					outputVersion);
		} else {
			LOGGER.error("input model version {} or output model version {} are invalid!", getInputModelVersion(), getOutputModelVersion());
		}
	}

}
