/**
 ********************************************************************************
 * Copyright (c) 2019, 2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters094.impl;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.base.IConverter;
import org.eclipse.app4mc.amalthea.converters.common.converter.AbstractConverter;
import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(
		property = {
			ServiceConstants.INPUT_MODEL_VERSION_PROPERTY + "=0.9.3",
			ServiceConstants.OUTPUT_MODEL_VERSION_PROPERTY + "=0.9.4"
		},
		service = IConverter.class)
public class SwConverter extends AbstractConverter {

	private static final Logger LOGGER = LoggerFactory.getLogger(SwConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> fileName_documentsMap, List<ICache> caches) {

		LOGGER.info("Migration from 0.9.3 to 0.9.4 : Executing Sw converter for model file : {}", targetFile.getName());

		final Document document = fileName_documentsMap.get(targetFile);
		if (document == null) {
			return;
		}

		final Element rootElement = document.getRootElement();

		update_Runnables(rootElement);
		update_Datatypes(rootElement);

		update_ModeLabels(rootElement);
		update_ModeLabelAccesses(rootElement);

		update_ModeValueLists(rootElement);
		update_ModeConditions(rootElement);
		update_EnablingModeValueLists(rootElement);
	}

	private void update_Datatypes(final Element rootElement) {
		final String xpath ="./swModel/typeDefinitions[@xsi:type=\"am:BaseTypeDefinition\"]";

		final List<Element> types = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element baseType : types) {
			for (Element mapping : baseType.getChildren("dataMapping")) {
				String platformName = mapping.getAttributeValue("platformName");
				String platformType = mapping.getAttributeValue("platformType");

				mapping.removeAttribute("platformName");
				mapping.removeAttribute("platformType");

				mapping.setName("aliases");
				mapping.setAttribute("target", platformName);
				mapping.setAttribute("alias", platformType);
			}
		}
	}

	private void update_Runnables(final Element rootElement) {
		final String xpath = "./swModel/runnables";

		final String deadlineXpath = "./swModel/runnables/deadline";

		final List<Element> runnables = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		final List<Element> deadlines = HelperUtil.getXpathResult(
				rootElement,
				deadlineXpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		Element constraintModel = rootElement.getChild("constraintsModel");

		if (constraintModel == null && !deadlines.isEmpty()) {
			constraintModel = new Element("constraintsModel");
			rootElement.addContent(constraintModel);
		}

		if (constraintModel != null) {
			for (Element runnable : runnables) {
				Element deadline = runnable.getChild("deadline");
				String runnableName = runnable.getAttributeValue("name");
				
				if (deadline != null) {
					// create RunnableRequirement
					Element runnableReq = new Element("requirements");
					constraintModel.addContent(runnableReq);
					runnableReq.setAttribute("type", "am:RunnableRequirement", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
					runnableReq.setAttribute("name", "deadline");
					runnableReq.setAttribute("runnable", HelperUtil.encodeName(runnableName) + "?type=Runnable");
					
					Element limit = new Element("limit");
					runnableReq.addContent(limit);
					limit.setAttribute("type", "am:TimeRequirementLimit", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
					limit.setAttribute("limitType", "UpperLimit");
					limit.setAttribute("metric", "ResponseTime");
					
					Element limitValue = deadline.clone();
					limitValue.setName("limitValue");
					limit.addContent(limitValue);
					
					runnable.removeContent(deadline);
				}
			}
		}

	}

	private void update_ModeLabels(final Element rootElement) {
		String xpath;

		// convert Modes
		xpath = "./swModel/modes";

		final List<Element> modes = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element mode : modes) {
			mode.setAttribute("type", "am:EnumMode", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
		}

		// convert ModeLabels
		xpath = "./swModel/modeLabels";

		final List<Element> modeLabels = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element modeLabel : modeLabels) {

			// modify attribute (standard reference)
			String attributeValue = modeLabel.getAttributeValue("initialValue");
			if (attributeValue != null) {
				// pattern: "mode/literal?type=modeliteral"
				Pattern p = Pattern.compile("(.+)/(.+)\\?type=ModeLiteral");
				Matcher m = p.matcher(attributeValue);
				if (m.find()) {
					String mode = m.group(1);
					String literal = HelperUtil.decodeName(m.group(2));

					modeLabel.setAttribute("mode", mode + "?type=EnumMode");
					modeLabel.setAttribute("initialValue", literal);
				}
			}

			// modify content (cross file reference)
			Element element = getSingleChild(modeLabel, "initialValue");
			if (element != null) {
				String hrefValue = element.getAttributeValue("href");
				if (hrefValue != null) {
					// pattern: "amlt:/#mode/literal?type=modeliteral"
					Pattern p = Pattern.compile("amlt:/#(.+)/(.+)\\?type=ModeLiteral");
					Matcher m = p.matcher(hrefValue);
					if (m.find()) {
						String mode = m.group(1);
						String literal = HelperUtil.decodeName(m.group(2));

						element.setName("mode");
						element.setAttribute("type", "am:EnumMode", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
						element.setAttribute("href", "amlt:/#" + mode + "?type=EnumMode");

						modeLabel.setAttribute("initialValue", literal);
					}
				}
			}
		}
	}

	private void update_ModeLabelAccesses(final Element rootElement) {
		// convert mode label access in RunnableItems
		final String xpath = "./swModel/runnables//*[@xsi:type=\"am:ModeLabelAccess\"]";

		final List<Element> modeLabelAccesses = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element modeLabelAccess : modeLabelAccesses) {
			// convert access "write" to "set"
			Attribute access = modeLabelAccess.getAttribute("access");
			if (access != null) {
				if(access.getValue().equals("write")) {
					access.setValue("set");
				}
			}

			// modify attribute (standard reference)
			Attribute modeValue = modeLabelAccess.getAttribute("modeValue");
			if (modeValue != null) {
				modeValue.setName("value");
				modeValue.setValue(extractAndDecodeLiteral(modeValue.getValue()));
			}

			// modify content (cross file reference)
			Element element = getSingleChild(modeLabelAccess, "modeValue");
			if (element != null) {
				String hrefValue = element.getAttributeValue("href");
				if (hrefValue != null) {
					modeLabelAccess.setAttribute("value", extractAndDecodeLiteral(hrefValue));
					modeLabelAccess.removeChild("modeValue");
				}
			}
		}
	}

	private void update_ModeValueLists(final Element rootElement) {
		// convert mode assignments in mode value lists
		final String xpath ="./stimuliModel/stimuli/setModeValueList/entries";

		final List<Element> assignments = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element assignment : assignments) {
			convertModeValueContents(assignment);
		}
	}

	private void update_ModeConditions(final Element rootElement) {
		// handle mode value
		final String xpath =
			"./swModel/runnables//*[@xsi:type=\"am:RunnableModeSwitch\"]/entries/condition/entries"
		+ "|./swModel/tasks//*[@xsi:type=\"am:ModeSwitch\"]/entries/condition/entries"
		+ "|./stimuliModel/stimuli/enablingModeValueList/entries";

		final List<Element> conditionDisjunctionEntries = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element disjunctionEntry : conditionDisjunctionEntries) {
			String entryType = disjunctionEntry.getAttributeValue("type", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

			if (entryType.equals("am:ModeValue")) {
				convertModeValue(disjunctionEntry, true);
			}
			else if (entryType.equals("am:ModeValueConjunction")) {
				disjunctionEntry.setAttribute("type", "am:ModeConditionConjunction", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
				for (Element modeValue : disjunctionEntry.getChildren("entries")) {
					convertModeValue(modeValue, false);
				}
			}
		}
	}

	private void convertModeValue(final Element modeValueElement, boolean setType) {
		if (setType) {
			modeValueElement.setAttribute("type", "am:ModeCondition", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
		}

		modeValueElement.setAttribute("relation", "EQUAL");
		convertModeValueContents(modeValueElement);
	}

	private void convertModeValueContents(final Element modeValueElement) {

		// 1. rename valueProvider to label

		// - modify attribute (standard reference)
		Attribute providerAttribute = modeValueElement.getAttribute("valueProvider");
		if (providerAttribute != null) {
			providerAttribute.setName("label");
		}

		// - modify content (cross file reference)
		Element providerElement = getSingleChild(modeValueElement, "valueProvider");
		if (providerElement != null) {
			providerElement.setName("label");
		}

		// 2. extract and set value

		// modify attribute (standard reference)
		String refValue = modeValueElement.getAttributeValue("value");
		if (refValue != null) {
			modeValueElement.setAttribute("value", extractAndDecodeLiteral(refValue));
		}

		// modify content (cross file reference)
		Element valueElement = getSingleChild(modeValueElement, "value");
		if (valueElement != null) {
			String hrefValue = valueElement.getAttributeValue("href");
			if (hrefValue != null) {
				modeValueElement.setAttribute("value", extractAndDecodeLiteral(hrefValue));
				modeValueElement.removeChild("value");
			}
		}
	}

	private void update_EnablingModeValueLists(final Element rootElement) {
		final String xpath = "./stimuliModel/stimuli";

		final List<Element> stimuli = HelperUtil.getXpathResult(
				rootElement,
				xpath,
				Element.class,
				AmaltheaNamespaceRegistry.getNamespace(ModelVersion._094, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element stimulus : stimuli) {
			final String stimulusName = stimulus.getAttributeValue("name");
			// rename enabling list
			Element enablingList = getSingleChild(stimulus, "enablingModeValueList");
			if(enablingList != null) {
				enablingList.setName("executionCondition");
			}

			// remove disabling list
			Element disablingList = getSingleChild(stimulus, "disablingModeValueList");
			if (disablingList != null) {
				LOGGER.warn("From Stimulus : {}, disablingModeValueList element is removed (as this element is no longer supported by AMALTHEA meta model from 0.9.4 version) ", stimulusName);
				stimulus.removeChild("disablingModeValueList");
			}
		}
	}


	// helper methods

	private String extractAndDecodeLiteral(String modeValueReference) {
		String value = modeValueReference;
		Pattern p = Pattern.compile(".*/(.+?)\\?type=ModeLiteral");
		Matcher m = p.matcher(value);
		if(m.find()) {
			return HelperUtil.decodeName(m.group(1));
		}

		return modeValueReference;
	}

	private Element getSingleChild(final Element parent, String name) {
		List<Element> list = parent.getChildren(name);
		if (list.size() == 1) {
			return list.get(0);
		} else {
			return null;
		}
	}

}
