/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters.ui.dialog;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.app4mc.amalthea.converters.common.MigrationInputFile;
import org.eclipse.app4mc.amalthea.converters.common.MigrationProcessor;
import org.eclipse.app4mc.amalthea.converters.common.MigrationSettings;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.IMigrationJobConstants;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.MigrationJobChangeListener;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.ModelMigrationJob;
import org.eclipse.app4mc.amalthea.converters.ui.providers.MigrationInputDataProvider;
import org.eclipse.app4mc.amalthea.converters.ui.providers.StyledLabelProvider;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.beans.typed.PojoProperties;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.databinding.swt.ISWTObservableValue;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

public class ModelMigrationDialog extends Dialog {

	private MigrationProcessor migrationProcessor;
	private MigrationSettings migrationSettings;
	private IProject iProject;

	public ModelMigrationDialog(Shell parentShell, MigrationProcessor migrationProcessor, MigrationSettings settings, IProject iProject) {
		super(parentShell);
		setShellStyle(SWT.APPLICATION_MODAL | SWT.DIALOG_TRIM | SWT.RESIZE);

		if (migrationProcessor == null) {
			throw new IllegalStateException("MigrationProcessor cannot be null");
		}

		this.migrationProcessor = migrationProcessor;

		if (settings == null) {
			this.migrationSettings = new MigrationSettings();
		} else {
			this.migrationSettings = settings;
		}

		this.iProject = iProject;
	}

	@Override
	protected void configureShell(final Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("AMALTHEA Model Migration");
	}

	@Override
	protected Point getInitialSize() {
		return new Point(600, 600);
	}

	@Override
	protected Control createDialogArea(final Composite parent) {

		parent.setToolTipText("AMALTHEA Model Migration");

		Composite parentComposite = (Composite) super.createDialogArea(parent);
		GridLayout gridLayout = (GridLayout) parentComposite.getLayout();
		gridLayout.numColumns = 1;

		Group grpInitialModel = new Group(parentComposite, SWT.NONE);
		grpInitialModel.setText("AMALTHEA Models");
		grpInitialModel.setLayout(new GridLayout(2, false));
		GridDataFactory
			.fillDefaults()
			.grab(true, true)
			.applyTo(grpInitialModel);

		Composite tableComposite = createFileTableViewer(grpInitialModel);
		GridDataFactory
			.fillDefaults()
			.grab(true, true)
			.span(2, 1)
			.applyTo(tableComposite);

		Label lblInputModelsIdentificationInfo = new Label(grpInitialModel, SWT.NONE);
		lblInputModelsIdentificationInfo.setText("* Blue color: Selected Model files. Black color: Model scope files");
		GridDataFactory
			.swtDefaults()
			.align(SWT.FILL, SWT.CENTER)
			.grab(false, false)
			.span(2, 1)
			.applyTo(lblInputModelsIdentificationInfo);

		Label lblModelVersion = new Label(grpInitialModel, SWT.NONE);
		lblModelVersion.setText("Model Version: ");
		GridDataFactory
			.swtDefaults()
			.align(SWT.LEFT, SWT.CENTER)
			.grab(false, false)
			.applyTo(lblModelVersion);

		Text txtInputModelVersion = new Text(grpInitialModel, SWT.BORDER);
		txtInputModelVersion.setEnabled(false);
		txtInputModelVersion.setEditable(false);
		txtInputModelVersion.setText(this.migrationSettings.getInputModelVersion());
		GridDataFactory
			.swtDefaults()
			.align(SWT.LEFT, SWT.CENTER)
			.grab(true, false)
			.applyTo(txtInputModelVersion);

		final Group grpMigrationModels = new Group(parentComposite, SWT.NONE);
		grpMigrationModels.setText("Migration details");
		grpMigrationModels.setLayout(new GridLayout(3, false));
		GridDataFactory
			.fillDefaults()
			.grab(true, false)
			.applyTo(grpMigrationModels);

		Label migModelVersionText = new Label(grpMigrationModels, SWT.NONE);
		migModelVersionText.setText("Model Version");
		GridDataFactory
			.fillDefaults()
			.grab(false, false)
			.applyTo(migModelVersionText);

		Combo migModelVersionCombo = new Combo(grpMigrationModels, SWT.READ_ONLY);
		migModelVersionCombo.select(0);
		GridDataFactory
			.swtDefaults()
			.grab(false, false)
			.span(2, 1)
			.applyTo(migModelVersionCombo);

		String version = this.migrationSettings.getInputModelVersion();
		if (version != null) {
			List<String> allSupportedVersions = ModelVersion.getAllSupportedVersions();
			int totalVersionSize = allSupportedVersions.size();

			int startIndex = allSupportedVersions.indexOf(version);
			if (startIndex != -1) {
				List<String> subList = allSupportedVersions.subList(startIndex + 1, totalVersionSize);
				Collections.reverse(subList);
				migModelVersionCombo.setItems(subList.toArray(new String[] {}));

				// TODO select the model version currently supported in the installed Amalthea editor
				migModelVersionCombo.select(0);
			}
		}

		Label lblOutputDirectory = new Label(grpMigrationModels, SWT.NONE);
		lblOutputDirectory.setText("Output directory");
		GridDataFactory
			.fillDefaults()
			.grab(false, false)
			.applyTo(lblOutputDirectory);

		Text txtOutputDirectory = new Text(grpMigrationModels, SWT.BORDER);
		txtOutputDirectory.setEditable(false);
		txtOutputDirectory.setTouchEnabled(true);
		GridDataFactory
			.fillDefaults()
			.grab(true, false)
			.applyTo(txtOutputDirectory);


		// Added SWT.Modify listener to Checkbox (for selection of output model version) - this is work around for JFace databinding issue reported on MAC
		migModelVersionCombo.addListener(SWT.Modify, new Listener() {

			@Override
			public void handleEvent(final Event event) {
				String selection_outputModelVersion = migModelVersionCombo.getText();
				if (selection_outputModelVersion != null && selection_outputModelVersion.length() > 0) {
					ModelMigrationDialog.this.migrationSettings.setMigrationModelVersion(selection_outputModelVersion);
				}
			}
		});

		final Button btnBrowse = new Button(grpMigrationModels, SWT.NONE);
		btnBrowse.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {

				final DirectoryDialog dialog = new DirectoryDialog(e.display.getActiveShell());

				dialog.setFilterPath(ModelMigrationDialog.this.migrationSettings.getProject().getAbsolutePath());

				final String selectedPath = dialog.open();

				if (selectedPath != null) {
					ModelMigrationDialog.this.migrationSettings.setOutputDirectoryLocation(selectedPath);
					txtOutputDirectory.setText(selectedPath);
				}
			}
		});
		btnBrowse.setText("Browse...");
		GridDataFactory
			.swtDefaults()
			.grab(false, false)
			.applyTo(btnBrowse);

		initDataBindings(txtOutputDirectory, migModelVersionCombo);

		return parentComposite;
	}

	private Composite createFileTableViewer(Composite parent) {
		Composite tableComposite = new Composite(parent, SWT.NONE);

		TableColumnLayout tableColumnLayout = new TableColumnLayout();
		tableComposite.setLayout(tableColumnLayout);

		TableViewer tableViewer = new TableViewer(tableComposite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.FULL_SELECTION);
		Table table = tableViewer.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		table.getHorizontalBar().setEnabled(true);

		TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		TableColumn column = viewerColumn.getColumn();
		column.setText("Relative file paths");

		tableColumnLayout.setColumnData(viewerColumn.getColumn(), new ColumnWeightData(100, false));

		MigrationInputDataProvider provider = new MigrationInputDataProvider(new StyledLabelProvider());
		tableViewer.setContentProvider(provider);
		tableViewer.setLabelProvider(provider);
		this.migrationSettings.getMigModelFiles().sort(new Comparator<MigrationInputFile>() {

			@Override
			public int compare(final MigrationInputFile o1, final MigrationInputFile o2) {
				if (o1.isSelectedFile()) {
					return -1;
				}
				return 1;
			}
		});
		tableViewer.setInput(this.migrationSettings.getMigModelFiles());

		return tableComposite;
	}

	/**
	 * Create contents of the button bar.
	 *
	 * @param parent
	 */
	@Override
	protected void createButtonsForButtonBar(final Composite parent) {
		Button migrateModelsButton = createButton(parent, IDialogConstants.OK_ID, "Migrate Models", true);
		migrateModelsButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				 //perform model migration
				ModelMigrationJob migrationJob = new ModelMigrationJob(
						"AMALTHEA Model Migration",
						migrationProcessor,
						migrationSettings);

				migrationJob.setUser(true);
				migrationJob.schedule();
				migrationJob.addJobChangeListener(new MigrationJobChangeListener(getShell(), iProject));
			}
		});

		Button cancelMigrationButton = createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
		cancelMigrationButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				super.widgetSelected(e);
				Job.getJobManager().cancel(IMigrationJobConstants.FAMILY);
			}
		});
	}

	protected DataBindingContext initDataBindings(Text txtOutputDirectory, Combo migModelVersionCombo) {
		DataBindingContext bindingContext = new DataBindingContext();

		ISWTObservableValue<String> observeTextTxtOutputDirectoryObserveWidget =
				WidgetProperties.text(SWT.Modify).observe(txtOutputDirectory);
		ISWTObservableValue<?> observeSelectionMig_model_version_comboObserveWidget =
				WidgetProperties.comboSelection().observe(migModelVersionCombo);

		IObservableValue<?> outputDirectoryLocationMigDataModelObserveValue =
			PojoProperties.value("outputDirectoryLocation").observe(this.migrationSettings);
		bindingContext.bindValue(observeTextTxtOutputDirectoryObserveWidget,
				outputDirectoryLocationMigDataModelObserveValue, null, null);

		IObservableValue<?> migrationModelVersionMigrationSettingsObserveValue =
			PojoProperties.value("migrationModelVersion").observe(this.migrationSettings);
		bindingContext.bindValue(observeSelectionMig_model_version_comboObserveWidget,
				migrationModelVersionMigrationSettingsObserveValue, null, null);

		return bindingContext;
	}

}
