/**
 ********************************************************************************
 * Copyright (c) 2018-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters093.impl;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.base.IConverter;
import org.eclipse.app4mc.amalthea.converters.common.converter.AbstractConverter;
import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.eclipse.app4mc.amalthea.converters093.utils.HelperUtils_092_093;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(
		property = {
			ServiceConstants.INPUT_MODEL_VERSION_PROPERTY + "=0.9.2",
			ServiceConstants.OUTPUT_MODEL_VERSION_PROPERTY + "=0.9.3"
		},
		service = IConverter.class)
public class StimulusConverter extends AbstractConverter {

	private static final Logger LOGGER = LoggerFactory.getLogger(StimulusConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> fileName_documentsMap, List<ICache> caches) {

		LOGGER.info(
				"Migration from 0.9.2 to 0.9.3 : Executing Stimulus converter for model file : {}", targetFile.getName());

		final Document root = fileName_documentsMap.get(targetFile);
		if (root == null) return;

		final Element rootElement = root.getRootElement();
		update_Stimuli(rootElement);

	}

	private void update_Stimuli(final Element rootElement) {
		final StringBuilder xpathBuffer = new StringBuilder();
		xpathBuffer.append("./stimuliModel/stimuli");

		final List<Element> stimuliElements = HelperUtil.getXpathResult(rootElement, xpathBuffer.toString(),
				Element.class, AmaltheaNamespaceRegistry.getNamespace(ModelVersion._093, "am"),
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element stimuliElement : stimuliElements) {

			String stimuliType = stimuliElement.getAttributeValue("type",
					AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

			if (stimuliType != null) {
				if (stimuliType.equals("am:PeriodicStimulus")) {

					Element jitter_time_deviationElement = stimuliElement.getChild("jitter");

					if (jitter_time_deviationElement != null) {

						Element migratedElement = HelperUtils_092_093.migrateDeviationElement_Containing_TimeValue(jitter_time_deviationElement, "jitter");

						int indexOf = stimuliElement.indexOf(jitter_time_deviationElement);

						stimuliElement.removeContent(jitter_time_deviationElement);

						if (migratedElement != null) {
							// adding migrated element to the stimuli element
							stimuliElement.addContent(indexOf, migratedElement);
						}
					}

				} else if (stimuliType.equals("am:RelativePeriodicStimulus")) {

					Element nextOccurrence_time_deviationElement = stimuliElement.getChild("nextOccurrence");

					if (nextOccurrence_time_deviationElement != null) {

						Element migratedElement = HelperUtils_092_093.migrateDeviationElement_Containing_TimeValue(nextOccurrence_time_deviationElement, "nextOccurrence");

						int indexOf = stimuliElement.indexOf(nextOccurrence_time_deviationElement);

						stimuliElement.removeContent(nextOccurrence_time_deviationElement);

						if(migratedElement!=null) {
							//adding migrated element to the stimuli element
							stimuliElement.addContent(indexOf, migratedElement);

						}
					}



				}else if(stimuliType.equals("am:VariableRateStimulus")) {

					Element occurrencesPerStep_double_deviationElement = stimuliElement.getChild("occurrencesPerStep");

					if(occurrencesPerStep_double_deviationElement!=null) {

						Element migratedElement = HelperUtils_092_093.migrateDeviationElement_Containing_DoubleValue(occurrencesPerStep_double_deviationElement, "occurrencesPerStep");

						int indexOf = stimuliElement.indexOf(occurrencesPerStep_double_deviationElement);

						stimuliElement.removeContent(occurrencesPerStep_double_deviationElement);

						if(migratedElement!=null) {
							//adding migrated element to the stimuli element
							stimuliElement.addContent(indexOf, migratedElement);
						}

					}

					Element maxIncreasePerStepElement = stimuliElement.getChild("maxIncreasePerStep");

					Element maxDecreasePerStepElement = stimuliElement.getChild("maxDecreasePerStep");

					//As DoubleObject is changed to Double (wrapper class) in model. Instead of child elements, xmi will have attributes for the below two parameters
					if(maxIncreasePerStepElement!=null) {
						stimuliElement.setAttribute("maxIncreasePerStep", maxIncreasePerStepElement.getAttributeValue("value"));
						stimuliElement.removeContent(maxIncreasePerStepElement);
					}

					if(maxDecreasePerStepElement!=null) {
						stimuliElement.setAttribute("maxDecreasePerStep", maxDecreasePerStepElement.getAttributeValue("value"));
						stimuliElement.removeContent(maxDecreasePerStepElement);
					}
				}
			}
		}
	}
}
