/**
* *******************************************************************************
* Copyright (c) 2019-2021 Robert Bosch GmbH.
* 
* This program and the accompanying materials are made
* available under the terms of the Eclipse Public License 2.0
* which is available at https://www.eclipse.org/legal/epl-2.0/
* 
* SPDX-License-Identifier: EPL-2.0
* 
* Contributors:
*     Robert Bosch GmbH - initial API and implementation
* *******************************************************************************
*/

package org.eclipse.app4mc.transformation.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import org.eclipse.app4mc.util.sessionlog.SessionLogger;

import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class OutputBuffer {

	SessionLogger logger;

	private final HashMap<String, TypedBuffer> filetypeToBufferMap = new HashMap<>();

	private String outputFolder;
	
	private List<Path> excludePathsOfResult = Collections.emptyList();

	@Inject
	public void setSessionLogger(SessionLogger logger) {
		this.logger = logger;
	}
	
	public void initialize(String outputFolder) {
		this.outputFolder = outputFolder;
	}

	public void configureFiletype(String type, String fileExtension, String header, String footer) {
		if (type == null || type.isEmpty())
			throw new IllegalArgumentException("Illegal type string");

		if (filetypeToBufferMap.containsKey(type))
			throw new IllegalArgumentException("Type is already configured");

		filetypeToBufferMap.put(type, new TypedBuffer(fileExtension, header, footer));
	}

	/**
	 * @return: true if new file handle has been created
	 */
	public boolean appendTo(String type, String targetFile, String content) {
		if (! filetypeToBufferMap.containsKey(type))
			throw new IllegalArgumentException("Unknown type");

		if (targetFile == null || targetFile.isEmpty())
			throw new IllegalArgumentException("Undefined filename");


		return filetypeToBufferMap.get(type).appendTo(targetFile, content);
	}

	public void finish() {
		finish(true, false);
	}
	
	public void finish(boolean createZip, boolean writeChangedFilesOnly) {
		
		// Write files to disk
		for (TypedBuffer typedBuffer : filetypeToBufferMap.values()) {

			for (Entry<String, StringBuilder> entry : typedBuffer.filenameToBufferMap.entrySet()) {
				String targetFile = entry.getKey();
				StringBuilder buffer = entry.getValue();

				StringBuilder builder = new StringBuilder(
						typedBuffer.header.length() + buffer.length() + typedBuffer.footer.length());
				builder.append(typedBuffer.header);
				builder.append(buffer);
				builder.append(typedBuffer.footer);
				String newContent = builder.toString();

				File file = getFile(outputFolder, targetFile + typedBuffer.fileExtension);

				if (writeChangedFilesOnly && contentEquals(file, newContent))
					continue;

				try (FileWriter fw = new FileWriter(file); BufferedWriter writer = new BufferedWriter(fw);) {
					writer.append(newContent);
					writer.flush();
				} catch (IOException e) {
					logger.error("Failed to write result file: {0}", targetFile);
				}
			}
		}

		if (createZip) {
			// Create Zip archive
			try {
				FileHelper.zipResult(outputFolder,excludePathsOfResult, logger);
			} catch (Exception e) {
				logger.error("Failed to produce result zip archive: ", e);
			}
		}
	}
	
	private boolean contentEquals(File file, String newContent) {
		if (!file.exists() || !file.canRead() || file.length() != newContent.getBytes().length)
			return false;

		try (FileReader fr = new FileReader(file); BufferedReader reader = new BufferedReader(fr);) {
			// read the complete file
			char[] fileContent = new char[newContent.length()];
			int fileLength = reader.read(fileContent);

			if (fileLength != newContent.length())
				return false;

			return newContent.equals(String.valueOf(fileContent));

		} catch (IOException e1) {
			logger.error("Failed to read File {0}, which should be present", file.getPath(), logger);
		}
		return false;
	}

	/**
	 * This method will get the File object by creating all the parent directories
	 * of the file
	 */
	private File getFile(final String outputFolder, final String targetFile) {
		final File file = new File(outputFolder, targetFile);
		final File parentFile = file.getParentFile();
	
		if ( ! parentFile.exists())
			parentFile.mkdirs();
	
		return file;
	}

	public String getOutputFolder() {
		return outputFolder;
	}

	public String getFileExtension(String type) {
		TypedBuffer typedBuffer = filetypeToBufferMap.get(type);
		if (typedBuffer == null)
			return null;

		return typedBuffer.getFileExtension();
	}
	/**
	 * @return: true if buffer has already been created
	 */
	public boolean bufferExists(String type, String targetFile) {
		TypedBuffer typedBuffer = filetypeToBufferMap.get(type);

		return typedBuffer != null && typedBuffer.filenameToBufferMap.containsKey(targetFile);
	}
	
	public void setExcludePathsOfResult(List<Path> excludePathsOfResult) {
		this.excludePathsOfResult = excludePathsOfResult;
	}

	private class TypedBuffer {
		private final HashMap<String, StringBuilder> filenameToBufferMap = new HashMap<>();

		private String fileExtension = "";
		private String header = "";
		private String footer = "";

		public TypedBuffer(String fileExtension, String header, String footer) {
			super();
			if  (fileExtension != null) {
				this.fileExtension = fileExtension;				
			}
			if  (header != null) {
				this.header = header;				
			}
			if  (footer != null) {
				this.footer = footer;				
			}
		}

		/**
		 * @return: true if new file buffer has been created
		 */
		public boolean appendTo(String targetFile, String content) {
			boolean isNewBuffer = false;
			if (!filenameToBufferMap.containsKey(targetFile)) {
				filenameToBufferMap.put(targetFile, new StringBuilder());
				isNewBuffer = true;
			}
			
			filenameToBufferMap.get(targetFile).append(content);
			
			return isNewBuffer;
		}

		public String getFileExtension() {
			return fileExtension;
		}
	}

}
