/**
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 */
package org.eclipse.app4mc.slg.ros2.generators;

import com.google.inject.Singleton;
import java.util.Set;
import org.eclipse.xtend2.lib.StringConcatenation;

@Singleton
@SuppressWarnings("all")
public class RosModel2TextGenerator {
  public static String toLaunchFile(final Set<String> nodes) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("from launch import LaunchDescription");
    _builder.newLine();
    _builder.append("from launch_ros.actions import Node");
    _builder.newLine();
    _builder.newLine();
    _builder.append("def generate_launch_description():");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("return LaunchDescription([");
    _builder.newLine();
    {
      for(final String node : nodes) {
        _builder.append("\t\t");
        _builder.append("Node(");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("\t");
        _builder.append("package=\'amalthea_ros_model\',");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("\t");
        _builder.append("node_namespace=\'\',");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("\t");
        _builder.append("node_executable=\'");
        _builder.append(node, "\t\t\t");
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("),");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.append("])");
    _builder.newLine();
    return _builder.toString();
  }
  
  public static String toBuildScript(final Set<String> services) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final String service : services) {
        _builder.append("#building service ");
        _builder.append(service);
        _builder.newLineIfNotEmpty();
        _builder.newLine();
        _builder.append("cd services/");
        _builder.append(service);
        _builder.newLineIfNotEmpty();
        _builder.append("colcon build");
        _builder.newLine();
        _builder.append(". install/setup.bash");
        _builder.newLine();
        _builder.append("cd ../..");
        _builder.newLine();
        _builder.newLine();
      }
    }
    _builder.append("colcon build");
    _builder.newLine();
    _builder.append(". install/setup.bash");
    _builder.newLine();
    return _builder.toString();
  }
  
  public static String toPackageXml(final Set<String> services) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<?xml version=\"1.0\"?>");
    _builder.newLine();
    _builder.append("<?xml-model href=\"http://download.ros.org/schema/package_format3.xsd\" schematypens=\"http://www.w3.org/2001/XMLSchema\"?>");
    _builder.newLine();
    _builder.append("<package format=\"3\">");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<name>amalthea_ros_model</name>");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<version>0.0.0</version>");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<description>TODO: Package description</description>");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<maintainer email=\"fixedterm.Patrick.Will@de.bosch.com\">wlp8fe</maintainer>");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<license>TODO: License declaration</license>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<buildtool_depend>ament_cmake</buildtool_depend>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<depend>rclcpp</depend>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<depend>std_msgs</depend>");
    _builder.newLine();
    _builder.newLine();
    {
      for(final String service : services) {
        _builder.append("<depend>");
        _builder.append(service);
        _builder.append("</depend>");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<test_depend>ament_lint_auto</test_depend>");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("<test_depend>ament_lint_common</test_depend>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  ");
    _builder.append("<export>");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("<build_type>ament_cmake</build_type>");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("</export>");
    _builder.newLine();
    _builder.append("</package>");
    _builder.newLine();
    return _builder.toString();
  }
  
  public static String toCmake(final Set<String> tags, final Set<String> services, final boolean label, final boolean ticks, final boolean interprocess, final boolean channelSend, final boolean performance_measurement) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# CMakeLists for Nodes");
    _builder.newLine();
    _builder.append("cmake_minimum_required(VERSION 3.5)");
    _builder.newLine();
    _builder.append("project(amalthea_ros_model)");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# Default to C++14");
    _builder.newLine();
    _builder.append("if(NOT CMAKE_CXX_STANDARD)");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("set(CMAKE_CXX_STANDARD 14)");
    _builder.newLine();
    _builder.append("endif()");
    _builder.newLine();
    _builder.newLine();
    _builder.append("if(CMAKE_COMPILER_IS_GNUCXX OR CMAKE_CXX_COMPILER_ID MATCHES \"Clang\")");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("add_compile_options(-Wall -Wextra -Wpedantic)");
    _builder.newLine();
    _builder.append("endif()");
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    _builder.append("find_package(ament_cmake REQUIRED)");
    _builder.newLine();
    _builder.append("find_package(rclcpp REQUIRED)");
    _builder.newLine();
    _builder.append("find_package(std_msgs REQUIRED)");
    _builder.newLine();
    _builder.newLine();
    {
      for(final String service : services) {
        _builder.append("find_package(");
        _builder.append(service);
        _builder.append(" REQUIRED)");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    {
      if (channelSend) {
        _builder.append("add_library(CHANNELSEND_UTILS  STATIC");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("synthetic_gen/channelSendUtils/_src/channelSendUtils.cpp");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
        _builder.newLine();
        _builder.append("ament_target_dependencies(CHANNELSEND_UTILS rclcpp std_msgs)");
        _builder.newLine();
        _builder.newLine();
        _builder.newLine();
        _builder.append("target_include_directories(CHANNELSEND_UTILS");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("PUBLIC synthetic_gen/channelSendUtils/_inc/");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    {
      if (interprocess) {
        _builder.append("add_library(INTERPROCESSTRIGGER_UTIL STATIC");
        _builder.newLine();
        _builder.append("\t  ");
        _builder.append("synthetic_gen/interProcessTriggerUtils/_src/interProcessTriggerUtils.cpp");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
        _builder.append("ament_target_dependencies(INTERPROCESSTRIGGER_UTIL rclcpp ");
        {
          for(final String service_1 : services) {
            _builder.append(service_1);
            _builder.append(" ");
          }
        }
        _builder.append(")");
        _builder.newLineIfNotEmpty();
        _builder.append("target_include_directories(INTERPROCESSTRIGGER_UTIL");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("PUBLIC  synthetic_gen/interProcessTriggerUtils/_inc");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      if (performance_measurement) {
        _builder.append("add_library(AML_LIB  STATIC");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("utils/aml/src/aml.cpp");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
        _builder.append("target_include_directories(AML_LIB");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("PUBLIC utils/aml/inc/");
        _builder.newLine();
        _builder.append(")");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      if (label) {
        _builder.append("add_subdirectory (synthetic_gen/labels)");
        _builder.newLine();
      }
    }
    {
      if (ticks) {
        _builder.append("add_subdirectory (synthetic_gen/ticksUtils)");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("# RUNNABLES_LIB ################################################################");
    _builder.newLine();
    _builder.newLine();
    _builder.append("####");
    _builder.newLine();
    _builder.append("add_library(RUNNABLES_LIB STATIC");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("synthetic_gen/runnables/_src/runnables.cpp");
    _builder.newLine();
    _builder.append(")");
    _builder.newLine();
    _builder.newLine();
    _builder.append("target_include_directories(RUNNABLES_LIB");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("PUBLIC  synthetic_gen/runnables/_inc");
    _builder.newLine();
    _builder.append(")\t");
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    _builder.append("target_include_directories(RUNNABLES_LIB");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("PUBLIC ");
    _builder.newLine();
    {
      if (ticks) {
        _builder.append("    ");
        _builder.append("synthetic_gen/ticksUtils/_inc");
        _builder.newLine();
      }
    }
    {
      if (label) {
        _builder.append("\t");
        _builder.append("synthetic_gen/labels/_inc");
        _builder.newLine();
      }
    }
    {
      if (channelSend) {
        _builder.append("\t");
        _builder.append("synthetic_gen/channelSendUtils/_inc");
        _builder.newLine();
      }
    }
    {
      if (interprocess) {
        _builder.append("\t");
        _builder.append("synthetic_gen/interProcessTriggerUtils/_inc");
        _builder.newLine();
      }
    }
    {
      if (performance_measurement) {
        _builder.append("\t");
        _builder.append("utils/aml/inc");
        _builder.newLine();
      }
    }
    _builder.append(")");
    _builder.newLine();
    _builder.append("target_link_libraries(RUNNABLES_LIB");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("PRIVATE ");
    {
      if (performance_measurement) {
        _builder.append("AML_LIB");
      }
    }
    _builder.append(" ");
    {
      if (label) {
        _builder.append("LABELS_LIB");
      }
    }
    _builder.append(" ");
    {
      if (ticks) {
        _builder.append("TICKS_UTILS");
      }
    }
    _builder.append(" ");
    {
      if (channelSend) {
        _builder.append("CHANNELSEND_UTILS");
      }
    }
    _builder.append(" ");
    {
      if (interprocess) {
        _builder.append("INTERPROCESSTRIGGER_UTIL");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append(")");
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    {
      for(final String tag : tags) {
        _builder.append("add_executable(");
        _builder.append(tag);
        _builder.append(" ");
        _builder.append(tag);
        _builder.append("/_src/");
        _builder.append(tag);
        _builder.append(".cpp)");
        _builder.newLineIfNotEmpty();
        _builder.append("target_link_libraries(");
        _builder.append(tag);
        _builder.append("  ");
        {
          if (label) {
            _builder.append("LABELS_LIB");
          }
        }
        _builder.append(" RUNNABLES_LIB)");
        _builder.newLineIfNotEmpty();
        _builder.append("ament_target_dependencies(");
        _builder.append(tag);
        _builder.append(" rclcpp std_msgs ");
        {
          for(final String service_2 : services) {
            _builder.append(service_2);
            _builder.append(" ");
          }
        }
        _builder.append(")");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.newLine();
    _builder.append("install(TARGETS");
    _builder.newLine();
    {
      for(final String tag_1 : tags) {
        _builder.append(" \t");
        _builder.append(tag_1, " \t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append(" \t");
    _builder.append("DESTINATION lib/${PROJECT_NAME})");
    _builder.newLine();
    _builder.newLine();
    _builder.append("ament_package()");
    _builder.newLine();
    _builder.newLine();
    _builder.newLine();
    return _builder.toString();
  }
}
