/**
 ********************************************************************************
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.slg.ros2.transformers;

import static org.eclipse.app4mc.slg.ros2.transformers.RosBaseSettings.OTHER_TYPE;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.app4mc.amalthea.model.ActivityGraphItem;
import org.eclipse.app4mc.amalthea.model.Amalthea;
import org.eclipse.app4mc.amalthea.model.BooleanObject;
import org.eclipse.app4mc.amalthea.model.ChannelSend;
import org.eclipse.app4mc.amalthea.model.CommonElements;
import org.eclipse.app4mc.amalthea.model.InterProcessStimulus;
import org.eclipse.app4mc.amalthea.model.InterProcessTrigger;
import org.eclipse.app4mc.amalthea.model.LabelAccess;
import org.eclipse.app4mc.amalthea.model.Runnable;
import org.eclipse.app4mc.amalthea.model.SWModel;
import org.eclipse.app4mc.amalthea.model.Stimulus;
import org.eclipse.app4mc.amalthea.model.Tag;
import org.eclipse.app4mc.amalthea.model.Task;
import org.eclipse.app4mc.amalthea.model.Ticks;
import org.eclipse.app4mc.amalthea.model.Value;
import org.eclipse.app4mc.amalthea.model.util.ModelUtil;
import org.eclipse.app4mc.amalthea.model.util.SoftwareUtil;
import org.eclipse.app4mc.slg.commons.m2t.transformers.AmaltheaModel2TextTransformer;
import org.eclipse.app4mc.slg.commons.m2t.transformers.sw.TicksUtilsTransformer;
import org.eclipse.app4mc.slg.ros2.generators.RosModel2TextGenerator;
import org.eclipse.app4mc.slg.ros2.transformers.common.RosAmlTransformer;
import org.eclipse.app4mc.slg.ros2.transformers.common.RosTagTransformer;
import org.eclipse.app4mc.slg.ros2.transformers.sw.RosLabelTransformer;
import org.eclipse.app4mc.transformation.util.OutputBuffer;

import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class RosModel2TextTransformer extends AmaltheaModel2TextTransformer {

	@Inject private OutputBuffer outputBuffer;
	@Inject private RosAmlTransformer amlTransformer;
	@Inject private RosTagTransformer tagTransformer;
	@Inject private TicksUtilsTransformer ticksUtilsTransformer;
	@Inject private RosLabelTransformer labelTransformer;

	private boolean label = false;
	private boolean ticks = false;
	private boolean interprocess = false;
	private boolean channelSend = false;
	private boolean performanceMeasurement = false;

	@Override
	public void transform(final Amalthea model, final String outputFolder) {

		RosBaseSettings.initializeOutputBuffer(this.outputBuffer, outputFolder);

		amlTransformer.transform(model);
		
		CommonElements commonElements = ModelUtil.getOrCreateCommonElements(model);
		for (Tag tag : commonElements.getTags()) {
			// instance to know which node published a message
			tagTransformer.transform(tag, model);
		}

		/* extract information needed for top-level CMake file */
		/* find runnables that use service files and use the tag to set the dependencies */

		Set<String> nodes = new HashSet<>();
		Set<String> services = new HashSet<>();

		SWModel swModel = ModelUtil.getOrCreateSwModel(model);
		for (Task task : swModel.getTasks()) {

			Stimulus stimulus = task.getStimuli().get(0);	// Mc: Why only the first ?
			for (final Tag tag : task.getTags()) {
				nodes.add(tag.getName());
			}

			// check if task has interprocess stimulus, if so task is service callback
			if ((stimulus instanceof InterProcessStimulus)) {
				services.add(((InterProcessStimulus) stimulus).getName() + "_service");
			} else {
				// search for runnables called by task
				for (Runnable run : SoftwareUtil.getRunnableList(task, null)) {
					// check custom properties
					Value perf = run.getCustomProperties().get("measure_performance");
					if (perf instanceof BooleanObject && ((BooleanObject) perf).isValue()) {
						performanceMeasurement = true;
					}

					// check activity graph
					for (ActivityGraphItem item : run.getRunnableItems()) {
						if (item instanceof LabelAccess) {
							label = true;
						}
						if (item instanceof Ticks) {
							ticks = true;
						}
						if (item instanceof InterProcessTrigger) {
							interprocess = true;
						}
						if (item instanceof ChannelSend) {
							channelSend = true;
						}
					}
				}
			}
		}

		ticksUtilsTransformer.createCMake();
		labelTransformer.createCMake();

		outputBuffer.appendTo(OTHER_TYPE, "CMakeLists.txt", RosModel2TextGenerator.toCmake(nodes, services, label,
				ticks, interprocess, channelSend, performanceMeasurement));
		outputBuffer.appendTo(OTHER_TYPE, "package.xml", RosModel2TextGenerator.toPackageXml(services));
		outputBuffer.appendTo(OTHER_TYPE, "build.sh", RosModel2TextGenerator.toBuildScript(services));
		outputBuffer.appendTo(OTHER_TYPE, "launch.py", RosModel2TextGenerator.toLaunchFile(nodes));

		outputBuffer.finish();
	}

}
