/**
 ********************************************************************************
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 * *******************************************************************************
 */

package org.eclipse.app4mc.slg.ros2.transformers.common;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.model.Amalthea;
import org.eclipse.app4mc.amalthea.model.Tag;
import org.eclipse.app4mc.amalthea.model.Task;
import org.eclipse.app4mc.slg.commons.m2t.transformers.SLGTranslationUnit;
import org.eclipse.app4mc.slg.ros2.generators.RosTagGenerator;
import org.eclipse.app4mc.slg.ros2.transformers.RosBaseTransformer;
import org.eclipse.app4mc.slg.ros2.transformers.sw.RosTaskCache;
import org.eclipse.app4mc.slg.ros2.transformers.sw.RosTaskCache.TaskStore;
import org.eclipse.app4mc.slg.ros2.transformers.sw.RosTaskTransformer;

import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class RosTagTransformer extends RosBaseTransformer {

	@Inject private RosTaskTransformer rosTaskTransformer;
	@Inject private RosTaskCache rosTaskCache;

	// ---------- generic part "def create new transform(...)" ----------

	private final Map<List<Object>, SLGTranslationUnit> transformCache = new HashMap<>();

	@Override
	public Map<List<Object>, SLGTranslationUnit> getCache() {
		return this.transformCache;
	}

	public SLGTranslationUnit transform(final Tag tag, final Amalthea model) {
		final List<Object> key = new ArrayList<>(Arrays.asList(tag, model));
		final SLGTranslationUnit tu;

		synchronized (transformCache) {
			if (transformCache.containsKey(key)) {
				return transformCache.get(key);
			}
			tu = createTranslationUnit(tag);
			transformCache.put(key, tu);
		}

		// if translation unit is newly created and valid -> create files
		if (tu.isValid()) {
			doTransform(tu, tag, model);
		}

		return tu;
	}

	// ---------------------------------------------------

	private SLGTranslationUnit createTranslationUnit(final Tag tag) {
		if ((tag == null)) {
			return new SLGTranslationUnit("UNSPECIFIED TAG");
		} else {
			String basePath = "";
			String moduleName = tag.getName();
			String call = "";
			return new SLGTranslationUnit(basePath, moduleName, call);
		}
	}

	private void doTransform(final SLGTranslationUnit tu, final Tag tag, final Amalthea model) {
		genFiles(tu, tag, model);
	}

	private void genFiles(final SLGTranslationUnit tu, final Tag tag, final Amalthea model) {
		
		HashSet<String> header = new HashSet<>();
		List<String> declaration = new LinkedList<>();
		List<String> initialization = new LinkedList<>();
		List<String> calls = new LinkedList<>();
		List<String> serviceCallbacks = new LinkedList<>();

		if (model != null && model.getSwModel() != null) {
			for (Task task : model.getSwModel().getTasks()) {
				if (task.getTags().contains(tag)) {
					final SLGTranslationUnit taskTU = rosTaskTransformer.transform(task);
					final TaskStore store = rosTaskCache.getStore(taskTU);

					header.add(store.getHeaders());
					declaration.add(store.getDeclaration());
					initialization.add(store.getInitialisation(tu.getModuleName()));
					calls.add(store.getCallback());
					serviceCallbacks.add(store.getServiceCallback());
				}
			}
		}

		srcAppend(tu, RosTagGenerator.toCpp(tag, tu.getModuleName(), header, declaration, initialization, calls, serviceCallbacks));
	}

}
