/**
 ********************************************************************************
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 * *******************************************************************************
 */

package org.eclipse.app4mc.slg.ros2.transformers.sw;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.eclipse.app4mc.amalthea.model.ActivityGraphItem;
import org.eclipse.app4mc.amalthea.model.Group;
import org.eclipse.app4mc.amalthea.model.ITaggable;
import org.eclipse.app4mc.amalthea.model.InterProcessStimulus;
import org.eclipse.app4mc.amalthea.model.Runnable;
import org.eclipse.app4mc.amalthea.model.RunnableCall;
import org.eclipse.app4mc.amalthea.model.Stimulus;
import org.eclipse.app4mc.amalthea.model.Tag;
import org.eclipse.app4mc.amalthea.model.Task;
import org.eclipse.app4mc.slg.commons.m2t.transformers.SLGTranslationUnit;
import org.eclipse.app4mc.slg.commons.m2t.transformers.sw.TaskTransformer;
import org.eclipse.app4mc.slg.ros2.generators.RosLabelGenerator;
import org.eclipse.app4mc.slg.ros2.transformers.stimuli.RosInterProcessStimulusTransformer;
import org.eclipse.app4mc.slg.ros2.transformers.sw.RosRunnableCache.RunnableStore;

import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class RosTaskTransformer extends TaskTransformer {

	@Inject private RosLabelTransformer rosLabelTransformer;
	@Inject private RosRunnableTransformer rosRunnableTransformer;
	@Inject private RosRunnableCache rosRunnableCache;
	@Inject private RosInterProcessStimulusTransformer rosInterProcessStimulusTransformer;
	@Inject private RosTaskCache rosTaskCache;

	@Override
	protected void genFiles(SLGTranslationUnit tu, Task task) {

		final Set<String> includes = new LinkedHashSet<>();
		final List<String> initCalls = new LinkedList<>();
		final List<String> stepCalls = new LinkedList<>();
		final List<Stimulus> stimuli = new ArrayList<>();
		final List<String> publishers = new ArrayList<>();
		final List<String> clientDeclarations = new ArrayList<>();
		final List<String> clientInits = new ArrayList<>();

		if (task != null && task.getActivityGraph() != null) {
			for (ActivityGraphItem item : task.getActivityGraph().getItems()) {

				if ((item instanceof RunnableCall)) {
					final RunnableCall runnableCall = (RunnableCall) item;
					final Runnable runnable = runnableCall.getRunnable();

					SLGTranslationUnit runnableTU = rosRunnableTransformer.transform(runnable);
					RunnableStore runnableStore = rosRunnableCache.getStore(runnableTU);

					includes.add(getIncFile(runnableTU));

					if (hasTagNamed(runnableCall, "initialize")) {
						initCalls.add(runnableStore.getNodeCall());
					} else {
						stepCalls.add(runnableStore.getNodeCall());
					}

					// TODO: Make set
					publishers.addAll(runnableStore.getPublishers());
					clientDeclarations.addAll(runnableStore.getClientDeclarations());
					clientInits.addAll(runnableStore.getClientInits());
					// TODO: add terminate function, if requested

				} else if (item instanceof Group) {
					final Group group = ((Group) item);
					for (ActivityGraphItem groupitem : group.getItems()) {
						if ((groupitem instanceof RunnableCall)) {
							final RunnableCall runnableCall = (RunnableCall) groupitem;
							final Runnable runnable = runnableCall.getRunnable();

							SLGTranslationUnit runnableTU = rosRunnableTransformer.transform(runnable);
							RunnableStore runnableStore = rosRunnableCache.getStore(runnableTU);

							includes.add(getIncFile(runnableTU));

							if (hasTagNamed(runnableCall, "initialize")) {
								initCalls.add(runnableStore.getNodeCall());
							} else {
								stepCalls.add(runnableStore.getNodeCall());
							}
							// TODO: add terminate function, if requested
						}
					}
				}
			}
		}

		// labels must be initialized before usage, generated labels provide this method

		for (SLGTranslationUnit labelTU : rosLabelTransformer.getCache().values()) {
			includes.add(getIncFile(labelTU));
			initCalls.add(RosLabelGenerator.initCall(labelTU.getCall()));
		}

//		rosLabelTransformer.getCache().forEach(
//				(BiConsumer<ArrayList<?>, LabelTranslationUnit>) (ArrayList<?> label, LabelTranslationUnit tu) -> {
//
//				});

		// add header for srv file in case of an interprocessstimulus
		// create .srv file for the messages to be translated

		if (task != null) {
			for (Stimulus stimulus : task.getStimuli()) {
				if (stimulus instanceof InterProcessStimulus) {
					String name = stimulus.getName();
					includes.add(name + "_service/srv/" + name + "_service.hpp");

					rosInterProcessStimulusTransformer.transform(((InterProcessStimulus) stimulus));
				}
			}

			stimuli.addAll(task.getStimuli());
		}

		// store characteristic values in task cache
		rosTaskCache.storeValues(tu, task, stimuli, includes, initCalls, stepCalls, publishers, clientDeclarations, clientInits);

	}

	private boolean hasTagNamed(ITaggable element, String name) {
		for (Tag tag : element.getTags()) {
			if (tag.getName().equals(name))
				return true;
		}
		return false;
	}

}
