/**
 ********************************************************************************
 * Copyright (c) 2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.slg.commons.m2t.transformers.sw;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.model.Label;
import org.eclipse.app4mc.slg.commons.m2t.generators.LabelGenerator;
import org.eclipse.app4mc.slg.commons.m2t.transformers.SLGBaseTransformer;
import org.eclipse.app4mc.slg.commons.m2t.transformers.SLGTranslationUnit;
import org.eclipse.app4mc.transformation.util.OutputBuffer;

import com.google.inject.Inject;
import com.google.inject.Singleton;

@Singleton
public class LabelTransformer extends SLGBaseTransformer {

	public static final String LIB_NAME = "LABELS_LIB";
	public static final String BASE_PATH = "synthetic_gen";
	public static final String MODULE_NAME = "labels";
	public static final String MODULE_PATH = BASE_PATH + "/" + MODULE_NAME;
	public static final String MAKEFILE_PATH = MODULE_PATH + "/CMakeLists.txt";

	@Inject private OutputBuffer outputBuffer;

	// ---------- generic part "def create new transform(...)" ----------

	private final Map<List<Object>, SLGTranslationUnit> transformCache = new HashMap<>();

	@Override
	public Map<List<Object>, SLGTranslationUnit> getCache() {
		return this.transformCache;
	}

	public SLGTranslationUnit transform(final Label label) {
		final List<Object> key = new ArrayList<>(Arrays.asList(label));
		final SLGTranslationUnit tu;

		synchronized (transformCache) {
			if (transformCache.containsKey(key)) {
				return transformCache.get(key);
			}
			tu = createTranslationUnit(label);
			transformCache.put(key, tu);
		}

		// if translation unit is newly created and valid -> create files
		if (tu.isValid()) {
			doTransform(tu, label);
		}

		return tu;
	}

	// ---------------------------------------------------

	protected SLGTranslationUnit createTranslationUnit(final Label label) {
		if ((label == null)) {
			return new SLGTranslationUnit("UNSPECIFIED LABEL");
		} else {
			String basePath = BASE_PATH;
			String moduleName = MODULE_NAME;
			String call = label.getName();
			return new SLGTranslationUnit(basePath, moduleName, call);
		}
	}

	protected void doTransform(final SLGTranslationUnit tu, final Label label) {
		genFiles(tu, label);
	}

	protected void genFiles(SLGTranslationUnit tu, Label label) {
		if (isSrcFileEmpty(tu)) {
			srcAppend(tu, "#include \"" + getIncFile(tu) + "\"\n\n");
		}
		if (isIncFileEmpty(tu)) {	// Mc: ???
			incAppend(tu, "#include <stdbool.h>\n\n");
		}

		incAppend(tu, LabelGenerator.toH(label));
		srcAppend(tu, LabelGenerator.toCpp(label));
	}

	public boolean createCMake() {
		return outputBuffer.appendTo(
				"OTHER", MAKEFILE_PATH, LabelGenerator.toCMake(LIB_NAME, getSrcFiles()));
	}

}
