/**
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 */

package org.eclipse.app4mc.slg.ros2.generators

import com.google.inject.Singleton
import java.util.Set

@Singleton
class RosModel2TextGenerator {

	static def String toLaunchFile(Set<String> nodes) '''
		from launch import LaunchDescription
		from launch_ros.actions import Node
		
		def generate_launch_description():
		    return LaunchDescription([
				«FOR node : nodes»
					Node(
						package='amalthea_ros_model',
						node_namespace='',
						node_executable='«node»',
					),
				«ENDFOR»
			])
	'''

	static def String toBuildScript(Set<String> services) '''
		«FOR service : services»
			#building service «service»
			
			cd services/«service»
			colcon build
			. install/setup.bash
			cd ../..
			
		«ENDFOR»
		colcon build
		. install/setup.bash
	'''

	// TODO: define attributes via ConfigModel
	static def String toPackageXml(Set<String> services) '''
		<?xml version="1.0"?>
		<?xml-model href="http://download.ros.org/schema/package_format3.xsd" schematypens="http://www.w3.org/2001/XMLSchema"?>
		<package format="3">
		  	<name>amalthea_ros_model</name>
		  	<version>0.0.0</version>
		  	<description>TODO: Package description</description>
		  	<maintainer email="fixedterm.Patrick.Will@de.bosch.com">wlp8fe</maintainer>
		  	<license>TODO: License declaration</license>
		
		  	<buildtool_depend>ament_cmake</buildtool_depend>
			<depend>rclcpp</depend>
			<depend>std_msgs</depend>
		
		«FOR service : services»
			<depend>«service»</depend>
		«ENDFOR»
		
		  	<test_depend>ament_lint_auto</test_depend>
		  	<test_depend>ament_lint_common</test_depend>
		
		  <export>
		    <build_type>ament_cmake</build_type>
		  </export>
		</package>
	'''

	static def String toCmake(Set<String> tags, Set<String> services, boolean label, boolean ticks, boolean interprocess, boolean channelSend, boolean performance_measurement) '''
		# CMakeLists for Nodes
		cmake_minimum_required(VERSION 3.5)
		project(amalthea_ros_model)
		
		# Default to C++14
		if(NOT CMAKE_CXX_STANDARD)
		  set(CMAKE_CXX_STANDARD 14)
		endif()
		
		if(CMAKE_COMPILER_IS_GNUCXX OR CMAKE_CXX_COMPILER_ID MATCHES "Clang")
		  add_compile_options(-Wall -Wextra -Wpedantic)
		endif()
		
		
		find_package(ament_cmake REQUIRED)
		find_package(rclcpp REQUIRED)
		find_package(std_msgs REQUIRED)
		
		«FOR service : services»
			find_package(«service» REQUIRED)
		«ENDFOR»
		
		«IF channelSend»
			add_library(CHANNELSEND_UTILS  STATIC
				synthetic_gen/channelSendUtils/_src/channelSendUtils.cpp
			)
			
			ament_target_dependencies(CHANNELSEND_UTILS rclcpp std_msgs)
			
			
			target_include_directories(CHANNELSEND_UTILS
			    PUBLIC synthetic_gen/channelSendUtils/_inc/
			)
		«ENDIF»
		
		
		
		«««Only to compile if there are InterProcessTriggers»»»
	«IF interprocess»
			add_library(INTERPROCESSTRIGGER_UTIL STATIC
				  synthetic_gen/interProcessTriggerUtils/_src/interProcessTriggerUtils.cpp
			)
			ament_target_dependencies(INTERPROCESSTRIGGER_UTIL rclcpp «FOR service : services»«service» «ENDFOR»)
			target_include_directories(INTERPROCESSTRIGGER_UTIL
			    PUBLIC  synthetic_gen/interProcessTriggerUtils/_inc
			)
		«ENDIF»
		
		«IF performance_measurement»
			add_library(AML_LIB  STATIC
				utils/aml/src/aml.cpp
			)
			target_include_directories(AML_LIB
			    PUBLIC utils/aml/inc/
			)
		«ENDIF»
		
		«IF label»
			add_subdirectory (synthetic_gen/labels)
		«ENDIF»
		«IF ticks»
			add_subdirectory (synthetic_gen/ticksUtils)
		«ENDIF»
		
		# RUNNABLES_LIB ################################################################
		
		####
		add_library(RUNNABLES_LIB STATIC
			synthetic_gen/runnables/_src/runnables.cpp
		)
		
		target_include_directories(RUNNABLES_LIB
		    PUBLIC  synthetic_gen/runnables/_inc
		)	
		
		
		target_include_directories(RUNNABLES_LIB
		    PUBLIC 
		    «IF ticks»
		    	synthetic_gen/ticksUtils/_inc
		    «ENDIF»
			«IF label»
				synthetic_gen/labels/_inc
			«ENDIF»
			«IF channelSend»
				synthetic_gen/channelSendUtils/_inc
			«ENDIF»
			«IF interprocess»
				synthetic_gen/interProcessTriggerUtils/_inc
			«ENDIF»
			«IF performance_measurement»
			utils/aml/inc
			«ENDIF»
		)
		target_link_libraries(RUNNABLES_LIB
		    PRIVATE «IF performance_measurement»AML_LIB«ENDIF» «IF label»LABELS_LIB«ENDIF» «IF ticks»TICKS_UTILS«ENDIF» «IF channelSend»CHANNELSEND_UTILS«ENDIF» «IF interprocess»INTERPROCESSTRIGGER_UTIL«ENDIF»
		)
		
		
		
		«FOR tag : tags»
			add_executable(«tag» «tag»/_src/«tag».cpp)
			target_link_libraries(«tag»  «IF label»LABELS_LIB«ENDIF» RUNNABLES_LIB)
			ament_target_dependencies(«tag» rclcpp std_msgs «FOR service : services»«service» «ENDFOR»)
		«ENDFOR»
		
		
		install(TARGETS
		 	«FOR tag : tags»
		 		«tag»
		 	«ENDFOR»
		 	DESTINATION lib/${PROJECT_NAME})
		
		ament_package()
		
		
	'''

}
