/**
 * Copyright (c) 2020-2021 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 */
package org.eclipse.app4mc.slg.ros2.generators;

import java.math.BigInteger;
import java.util.Collection;
import org.eclipse.app4mc.amalthea.model.EventStimulus;
import org.eclipse.app4mc.amalthea.model.InterProcessStimulus;
import org.eclipse.app4mc.amalthea.model.PeriodicStimulus;
import org.eclipse.app4mc.amalthea.model.Stimulus;
import org.eclipse.app4mc.amalthea.model.TimeUnit;
import org.eclipse.app4mc.slg.ros2.transformers.utils.Utils;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class RosTaskGenerator {
  private RosTaskGenerator() {
    throw new IllegalStateException("Utility class");
  }
  
  public static String getDeclaration(final Collection<Stimulus> stimuli, final Collection<String> publishers, final Collection<String> clientDeclarations) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Stimulus stimulus : stimuli) {
        {
          if ((stimulus instanceof PeriodicStimulus)) {
            _builder.append("rclcpp::TimerBase::SharedPtr ");
            String _name = ((PeriodicStimulus) stimulus).getName();
            _builder.append(_name);
            _builder.append("_timer_;");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          if ((stimulus instanceof EventStimulus)) {
            _builder.append("rclcpp::Subscription<std_msgs::msg::String>::SharedPtr ");
            String _name_1 = ((EventStimulus) stimulus).getName();
            _builder.append(_name_1);
            _builder.append("_subscription_;");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          if ((stimulus instanceof InterProcessStimulus)) {
            _builder.append("rclcpp::Service<");
            String _name_2 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_2);
            _builder.append("_service::srv::");
            String _name_3 = ((InterProcessStimulus) stimulus).getName();
            String _plus = (_name_3 + "_service");
            String _idlCompliantName = Utils.toIdlCompliantName(_plus);
            _builder.append(_idlCompliantName);
            _builder.append(">::SharedPtr ");
            String _name_4 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_4);
            _builder.append("_service;");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    {
      for(final String publisher : publishers) {
        _builder.append("rclcpp::Publisher<std_msgs::msg::String>::SharedPtr ");
        _builder.append(publisher);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final String decl : clientDeclarations) {
        _builder.append(decl);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
  
  public static String getInitialisation(final String nodeName, final Collection<Stimulus> stimuli, final Collection<String> publishers, final Collection<String> clientInits) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Stimulus stimulus : stimuli) {
        {
          if ((stimulus instanceof PeriodicStimulus)) {
            String _name = ((PeriodicStimulus) stimulus).getName();
            _builder.append(_name);
            _builder.append("_timer_ = this->create_wall_timer(");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t");
            BigInteger _value = ((PeriodicStimulus) stimulus).getRecurrence().getValue();
            _builder.append(_value, "\t\t");
            TimeUnit _unit = ((PeriodicStimulus) stimulus).getRecurrence().getUnit();
            _builder.append(_unit, "\t\t");
            _builder.append(", std::bind(&");
            _builder.append(nodeName, "\t\t");
            _builder.append("::");
            String _name_1 = ((PeriodicStimulus) stimulus).getName();
            _builder.append(_name_1, "\t\t");
            _builder.append("_timer_callback, this));");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          if ((stimulus instanceof EventStimulus)) {
            String _name_2 = ((EventStimulus) stimulus).getName();
            _builder.append(_name_2);
            _builder.append("_subscription_ = this->create_subscription<std_msgs::msg::String>(");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\"");
            String _name_3 = ((EventStimulus) stimulus).getName();
            _builder.append(_name_3, "\t");
            _builder.append("\", 10, std::bind(&");
            _builder.append(nodeName, "\t");
            _builder.append("::");
            String _name_4 = ((EventStimulus) stimulus).getName();
            _builder.append(_name_4, "\t");
            _builder.append("_subscription_callback, this, _1));");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          if ((stimulus instanceof InterProcessStimulus)) {
            String _name_5 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_5);
            _builder.append("_service = this->create_service<");
            String _name_6 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_6);
            _builder.append("_service::srv::");
            String _name_7 = ((InterProcessStimulus) stimulus).getName();
            String _plus = (_name_7 + "_service");
            String _idlCompliantName = Utils.toIdlCompliantName(_plus);
            _builder.append(_idlCompliantName);
            _builder.append(">(\"");
            String _name_8 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_8);
            _builder.append("_service\", &");
            String _name_9 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_9);
            _builder.append("_service_callback);");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    {
      for(final String publisher : publishers) {
        _builder.append(publisher);
        _builder.append(" = this->create_publisher<std_msgs::msg::String>(\"");
        String _replace = publisher.replace("_publisher", "");
        _builder.append(_replace);
        _builder.append("\", 10);");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final String init : clientInits) {
        _builder.append(init);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
  
  public static String getServiceCallback(final Collection<Stimulus> stimuli, final Collection<String> stepCalls) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Stimulus stimulus : stimuli) {
        {
          if ((stimulus instanceof InterProcessStimulus)) {
            _builder.append("void ");
            String _name = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name);
            _builder.append("_service_callback(const std::shared_ptr<");
            String _name_1 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_1);
            _builder.append("_service::srv::");
            String _name_2 = ((InterProcessStimulus) stimulus).getName();
            String _plus = (_name_2 + "_service");
            String _idlCompliantName = Utils.toIdlCompliantName(_plus);
            _builder.append(_idlCompliantName);
            _builder.append("::Request> request,");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("std::shared_ptr<");
            String _name_3 = ((InterProcessStimulus) stimulus).getName();
            _builder.append(_name_3, "\t");
            _builder.append("_service::srv::");
            String _name_4 = ((InterProcessStimulus) stimulus).getName();
            String _plus_1 = (_name_4 + "_service");
            String _idlCompliantName_1 = Utils.toIdlCompliantName(_plus_1);
            _builder.append(_idlCompliantName_1, "\t");
            _builder.append("::Response>      response) {");
            _builder.newLineIfNotEmpty();
            {
              for(final String call : stepCalls) {
                _builder.append("\t\t");
                _builder.append(call);
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("\t");
            _builder.append("}");
            _builder.newLine();
          }
        }
        _builder.newLine();
        _builder.newLine();
      }
    }
    return _builder.toString();
  }
  
  public static String getCallback(final Collection<Stimulus> stimuli, final Collection<String> stepCalls) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Stimulus stimulus : stimuli) {
        {
          if ((stimulus instanceof PeriodicStimulus)) {
            _builder.append("void ");
            String _name = ((PeriodicStimulus) stimulus).getName();
            _builder.append(_name);
            _builder.append("_timer_callback() {");
            _builder.newLineIfNotEmpty();
            {
              for(final String call : stepCalls) {
                _builder.append("\t");
                _builder.append(call);
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("}");
            _builder.newLine();
          }
        }
        {
          if ((stimulus instanceof EventStimulus)) {
            _builder.append("void ");
            String _name_1 = ((EventStimulus) stimulus).getName();
            _builder.append(_name_1);
            _builder.append("_subscription_callback(const std_msgs::msg::String::SharedPtr msg) {");
            _builder.newLineIfNotEmpty();
            {
              for(final String call_1 : stepCalls) {
                _builder.append("\t");
                _builder.append(call_1);
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("}");
            _builder.newLine();
          }
        }
      }
    }
    return _builder.toString();
  }
  
  public static String getHeaders(final Collection<String> includes) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final String include : includes) {
        _builder.append("#include \"");
        _builder.append(include);
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
}
