/**
 ********************************************************************************
 * Copyright (c) 2019 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters096.impl;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.LogManager;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters096.utils.HelperUtils_095_096;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;

public class ConstraintsConverter extends AbstractConverter {

	public ConstraintsConverter() {
		this.helper = HelperUtils_095_096.getInstance();
		this.logger = LogManager.getLogger("org.eclipse.app4mc.amalthea.modelmigration");
	}

	@Override
	public void convert(final File file, final Map<File, Document> map, final List<ICache> caches) throws Exception {

		this.logger.info("Migration from 0.9.5 to 0.9.6 : Executing Constraints converter for model file : " + file.getName());

		basicConvert(file, map, caches);
	}

	public void basicConvert(final File file, final Map<File, Document> map, final List<ICache> caches) {

		final Document document = map.get(file);
		if (document == null) {
			return;
		}

		final Element rootElement = document.getRootElement();
		updateEventChains(rootElement);
	}
	
	private void updateEventChains(Element rootElement) {
		 final String xpath = "./constraintsModel/eventChains";
		 final List<Element> eventChains = this.helper.getXpathResult(rootElement, xpath, Element.class,
					this.helper.getNS_096("am"), this.helper.getGenericNS("xsi"));
		 boolean hasSegments = false;
		 for (Element element : eventChains) {
			 hasSegments = false;
			List<Element> segments = new ArrayList<Element>(element.getChildren("segments"));
			if(!segments.isEmpty()) { 
				//convert segments to items
				convertToItems(segments, element,"sequence");
				hasSegments = true;
			}
			
			List<Element> strands = new ArrayList<Element>(element.getChildren("strands"));
			if(hasSegments && !strands.isEmpty()) {
				this.logger.info("EventChains Migration from 0.9.5 to 0.9.6 : Removing strands because Segments are already migrated and both cannot exist at a time in 0.9.6.");
				for (Element strand : strands) {
					strand.detach();
				}
				continue;
			}
			
			if(!strands.isEmpty()) {
				//convert strands to items
				convertToItems(strands, element,"parallel");
			}
		}
	}
	
	private void convertToItems(List<Element> segments,Element eventChain,String itemType) {
		
		//create itemType
		Attribute itemTypeAttr = new Attribute("itemType",itemType);
		eventChain.setAttribute(itemTypeAttr);
		
		for (Element element : segments) {
			element.setName("items");
			if(element.getAttributeValue("type", this.helper.getGenericNS("xsi")).equals("am:EventChainContainer")) {
				Element subevent = element.getChild("eventChain");
				subevent.setAttribute("minItemsCompleted", "1");
				
				eventChain.setAttribute("minItemsCompleted", "1");
			}
		}
	}

	private void updateCustomPropsToCounter(Element rootElement) {
		//bring back counter for RunnableCall if the custom properties have counter info.
		
		 final StringBuffer xpathBuffer = new StringBuffer();
		 
		xpathBuffer.append("./swModel/tasks/callGraph//items[@xsi:type=\"am:RunnableCall\"]/customProperties[@key=\"counter-prescaler\"]");
		xpathBuffer.append("|");
		xpathBuffer.append("./swModel/isrs/callGraph//items[@xsi:type=\"am:RunnableCall\"]/customProperties[@key=\"counter-prescaler\"]");
		xpathBuffer.append("|");
		xpathBuffer.append("./swModel/runnables/callGraph//items[@xsi:type=\"am:RunnableCall\"]/customProperties[@key=\"counter-prescaler\"]");
		
		final List<Element> customProps = this.helper.getXpathResult(rootElement, xpathBuffer.toString(), Element.class,
				this.helper.getNS_096("am"), this.helper.getGenericNS("xsi"));
		
		for (Element element : customProps) {
			Element parentElement = element.getParentElement();
			
			List<Element> customProperties = new ArrayList<Element>(parentElement.getChildren("customProperties"));
			//create counter
			Element counter = new Element("counter");
			parentElement.addContent(counter);
			
			for (Element prop : customProperties) {
				if(prop.getAttributeValue("key").equals("counter-prescaler")) {
					counter.setAttribute("prescaler", prop.getChild("value").getAttributeValue("value"));
					//remove custom prop
					prop.detach();
					continue;
				}
				
				if(prop.getAttributeValue("key").equals("counter-offset")) {					
					counter.setAttribute("offset", prop.getChild("value").getAttributeValue("value"));
					//remove custom prop
					prop.detach();
				}
			}
			
			
		}
	}
 

}
