/**
 ********************************************************************************
 * Copyright (c) 2015-2019 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters.ui.jobs;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.MigrationHelper;
import org.eclipse.app4mc.amalthea.converters.common.MigrationInputFile;
import org.eclipse.app4mc.amalthea.converters.common.MigrationSettings;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;

public class ModelLoaderJob extends Job {

	private final MigrationSettings migrationSettings;
	private final List<File> inputModels;

	public ModelLoaderJob(final String name, final MigrationSettings migrationSettings, final List<File> inputModels) {
		super(name);
		this.migrationSettings = migrationSettings;
		this.inputModels = inputModels;
	}

	@Override
	protected IStatus run(final IProgressMonitor monitor) {

		final int size = this.inputModels.size();

		final SubMonitor subMonitor = SubMonitor.convert(monitor, size);

		subMonitor.setTaskName("Loading selected model files ...");

		try {
			Map<File, MigrationInputFile> modelFilesMap =
					MigrationHelper.populateModels(this.inputModels, this.migrationSettings, subMonitor);
			this.migrationSettings.getMigModelFiles().addAll(modelFilesMap.values());
		}
		catch (final Exception e1) {
			Bundle bundle = FrameworkUtil.getBundle(getClass());
			if (bundle != null) {
				Platform.getLog(bundle).log(
						new Status(
								IStatus.ERROR,
								bundle.getSymbolicName(),
								1,
								"Failed to load model files",
								e1));
			}

			return new Status(IStatus.CANCEL, "unknown", 1, e1.getMessage(), e1);
		}

		return Status.OK_STATUS;
	}

	@Override
	public boolean belongsTo(Object family) {
		if (family != null && family.equals(IMigrationJobConstants.FAMILY)) {
			return true;
		}

		return super.belongsTo(family);
	}
}
