/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters.ui.handlers;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import javax.inject.Named;

import org.eclipse.app4mc.amalthea.converters.common.MigrationException;
import org.eclipse.app4mc.amalthea.converters.common.MigrationHelper;
import org.eclipse.app4mc.amalthea.converters.common.MigrationInputFile;
import org.eclipse.app4mc.amalthea.converters.common.MigrationProcessor;
import org.eclipse.app4mc.amalthea.converters.common.MigrationSettings;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.eclipse.app4mc.amalthea.converters.ui.dialog.MigrationErrorDialog;
import org.eclipse.app4mc.amalthea.converters.ui.dialog.ModelMigrationDialog;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.MigrationJobChangeListener;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.ModelLoaderJob;
import org.eclipse.app4mc.amalthea.converters.ui.jobs.ModelMigrationJob;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.Service;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("restriction")
public class AmaltheaModelMigrationHandler {

	private static final Logger LOGGER = LoggerFactory.getLogger(AmaltheaModelMigrationHandler.class);
	private static final String SIMPLE_MIGRATION = "simplemigration";

	/**
	 * Executes the migration in an Eclipse UI. Opens dialogs and uses the jobs
	 * framework for asynchronous execution.
	 *
	 * @param shell              The current Shell.
	 * @param migrationProcessor The {@link MigrationProcessor} service that
	 *                           performs the migration.
	 * @param selection          The current active selection in the workspace.
	 * @param executionContext   Optional parameter that can be used to trigger a
	 *                           simple migration if set to "simplemigration".
	 * @param modelEditorVersion The model version that is supported by the
	 *                           available editor. Can be <code>null</code> which
	 *                           leads to using the latest supported version by the
	 *                           migration component.
	 */
	@Execute
	public void execute(
			Shell shell,
			@Service MigrationProcessor migrationProcessor,
			@Named(IServiceConstants.ACTIVE_SELECTION) ISelection selection,
			@Optional @Named("executioncontext") String executionContext,
			@Optional @Named("migrationversion") String modelEditorVersion) {

		if (selection instanceof TreeSelection) {
			MigrationSettings migrationSettings = new MigrationSettings();
			List<File> inputModels = new ArrayList<>();

			IProject iProject = collectInput((TreeSelection) selection, shell, migrationSettings, inputModels);

			if (modelEditorVersion != null) {
				migrationSettings.setMigrationModelVersion(modelEditorVersion);
			}

			final Job job = new ModelLoaderJob(
					"Parsing selected files to obtain referred AMALTHEA models and corresponding Model versions",
					migrationSettings, inputModels);
			job.setUser(true);
			job.schedule();

			job.addJobChangeListener(new JobChangeAdapter() {

				@Override
				public void done(final IJobChangeEvent jobEvent) {
					super.done(jobEvent);

					if (jobEvent.getResult().equals(Status.OK_STATUS)) {

						Display.getDefault().asyncExec(new Runnable() {

							@Override
							public void run() {

								try {
									boolean inputValid = MigrationHelper.isInputModelVersionValid(migrationSettings);

									if (!inputValid) {
										MigrationErrorDialog errorDialog = new MigrationErrorDialog(shell, migrationSettings);
										errorDialog.open();
									} else {
										if (migrationSettings.getInputModelVersion() != null
												&& migrationSettings.getInputModelVersion().equals(migrationSettings.getMigrationModelVersion())) {

											MessageDialog.openInformation(shell, "AMALTHEA Model Migration",
													"Selected models are compatible to latest AMALTHEA meta-model version ("
															+ migrationSettings.getMigrationModelVersion()
															+ ")\nIt is not required to migrate these models !!");

										} else if (!ModelVersion.isValidVersion(migrationSettings.getMigrationModelVersion())) {

											MessageDialog.openError(shell, "AMALTHEA Model Migration",
													"The AMALTHEA meta-model version ("
													+ migrationSettings.getMigrationModelVersion()
													+ ") is not supported by the migration component !!");

										} else if (migrationSettings.getInputModelVersion() != null
												&& migrationSettings.getMigrationModelVersion() != null
												&& ModelVersion.getModelVersion(migrationSettings.getInputModelVersion()).ordinal()
													> ModelVersion.getModelVersion(migrationSettings.getMigrationModelVersion()).ordinal()) {

											MessageDialog.openError(shell, "AMALTHEA Model Migration",
													"The supported model version ("
													+ migrationSettings.getMigrationModelVersion()
													+ ") is older than the input model version ("
													+ migrationSettings.getInputModelVersion() + ").\n\n"
													+ "A backwards migration is not supported !!");

										} else {
											if (isSimpleMigration(executionContext) && migrationSettings.getMigModelFiles().size() == 1) {
												// for simple migration perform simple settings and start migration directly

												// set the file parent folder as output location to convert the file at source
												MigrationInputFile migrationInputFile = migrationSettings.getMigModelFiles().get(0);
												migrationSettings.setOutputDirectoryLocation(migrationInputFile.getFile().getParent());

												// Rename or copy the original file to filename_currentversion.amxmi
												boolean backupSucceeded = MigrationHelper.createBackupFile(migrationInputFile);

												if (backupSucceeded) {
													//now call migration job to migrate the file to latest Amalthea version
													ModelMigrationJob migrationJob = new ModelMigrationJob(
															"AMALTHEA Model Migration",
															migrationProcessor,
															migrationSettings);

													migrationJob.setUser(true);
													migrationJob.schedule();
													migrationJob.addJobChangeListener(new MigrationJobChangeListener(shell, iProject));
												} else {
													// do nothing as we could not backup source and this can data loss to user if he
													// does not intend to loose his original model file
													LOGGER.error("Migration Stopped : Source file could not be backed up before migration");
												}
											} else {
												// open dialog so the user can configure the settings
												ModelMigrationDialog dialog = new ModelMigrationDialog(shell, migrationProcessor, migrationSettings, iProject);
												dialog.open();
											}
										}
									}
								} catch (MigrationException e) {
									MessageDialog.openError(shell, "AMALTHEA Model Migration",
											e.getLocalizedMessage());
								}

							}

						});
					}
					else if (jobEvent.getResult().getCode() == Status.CANCEL_STATUS.getCode()) {

						Display.getDefault().asyncExec(new Runnable() {

							@Override
							public void run() {
								MessageDialog.openError(shell, "AMALTHEA Model Migration",
										jobEvent.getResult().getMessage());
							}
						});
					}
				}
			});
		}
	}

	private IProject collectInput(TreeSelection selection, Shell shell, MigrationSettings settings, List<File> inputModels) {
		TreePath[] paths = selection.getPaths();

		String projectSegment = null;
		IProject iProject = null;

		HashSet<File> inputFiles = new HashSet<>();
		HashSet<File> parents = new HashSet<>();
		
		for (TreePath treePath : paths) {
			Object firstSegment = treePath.getFirstSegment();

			if (projectSegment == null) {
				projectSegment = firstSegment.toString();

				iProject = ((IAdaptable) firstSegment).getAdapter(IProject.class);

				if (iProject != null) {
					String path = iProject.getLocation().toOSString();
					File file = new File(path);
					settings.setProject(file);
				}
			}
			else if (!projectSegment.equals(firstSegment.toString())) {
				MessageDialog.openError(shell, "AMALTHEA Model Migration",
						"Model files across the projects should not be selected for migration");
			}

			final Object lastSegment = treePath.getLastSegment();

			if (lastSegment instanceof IAdaptable) {
				IFile iFile = ((IAdaptable) lastSegment).getAdapter(IFile.class);

				if (iFile != null) {
					String path = iFile.getRawLocation().toOSString();
					File file = new File(path);
					try {
						inputFiles.add(file.getCanonicalFile());
						parents.add(file.getParentFile());
					}
					catch (IOException e) {
						LOGGER.error("Error fetching the file : {}", iFile.toString(), e);
					}
				}
			}
		}

		for (File parent : parents) {
			// as the model scope is set to Folder, all the amxmi files present in the folder should be considered
			File[] allFilesInDirectory = parent.listFiles(new FilenameFilter() {
				@Override
				public boolean accept(final File arg0, final String arg1) {
					return arg1.endsWith(".amxmi");
				}
			});
			
			inputFiles.addAll(Arrays.asList(allFilesInDirectory));
		}

		inputModels.addAll(inputFiles);

		return iProject;
	}

	private boolean isSimpleMigration(String context) {
		return (context != null && context.equals(SIMPLE_MIGRATION));
	}

}
