/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters.common.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Enumeration of Amalthea model versions. Needs to be maintained in ascending order to support ordered processing.
 *
 * <p>
 * <b>Note: </b> New versions need to be added add the end of the enumeration definition!
 * </p>
 */
public enum ModelVersion {

	_103("itea.103"),
	_110("itea.110"),
	_111("itea.111"),
	_070("0.7.0"),
	_071("0.7.1"),
	_072("0.7.2"),
	_080("0.8.0"),
	_081("0.8.1"),
	_082("0.8.2"),
	_083("0.8.3"),
	_090("0.9.0"),
	_091("0.9.1"),
	_092("0.9.2"),
	_093("0.9.3"),
	_094("0.9.4"),
	_095("0.9.5"),
	_096("0.9.6"),
	_097("0.9.7"),
	_098("0.9.8");

	private final String value;

	private ModelVersion(String v) {
		this.value = v;
	}

	/**
	 *
	 * @return The String representation of this {@link ModelVersion}.
	 */
	public String getVersion() {
		return this.value;
	}

	/**
	 *
	 * @param version The version string for which the {@link ModelVersion} is
	 *                requested.
	 * @return The {@link ModelVersion} for the given version String or
	 *         <code>null</code> if the given input is not a valid model version.
	 */
	public static ModelVersion getModelVersion(String version) {
		return Arrays.stream(ModelVersion.values())
				.filter(v -> v.value.equals(version))
				.findFirst()
				.orElse(null);
	}

	/**
	 * Returns all {@link ModelVersion}s before the given {@link ModelVersion}.
	 *
	 * @param version The {@link ModelVersion} for which the previous
	 *                {@link ModelVersion}s are requested.
	 * @param include <code>true</code> if the given {@link ModelVersion} should be
	 *                included in the result, <code>false</code> if not.
	 * @return All {@link ModelVersion}s before the given {@link ModelVersion}.
	 */
	public static ModelVersion[] getVersionsBefore(ModelVersion version, boolean include) {
		List<ModelVersion> versions = new ArrayList<>();
		ModelVersion[] allVersions = ModelVersion.values();
		for (int i = 0; i < (include ? version.ordinal() + 1 : version.ordinal()); i++) {
			versions.add(allVersions[i]);
		}
		return versions.toArray(new ModelVersion[0]);
	}

	/**
	 *
	 * @return The version numbers of all versions that are supported for migration.
	 */
	public static List<String> getAllSupportedVersions() {
		final List<String> allSupportedVersions = new ArrayList<>();

		ModelVersion[] allVersions = ModelVersion.values();
		for (int i = 0; i < allVersions.length; i++) {
			allSupportedVersions.add(allVersions[i].value);
		}

		// no support for itea versions
		allSupportedVersions.remove("itea.103");
		allSupportedVersions.remove("itea.110");
		allSupportedVersions.remove("itea.111");

		return allSupportedVersions;
	}

	/**
	 *
	 * @param version The version string to check.
	 * @return <code>true</code> if the given version is a valid version number,
	 *         <code>false</code> if it is invalid.
	 */
	public static boolean isValidVersion(String version) {
		if (version != null) {
			// check if the given version is contained in the collection of supported
			// versions
			return getAllSupportedVersions().contains(version);
		}
		return false;
	}

	/**
	 *
	 * @return The String representation of the latest {@link ModelVersion}.
	 */
	public static String getLatestVersion() {
		ModelVersion[] values = ModelVersion.values();
		return values[values.length-1].value;
	}
}
