/**
 ********************************************************************************
 * Copyright (c) 2019-2020 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.model.util;

import static com.google.common.base.Preconditions.checkArgument;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.eclipse.app4mc.amalthea.model.AbstractProcess;
import org.eclipse.app4mc.amalthea.model.Amalthea;
import org.eclipse.app4mc.amalthea.model.AmaltheaFactory;
import org.eclipse.app4mc.amalthea.model.AmaltheaIndex;
import org.eclipse.app4mc.amalthea.model.AmaltheaServices;
import org.eclipse.app4mc.amalthea.model.ConstraintsModel;
import org.eclipse.app4mc.amalthea.model.LimitType;
import org.eclipse.app4mc.amalthea.model.ProcessRequirement;
import org.eclipse.app4mc.amalthea.model.Runnable;
import org.eclipse.app4mc.amalthea.model.RunnableRequirement;
import org.eclipse.app4mc.amalthea.model.Task;
import org.eclipse.app4mc.amalthea.model.Time;
import org.eclipse.app4mc.amalthea.model.TimeMetric;
import org.eclipse.app4mc.amalthea.model.TimeRequirementLimit;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;

public class ConstraintsUtil {

	// Suppress default constructor
	private ConstraintsUtil() {
		throw new IllegalStateException("Utility class");
	}

	private static final String ARG_NULL_MESSAGE = "Argument is null";

	public static @Nullable Time getDeadline(final @NonNull AbstractProcess process) {
		checkArgument(process != null, ARG_NULL_MESSAGE);
		
		final ConstraintsModel constraintsModel = getConstraintsModel(process);
		if (constraintsModel == null) return null;
		
		return getDeadline(process, constraintsModel);
	}

	public static @Nullable Time getDeadline(final @NonNull AbstractProcess process, final @NonNull ConstraintsModel constModel) {
		checkArgument(process != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);

		List<Time> list = getDeadlineRequirements(process, constModel).stream()
				.map(req -> ((TimeRequirementLimit) req.getLimit()).getLimitValue())
				.filter(Objects::nonNull)
				.sorted(Time::compareTo)
				.collect(Collectors.toList());
		if (list.isEmpty()) return null;
		
		return list.get(0);
	}

	public static @Nullable Time getDeadline(final @NonNull Runnable runnable) {
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		
		final ConstraintsModel constraintsModel = getConstraintsModel(runnable);
		if (constraintsModel == null) return null;

		return getDeadline(runnable, constraintsModel);
	}

	public static @Nullable Time getDeadline(final @NonNull Runnable runnable, final @NonNull ConstraintsModel constModel) {
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);
		
		List<Time> list = getDeadlineRequirements(runnable, constModel).stream()
				.map(req -> ((TimeRequirementLimit) req.getLimit()).getLimitValue())
				.filter(Objects::nonNull)
				.sorted(Time::compareTo)
				.collect(Collectors.toList());
		if (list.isEmpty()) return null;
		
		return list.get(0);
	}

	// ***** Deadline Requirements *****

	public static void addNewDeadlineRequirement(final @NonNull ConstraintsModel container, final @NonNull AbstractProcess process, final @NonNull Time deadline) {
		checkArgument(container != null, ARG_NULL_MESSAGE);
		checkArgument(process != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);

		final TimeRequirementLimit limit = AmaltheaFactory.eINSTANCE.createTimeRequirementLimit();
		limit.setMetric(TimeMetric.RESPONSE_TIME);
		limit.setLimitType(LimitType.UPPER_LIMIT);
		limit.setLimitValue(deadline);
		final ProcessRequirement req = AmaltheaFactory.eINSTANCE.createProcessRequirement();
		req.setName("Process deadline");
		req.setProcess(process);
		req.setLimit(limit);
		container.getRequirements().add(req);
	}

	public static void addNewDeadlineRequirement(final @NonNull ConstraintsModel container, final @NonNull Runnable runnable, final @NonNull Time deadline) {
		checkArgument(container != null, ARG_NULL_MESSAGE);
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);

		final TimeRequirementLimit limit = AmaltheaFactory.eINSTANCE.createTimeRequirementLimit();
		limit.setMetric(TimeMetric.RESPONSE_TIME);
		limit.setLimitType(LimitType.UPPER_LIMIT);
		limit.setLimitValue(deadline);
		final RunnableRequirement req = AmaltheaFactory.eINSTANCE.createRunnableRequirement();
		req.setName("Runnable deadline");
		req.setRunnable(runnable);
		req.setLimit(limit);
		container.getRequirements().add(req);
	}

	public static void updateDeadlineRequirement(final @NonNull Task task, final @NonNull Time deadline) {
		checkArgument(task != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);
		
		final ConstraintsModel constraintsModel = getConstraintsModel(task);
		if (constraintsModel == null) return;

		updateDeadlineRequirement(task, deadline, constraintsModel);
	}

	public static void updateDeadlineRequirement(final @NonNull Task task, final @NonNull Time deadline, final @NonNull ConstraintsModel constModel) {
		checkArgument(task != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);

		final List<ProcessRequirement> requirements = getDeadlineRequirements(task, constModel);
		
		if (requirements.isEmpty()) {
			// create deadline requirement
			addNewDeadlineRequirement(constModel, task, deadline);
		} else {
			// modify first deadline requirement
			ProcessRequirement first = requirements.remove(0);
			((TimeRequirementLimit) first.getLimit()).setLimitValue(deadline);
			// delete the rest
			AmaltheaIndex.deleteAll(requirements);
		}
	}

	public static void updateDeadlineRequirement(final @NonNull Runnable runnable, final @NonNull Time deadline) {
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);
		
		final ConstraintsModel constraintsModel = getConstraintsModel(runnable);
		if (constraintsModel == null) return;
		
		updateDeadlineRequirement(runnable, deadline, constraintsModel);
	}

	public static void updateDeadlineRequirement(final @NonNull Runnable runnable, final @NonNull Time deadline, final @NonNull ConstraintsModel constModel) {
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		checkArgument(deadline != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);

		final List<RunnableRequirement> requirements = getDeadlineRequirements(runnable, constModel);
		
		if (requirements.isEmpty()) {
			// create deadline requirement
			addNewDeadlineRequirement(constModel, runnable, deadline);
		} else {
			// modify first deadline requirement
			RunnableRequirement first = requirements.remove(0);
			((TimeRequirementLimit) first.getLimit()).setLimitValue(deadline);
			// delete the rest
			AmaltheaIndex.deleteAll(requirements);
		}
	}

	public static List<RunnableRequirement> getDeadlineRequirements(final @NonNull Runnable runnable, final @NonNull ConstraintsModel constModel) {
		checkArgument(runnable != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);

		return constModel.getRequirements().stream()
			.filter(req -> req.getLimit().getLimitType() == LimitType.UPPER_LIMIT
				&& req instanceof RunnableRequirement
				&& ((RunnableRequirement) req).getRunnable() == runnable
				&& req.getLimit() instanceof TimeRequirementLimit
				&& ((TimeRequirementLimit) req.getLimit()).getMetric() == TimeMetric.RESPONSE_TIME)
			.map(req -> (RunnableRequirement) req)
			.collect(Collectors.toList());
	}

	public static List<ProcessRequirement> getDeadlineRequirements(final @NonNull AbstractProcess process, final @NonNull ConstraintsModel constModel) {
		checkArgument(process != null, ARG_NULL_MESSAGE);
		checkArgument(constModel != null, ARG_NULL_MESSAGE);

		return constModel.getRequirements().stream()
		.filter(req -> req.getLimit().getLimitType() == LimitType.UPPER_LIMIT
			&& req instanceof ProcessRequirement
			&& ((ProcessRequirement) req).getProcess() == process
			&& req.getLimit() instanceof TimeRequirementLimit
			&& ((TimeRequirementLimit) req.getLimit()).getMetric() == TimeMetric.RESPONSE_TIME)
		.map(req -> (ProcessRequirement) req)
		.collect(Collectors.toList());
	}


	@SuppressWarnings("unused")
	private static ConstraintsModel getConstraintsModel(final @NonNull EObject object) {
		checkArgument(object != null, ARG_NULL_MESSAGE);
		
		Amalthea modelRoot = AmaltheaServices.getContainerOfType(object, Amalthea.class);
		if (modelRoot == null) return null;
		
		return ModelUtil.getOrCreateConstraintsModel(modelRoot);
	}

}
