/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters081.impl;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.base.IConverter;
import org.eclipse.app4mc.amalthea.converters.common.converter.AbstractConverter;
import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.eclipse.app4mc.amalthea.converters081.utils.HelperUtils_080_081;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is responsible for converting the SW Model elements from 0.8.0 to 0.8.1 version format of AMALTHEA model
 *
 * @author mez2rng
 *
 */
@Component(
		property = {
			ServiceConstants.INPUT_MODEL_VERSION_PROPERTY + "=0.8.0",
			ServiceConstants.OUTPUT_MODEL_VERSION_PROPERTY + "=0.8.1"
		},
		service = IConverter.class)
public class SwConverter extends AbstractConverter {

	private static final Logger LOGGER = LoggerFactory.getLogger(SwConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> fileName_documentsMap, List<ICache> caches) {

		LOGGER.info("Migration from 0.8.0 to 0.8.1 : Executing Software converter for model file : {}",
				targetFile.getName());


		final Document root = fileName_documentsMap.get(targetFile);

		if (root == null) {
			return;
		}
		final Element rootElement = root.getRootElement();

		updateRunnable(rootElement);

		updateActivations(rootElement);

		updateAllAbstractProcessElements(rootElement);
	}

	/**
	 * This method is used to remove priority from sub-classes of AbstractProcess (For further
	 * details, check : Bug 511284, 518070  )
	 *
	 *
	 * @param rootElement
	 *            Amalthea root element
	 */
	private void updateAllAbstractProcessElements(Element rootElement) {

		final StringBuilder xpathBuffer = new StringBuilder();

		xpathBuffer.append("./swModel/tasks");
		xpathBuffer.append("|");
		xpathBuffer.append("./swModel/isrs");
		xpathBuffer.append("|");
		xpathBuffer.append("./swModel/processPrototypes");

		final List<Element> abstractProcessElements = HelperUtil.getXpathResult(
				rootElement,
				xpathBuffer.toString(),
				Element.class,
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element abstractProcessElement : abstractProcessElements) {

			Attribute priorityAttribute = abstractProcessElement.getAttribute("priority");

			if(priorityAttribute !=null){
				String value = priorityAttribute.getValue();

				/*-- removing attribute based on the metamodel changes introduced in 0.8.1 --*/

				abstractProcessElement.removeAttribute(priorityAttribute);

				if(!value.equals("0")){

					Element customPropertiesElement=new Element("customProperties");

					customPropertiesElement.setAttribute("key", "priority");

					Element valueElement=new Element("value");

					valueElement.setAttribute("type", "am:StringObject", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

					valueElement.setAttribute("value", value);

					customPropertiesElement.addContent(valueElement);

					abstractProcessElement.addContent(customPropertiesElement);

					LOGGER.info("Priority is removed from : {} element ({}) and added as a CustomProperty with key as Priority",
							abstractProcessElement.getName(),
							abstractProcessElement.getAttributeValue("name"));

				}

			}

			/*-- removing osekTaskGroup attribute inside Task element:  based on the metamodel changes introduced in 0.8.1 --*/

			if(abstractProcessElement.getName().equals("tasks")){

				 Attribute osekTaskGroupAttribute = abstractProcessElement.getAttribute("osekTaskGroup");

				 if(osekTaskGroupAttribute !=null){

					 String osekTaskGroupValue = osekTaskGroupAttribute.getValue();

					 abstractProcessElement.removeAttribute(osekTaskGroupAttribute);

					 if(!osekTaskGroupValue.equals("0")){
						 HelperUtils_080_081.addCustomProperty(abstractProcessElement, "osekTaskGroup", osekTaskGroupValue);

						 LOGGER.info("osekTaskGroup attribute is removed from Task ({}) and added as a CustomProperty with key as osekTaskGroup",
								 abstractProcessElement.getAttributeValue("name"));
					 }

				 }


			}

		}


	}

	/**
	 * This method is used to migrate the Runnable data to support multiple Activations (For further
	 * details, check : Bug 519357  )
	 *
	 *
	 * @param rootElement
	 *            Amalthea root element
	 */
	private void updateRunnable(Element rootElement) {


		final StringBuilder xpathBuffer = new StringBuilder();

		xpathBuffer.append("./swModel/runnables");


		final List<Element> runnableElements = HelperUtil.getXpathResult(
				rootElement,
				xpathBuffer.toString(),
				Element.class,
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));



		for (Element runnableElement : runnableElements) {

			Attribute activationAttribute = runnableElement.getAttribute("activation");
			Element activationElement = runnableElement.getChild("activation");

			if(activationAttribute !=null){
				activationAttribute.setName("activations");
			}else if(activationElement !=null){
				activationElement.setName("activations");
			}
		}




	}
	/**
	 * This method is used to migrate both PeriodicAtivation (removal of deadline ) and EventActivation (change of trigger tag to triggeringEvents)
	 *
	 * @param rootElement
	 *            Amalthea root element
	 */
	private void updateActivations(Element rootElement) {


		final StringBuilder xpathBuffer = new StringBuilder();

		xpathBuffer.append("./swModel/activations");


		final List<Element> activationElements = HelperUtil.getXpathResult(
				rootElement,
				xpathBuffer.toString(),
				Element.class,
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));


		boolean removedDeadLine=false;

		for (Element activationElement : activationElements) {

			if(activationElement.getAttributeValue("type", AmaltheaNamespaceRegistry.getGenericNamespace("xsi")).contains("am:PeriodicActivation")){

				Element deadlineElement = activationElement.getChild("deadline");

				if(deadlineElement !=null){

					deadlineElement.detach();


					Element customPropertiesElement=new Element("customProperties");

					customPropertiesElement.setAttribute("key", "deadline");

					deadlineElement.setName("value");

					deadlineElement.setAttribute("type", "am:TimeObject", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

					//adding as a value to CustomProperty
					customPropertiesElement.addContent(deadlineElement);

					//adding customProperty to Activation Element
					activationElement.addContent(customPropertiesElement);

				removedDeadLine=true;

				}

			}else if(activationElement.getAttributeValue("type", AmaltheaNamespaceRegistry.getGenericNamespace("xsi")).contains("am:EventActivation")){

				List<Element> triggerElements = activationElement.getChildren("trigger");

				if(triggerElements !=null){
					for (Element element : triggerElements) {
						element.setName("triggeringEvents");
					}
				}

				Attribute triggerAttribute = activationElement.getAttribute("trigger");

				if(triggerAttribute!=null){
					triggerAttribute.setName("triggeringEvents");
				}


			}

		}

		if(removedDeadLine){
			LOGGER.warn("-- Deadline inside PeriodicActivation elements is removed, as there is no equivalent element for it in AMALTHEA 0.8.1");
		}
	}

}
