/**
 ********************************************************************************
 * Copyright (c) 2021 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.validations.sim.os;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.app4mc.amalthea.model.AmaltheaIndex;
import org.eclipse.app4mc.amalthea.model.InterruptController;
import org.eclipse.app4mc.amalthea.model.Scheduler;
import org.eclipse.app4mc.amalthea.model.SchedulerAllocation;
import org.eclipse.app4mc.amalthea.validation.core.AmaltheaValidation;
import org.eclipse.app4mc.validation.annotation.Validation;
import org.eclipse.app4mc.validation.core.ValidationDiagnostic;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;

/**
 * Checks if scheduler is referenced by one scheduler allocation mapping
 * 
 * <ul>
 * <li>Each scheduler must be linked to a processing unit. Therefore a unique
 *  scheduler allocation mapping is necessary.
 *  Further checks (e.g. mapping to a processing unit) are available in the
 *  mapping model validations</li>
 * </ul>
 */

@Validation(
		id = "Sim-OS-SchedulerAllocation",
		checks = { "Scheduler must be referred to by exactly one scheduler allocation" })

public class SimOsSchedulerAllocation extends AmaltheaValidation {

	@Override
	public EClassifier getEClassifier() {
		return ePackage.getScheduler();
	}

	@Override
	public void validate(EObject eObject, List<ValidationDiagnostic> results) {
		
		if (eObject instanceof InterruptController) {
			InterruptController isrController = (InterruptController) eObject;
			Set<?>allocations = AmaltheaIndex.getReferringObjects(isrController).stream().filter((referer) -> referer instanceof SchedulerAllocation).collect(Collectors.toSet());
			if (allocations.isEmpty()) {
				addIssue(results, isrController, ePackage.getScheduler_SchedulerAllocations(), getMessageNoInterruptControllerAllocation());
			} else if (allocations.size() > 1) {
				addIssue(results, isrController, ePackage.getScheduler_SchedulerAllocations(), getMessageManyInterruptControllerAllocations());
			}
		} else if (eObject instanceof Scheduler){ //TaskScheduler
			Scheduler scheduler = (Scheduler) eObject;
			Set<?>allocations = AmaltheaIndex.getReferringObjects(scheduler).stream().filter((referer) -> referer instanceof SchedulerAllocation).collect(Collectors.toSet());
			if (allocations.isEmpty()) {
				addIssue(results, scheduler, ePackage.getScheduler_SchedulerAllocations(), getMessageNoSchedulerAllocation());
			} else if (allocations.size() > 1) {
				addIssue(results, scheduler, ePackage.getScheduler_SchedulerAllocations(), getMessageManySchedulerAllocations());
			}
		}
	}
	

	public static String getMessageNoSchedulerAllocation() {
		return "Scheduler is not referred to by any scheduler allocation";
	}
	
	public static String getMessageManySchedulerAllocations() {
		return "Scheduler is referred to by more than one scheduler allocation";
	}
	
	public static String getMessageNoInterruptControllerAllocation() {
		return "ISR is not referred to by any scheduler allocation";
	}
	
	public static String getMessageManyInterruptControllerAllocations() {
		return "ISR is referred to by more than one scheduler allocation";
	}



}




