/**
 * Copyright (c) 2018-2022 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 */
package org.eclipse.app4mc.amalthea.visualization.hw.templates;

import java.util.HashSet;
import java.util.List;
import org.eclipse.app4mc.amalthea.model.HwAccessElement;
import org.eclipse.app4mc.amalthea.model.HwConnection;
import org.eclipse.app4mc.amalthea.model.HwDestination;
import org.eclipse.app4mc.amalthea.model.HwModule;
import org.eclipse.app4mc.amalthea.model.HwPort;
import org.eclipse.app4mc.amalthea.model.HwStructure;
import org.eclipse.app4mc.amalthea.model.INamed;
import org.eclipse.app4mc.amalthea.model.ProcessingUnit;
import org.eclipse.app4mc.visualization.util.svg.PlantUmlDiagram;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class HWBlockGenerator {
  private static class State {
    private boolean errorFlag;
    
    private String errorMessage;
    
    public boolean State() {
      return this.errorFlag = false;
    }
    
    public boolean getErrorFlag() {
      return this.errorFlag;
    }
    
    public boolean setErrorFlag(final boolean flag) {
      return this.errorFlag = flag;
    }
    
    public boolean error() {
      return this.errorFlag;
    }
    
    public boolean success() {
      return (!this.errorFlag);
    }
    
    public String getErrorMessage() {
      return this.errorMessage;
    }
    
    public void setErrorMessage(final String message) {
      this.errorMessage = message;
    }
  }
  
  public static void updateDiagram(final PlantUmlDiagram diagram, final List<HwStructure> structures) {
    diagram.resetDiagramData();
    final HWBlockGenerator.State state = new HWBlockGenerator.State();
    final HashSet<String> commands = new HashSet<String>();
    final CharSequence result = HWBlockGenerator.generateStructures(structures, commands, state);
    boolean _success = state.success();
    if (_success) {
      diagram.append(result);
    } else {
      diagram.append(HWBlockGenerator.error(state.errorMessage));
    }
  }
  
  private static CharSequence error(final String message) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@startuml");
    _builder.newLine();
    _builder.append("left to right direction");
    _builder.newLine();
    _builder.append("actor User");
    _builder.newLine();
    _builder.append("rectangle Visualization {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("usecase \"Invalid Input\" as ii");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.append("User ---> ii");
    _builder.newLine();
    _builder.append("note top of ii");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append(message, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("end note");
    _builder.newLine();
    _builder.append("@enduml");
    _builder.newLine();
    return _builder;
  }
  
  private static CharSequence generateStructures(final List<HwStructure> structures, final HashSet<String> commands, final HWBlockGenerator.State state) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@startuml");
    _builder.newLine();
    {
      boolean _isEmpty = structures.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.newLine();
        _builder.append("\' ===== FRAMES =====");
        _builder.newLine();
        {
          for(final HwStructure s : structures) {
            CharSequence _generateLevels = HWBlockGenerator.generateLevels(s, commands, state);
            _builder.append(_generateLevels);
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.newLine();
        _builder.append("\' ===== ROUTES =====");
        _builder.newLine();
        {
          for(final HwStructure s_1 : structures) {
            CharSequence _generateRoutes = HWBlockGenerator.generateRoutes(s_1, commands, state);
            _builder.append(_generateRoutes);
            _builder.newLineIfNotEmpty();
          }
        }
      } else {
        _builder.append(state.errorFlag = true);
        _builder.newLineIfNotEmpty();
        _builder.append(state.errorMessage = " No Structure in the model. ");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("skinparam component {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("backgroundColor<<ProcessingUnit>> #8CACFF");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("borderColor<<ProcessingUnit>> #000000");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("backgroundColor<<ConnectionHandler>> #FFFFA0");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("borderColor<<ConnectionHandler>> #000000");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("backgroundColor<<Memory>> #60FF82");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("borderColor<<Memory>> #000000");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("backgroundColor<<Cache>> #C8FFA6");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("borderColor<<Cache>> #000000");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("ArrowFontColor #C0C0C0");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@enduml");
    _builder.newLine();
    return _builder;
  }
  
  private static CharSequence generateLevels(final HwStructure s, final HashSet<String> commands, final HWBlockGenerator.State state) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _not = (!((s == null) || StringExtensions.isNullOrEmpty(s.getName())));
      if (_not) {
        _builder.newLine();
        _builder.append("\t");
        _builder.append("frame \"");
        String _name = s.getName();
        _builder.append(_name, "\t");
        _builder.append("\" as ");
        String _convertString = HWBlockGenerator.convertString(s.getName());
        _builder.append(_convertString, "\t");
        _builder.append(" {");
        _builder.newLineIfNotEmpty();
        {
          boolean _isEmpty = s.getModules().isEmpty();
          boolean _not_1 = (!_isEmpty);
          if (_not_1) {
            _builder.append("\t");
            _builder.newLine();
            {
              EList<HwModule> _modules = s.getModules();
              for(final HwModule m : _modules) {
                {
                  boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(m.getName());
                  boolean _not_2 = (!_isNullOrEmpty);
                  if (_not_2) {
                    _builder.append("\t");
                    _builder.append("\t");
                    CharSequence _addLevel = HWBlockGenerator.addLevel(m, commands);
                    _builder.append(_addLevel, "\t\t");
                    _builder.newLineIfNotEmpty();
                  } else {
                    _builder.append("\t");
                    _builder.append("\t");
                    _builder.append(state.errorFlag = true, "\t\t");
                    _builder.newLineIfNotEmpty();
                    _builder.append("\t");
                    _builder.append("\t");
                    _builder.append(state.errorMessage = " Missing HW Module name. ", "\t\t");
                    _builder.newLineIfNotEmpty();
                  }
                }
              }
            }
          }
        }
        {
          EList<HwStructure> _structures = s.getStructures();
          for(final HwStructure su : _structures) {
            _builder.append("\t");
            CharSequence _generateLevels = HWBlockGenerator.generateLevels(su, commands, state);
            _builder.append(_generateLevels, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
      } else {
        _builder.append(state.errorFlag = true);
        _builder.newLineIfNotEmpty();
        _builder.append(state.errorMessage = " Name of HwStructure is missing. ");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }
  
  private static CharSequence addLevel(final HwModule m, final HashSet<String> commands) {
    String _xblockexpression = null;
    {
      final String name = HWBlockGenerator.convertString(m.getName());
      String _name = m.eClass().getName();
      String _plus = ((((("[" + name) + "] as ") + name) + " <<") + _name);
      final String command = (_plus + ">>");
      String _xifexpression = null;
      boolean _contains = commands.contains(command);
      if (_contains) {
        _xifexpression = "";
      } else {
        String _xblockexpression_1 = null;
        {
          commands.add(command);
          _xblockexpression_1 = command;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private static CharSequence generateRoutes(final HwStructure s, final HashSet<String> commands, final HWBlockGenerator.State state) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _not = (!((s == null) || s.getConnections().isEmpty()));
      if (_not) {
        _builder.newLine();
        _builder.append("\' add physical connections of structure ");
        String _name = s.getName();
        _builder.append(_name);
        _builder.append(":");
        _builder.newLineIfNotEmpty();
        {
          EList<HwConnection> _connections = s.getConnections();
          for(final HwConnection c : _connections) {
            _builder.newLine();
            _builder.append("\t");
            HwPort _port1 = c.getPort1();
            EObject _eContainer = null;
            if (_port1!=null) {
              _eContainer=_port1.eContainer();
            }
            HwPort _port2 = c.getPort2();
            EObject _eContainer_1 = null;
            if (_port2!=null) {
              _eContainer_1=_port2.eContainer();
            }
            CharSequence _addRoute = HWBlockGenerator.addRoute(((INamed) _eContainer), ((INamed) _eContainer_1), c, " -- ", commands);
            _builder.append(_addRoute, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    {
      boolean _not_1 = (!((s == null) || s.getModules().isEmpty()));
      if (_not_1) {
        {
          EList<HwModule> _modules = s.getModules();
          for(final HwModule m : _modules) {
            {
              if ((m instanceof ProcessingUnit)) {
                final ProcessingUnit pUnit = ((ProcessingUnit) m);
                _builder.newLineIfNotEmpty();
                {
                  boolean _isEmpty = pUnit.getAccessElements().isEmpty();
                  boolean _not_2 = (!_isEmpty);
                  if (_not_2) {
                    _builder.newLine();
                    _builder.append("\' add logical connections of processing unit ");
                    String _name_1 = pUnit.getName();
                    _builder.append(_name_1);
                    _builder.append(":");
                    _builder.newLineIfNotEmpty();
                    {
                      EList<HwAccessElement> _accessElements = pUnit.getAccessElements();
                      for(final HwAccessElement ae : _accessElements) {
                        {
                          HwDestination _destination = ae.getDestination();
                          boolean _tripleNotEquals = (_destination != null);
                          if (_tripleNotEquals) {
                            _builder.newLine();
                            _builder.append("\t");
                            CharSequence _addRoute_1 = HWBlockGenerator.addRoute(pUnit, ae.getDestination(), ae, " ..>", commands);
                            _builder.append(_addRoute_1, "\t");
                            _builder.newLineIfNotEmpty();
                          } else {
                            _builder.append(state.errorFlag = true);
                            _builder.newLineIfNotEmpty();
                            _builder.append(state.errorMessage = " Missing Destination of AccessElement. ");
                            _builder.newLineIfNotEmpty();
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    {
      boolean _not_3 = (!((s == null) || s.getStructures().isEmpty()));
      if (_not_3) {
        {
          EList<HwStructure> _structures = s.getStructures();
          for(final HwStructure su : _structures) {
            CharSequence _generateRoutes = HWBlockGenerator.generateRoutes(su, commands, state);
            _builder.append(_generateRoutes);
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }
  
  private static CharSequence addRoute(final INamed source, final INamed destination, final INamed con, final String conType, final HashSet<String> commands) {
    String _xblockexpression = null;
    {
      final String name1 = HWBlockGenerator.convertString(HWBlockGenerator.name(source));
      final String name2 = HWBlockGenerator.convertString(HWBlockGenerator.name(destination));
      String _xifexpression = null;
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(con.getName());
      if (_isNullOrEmpty) {
        _xifexpression = con.toString();
      } else {
        _xifexpression = con.getName();
      }
      final String conLabel = _xifexpression;
      final String command = ((((name1 + conType) + name2) + " : ") + conLabel);
      String _xifexpression_1 = null;
      boolean _contains = commands.contains(command);
      if (_contains) {
        _xifexpression_1 = "";
      } else {
        String _xblockexpression_1 = null;
        {
          commands.add(command);
          _xblockexpression_1 = command;
        }
        _xifexpression_1 = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  private static String name(final INamed obj) {
    if ((obj == null)) {
      return "<undefined>";
    }
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(obj.getName());
    if (_isNullOrEmpty) {
      return "?";
    }
    return obj.getName();
  }
  
  private static String convertString(final String str) {
    String _xifexpression = null;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(str);
    if (_isNullOrEmpty) {
      _xifexpression = "<no name>";
    } else {
      _xifexpression = HWBlockGenerator.convertGenericString(str, "_");
    }
    return _xifexpression;
  }
  
  private static String convertGenericString(final String instring, final String replacestring) {
    final String Regex = "/";
    final String tmpstr0 = instring.replaceAll(Regex, replacestring);
    final String Regex1 = " ";
    final String tmpstr1 = tmpstr0.replaceAll(Regex1, replacestring);
    final String Regex2 = "\\(";
    final String tmpstr2 = tmpstr1.replaceAll(Regex2, replacestring);
    final String Regex3 = "\\)";
    final String tmpstr3 = tmpstr2.replaceAll(Regex3, replacestring);
    final String Regex4 = "-";
    final String tmpstr4 = tmpstr3.replaceAll(Regex4, replacestring);
    final String Regex5 = "\\+";
    final String tmpstr5 = tmpstr4.replaceAll(Regex5, replacestring);
    final String Regex6 = "%2f";
    final String tmpstr6 = tmpstr5.replaceAll(Regex6, replacestring);
    final String Regex7 = "\\(";
    final String tmpstr7 = tmpstr6.replaceAll(Regex7, replacestring);
    final String Regex8 = "\\)";
    final String tmpstr8 = tmpstr7.replaceAll(Regex8, replacestring);
    final String Regex9 = "%2b";
    final String Replace9 = "plus";
    final String tmpstr9 = tmpstr8.replaceAll(Regex9, Replace9);
    final String Regex10 = "%3a%3a";
    final String tmpstr10 = tmpstr9.replaceAll(Regex10, replacestring);
    final String Regex11 = "\\.";
    final String tmpstr11 = tmpstr10.replaceAll(Regex11, replacestring);
    final String Regex12 = "#";
    final String tmpstr12 = tmpstr11.replaceAll(Regex12, replacestring);
    final String Regex13 = "\"";
    final String tmpstr13 = tmpstr12.replaceAll(Regex13, replacestring);
    final String Regex14 = "$";
    final String tmpstr14 = tmpstr13.replaceAll(Regex14, replacestring);
    final String Regex15 = "~";
    final String tmpstr15 = tmpstr14.replaceAll(Regex15, replacestring);
    final String Regex16 = "%";
    final String tmpstr16 = tmpstr15.replaceAll(Regex16, replacestring);
    final String Regex17 = "&";
    final String tmpstr17 = tmpstr16.replaceAll(Regex17, replacestring);
    return tmpstr17;
  }
}
