/**
 ********************************************************************************
 * Copyright (c) 2017-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters093.utils;

import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.jdom2.Attribute;
import org.jdom2.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HelperUtils_092_093 {

	public static final Logger LOGGER = LoggerFactory.getLogger(HelperUtils_092_093.class);

	private HelperUtils_092_093() {
		// empty private default constructor
	}

	/**
	 * This method migrates the data of Deviation element (and the corresponding sub-elements)
	 * into the equivalent semantics as per 0.9.3 <br>
	 * <b>Note:</b><br>
	 * Below are the mappings for distribution<br>
	 *
	 * <table>
	 * 		<th>0.9.2</th>					<th>0.9.3</th>
	 * <tr>	<td>Boundaries</td>				<td>DiscreteValueBoundaries</td> </tr>
	 * <tr>	<td>UniformDistribution</td>	<td>DiscreteValueUniformDistribution</td> </tr>
	 * <tr>	<td>BetaDistribution</td>		<td>DiscreteValueBetaDistribution</td> </tr>
	 * <tr>	<td>WeibullEstimators</td>		<td>DiscreteValueWeibullEstimatorsDistribution</td> </tr>
	 * <tr>	<td>GaussDistribution</td>		<td>DiscreteValueGaussDistribution</td> </tr>
	 * </table>
	 * 
	 * @param oldDeviationElement
	 * @param newElementName
	 * @param ipcValue
	 * @return
	 */
	public static Element migrateDeviationElement_Containing_LongValue(Element oldDeviationElement, String newElementName,
			double ipcValue) {

		if (oldDeviationElement == null) return null;
		
		Element newDeviationElement = new Element(newElementName);

		Element lowerBound = oldDeviationElement.getChild("lowerBound");
		if (lowerBound != null) {
			String value = lowerBound.getAttributeValue("value");
			if (value != null) {
				newDeviationElement.setAttribute(
						new Attribute("lowerBound", getValueAfterApplyingIPC(value, ipcValue)));
			}
		}

		Element upperBound = oldDeviationElement.getChild("upperBound");
		if (upperBound != null) {
			String value = upperBound.getAttributeValue("value");
			if (value != null) {
				newDeviationElement.setAttribute(
						new Attribute("upperBound", getValueAfterApplyingIPC(value, ipcValue)));
			}
		}

		Element oldDistributionElement = oldDeviationElement.getChild("distribution");
		if (oldDistributionElement == null) {
			// fallback if no distribution is specified
			setType(newDeviationElement, "am:DiscreteValueBoundaries");
			return newDeviationElement;
		}
		
		String oldDistributionType = oldDistributionElement.getAttributeValue("type",
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));


		if (oldDistributionType.equals("am:BetaDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:LongObject" value="10"/>
				<upperBound xsi:type="am:LongObject" value="20"/>
				<distribution xsi:type="am:BetaDistribution" alpha="0.0" beta="0.0"/>
			</deviation>
			*/

			setType(newDeviationElement, "am:DiscreteValueBetaDistribution");

			String alpha = oldDistributionElement.getAttributeValue("alpha");
			if (alpha != null) {
				newDeviationElement.setAttribute(new Attribute("alpha", alpha));
			}
			
			String beta = oldDistributionElement.getAttributeValue("beta");
			if (beta != null) {
				newDeviationElement.setAttribute(new Attribute("beta", beta));
			}

		} else if (oldDistributionType.equals("am:Boundaries")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:LongObject" value="0"/>
				<upperBound xsi:type="am:LongObject" value="0"/>
				<distribution xsi:type="am:Boundaries" samplingType="WorstCase"/>
			</deviation>
			*/

			setType(newDeviationElement, "am:DiscreteValueBoundaries");

			String samplingType = oldDistributionElement.getAttributeValue("samplingType");
			if (samplingType != null) {
				newDeviationElement.setAttribute(new Attribute("samplingType", samplingType));
			}

		} else if (oldDistributionType.equals("am:GaussDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:LongObject" value="10"/>
				<upperBound xsi:type="am:LongObject" value="20"/>
				<distribution xsi:type="am:GaussDistribution">
					<sd xsi:type="am:LongObject" value="5"/>
					<mean xsi:type="am:LongObject" value="4"/>
				</distribution>
			</deviation>
			*/

			setType(newDeviationElement, "am:DiscreteValueGaussDistribution");

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			if (mean != null) {
				newDeviationElement.setAttribute(new Attribute("mean", getValueAfterApplyingIPC(mean, ipcValue)));
			}

			String sd = HelperUtil.getValueFromChildElement(oldDistributionElement, "sd", "value");
			if (sd != null) {
				newDeviationElement.setAttribute(new Attribute("sd", sd));
			}

		} else if (oldDistributionType.equals("am:UniformDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:LongObject" value="0"/>
				<upperBound xsi:type="am:LongObject" value="0"/>
				<distribution xsi:type="am:UniformDistribution"/>
			 </deviation>
			 */

			setType(newDeviationElement, "am:DiscreteValueUniformDistribution");

		} else if (oldDistributionType.equals("am:WeibullEstimators")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:LongObject" value="10"/>
				<upperBound xsi:type="am:LongObject" value="20"/>
				<distribution xsi:type="am:WeibullEstimators" pRemainPromille="5.0">
					<mean xsi:type="am:LongObject" value="20"/>
				</distribution>
			</deviation>
			*/

			setType(newDeviationElement, "am:DiscreteValueWeibullEstimatorsDistribution");

			String pRemainPromille = oldDistributionElement.getAttributeValue("pRemainPromille");
			if (pRemainPromille != null) {
				newDeviationElement.setAttribute(new Attribute("pRemainPromille", pRemainPromille));
			}

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			if (mean != null) {
				newDeviationElement
						.setAttribute(new Attribute("average", getValueAfterApplyingIPC(mean, ipcValue)));
			}

		} else if (oldDistributionType.equals("am:WeibullParameters")) {

			LOGGER.error(
					"Deviation with \"WeibullParameters\" as distribution can not be migrated as the semantics of \"WeibullParameters\" were not clearly described");
			return null;
		}

		return newDeviationElement;
	}

	/**
	 * This method migrates the data of Deviation element (and the corresponding sub-elements)
	 * into the equivalent semantics as per 0.9.3 <br>
	 * <b>Note:</b><br>
	 * Below are the mappings for distribution<br>
	 *
	 * <table>
	 * 		<th>0.9.2</th>					<th>0.9.3</th>
	 * <tr>	<td>Boundaries</td>				<td>ContinuousValueBoundaries</td> </tr>
	 * <tr>	<td>UniformDistribution</td>	<td>ContinuousValueUniformDistribution</td> </tr>
	 * <tr>	<td>BetaDistribution</td>		<td>ContinuousValueBetaDistribution</td> </tr>
	 * <tr>	<td>WeibullEstimators</td>		<td>ContinuousValueWeibullEstimatorsDistribution</td> </tr>
	 * <tr>	<td>GaussDistribution</td>		<td>ContinuousValueGaussDistribution</td> </tr>
	 * </table>
	 * 
	 * @param oldDeviationElement
	 * @param newElementName
	 * @return
	 */

	public static Element migrateDeviationElement_Containing_DoubleValue(Element oldDeviationElement, String newElementName) {

		if (oldDeviationElement == null) return null;
		
		Element newDeviationElement = new Element(newElementName);

		Element lowerBound = oldDeviationElement.getChild("lowerBound");
		if (lowerBound != null) {
			String value = lowerBound.getAttributeValue("value");
			if (value != null) {
				newDeviationElement.setAttribute(new Attribute("lowerBound", value));
			}
		}

		Element upperBound = oldDeviationElement.getChild("upperBound");
		if (upperBound != null) {
			String value = upperBound.getAttributeValue("value");
			if (value != null) {
				newDeviationElement.setAttribute(new Attribute("upperBound", value));
			}
		}

		Element oldDistributionElement = oldDeviationElement.getChild("distribution");
		if (oldDistributionElement == null) {
			// fallback if no distribution is specified
			setType(newDeviationElement, "am:ContinuousValueBoundaries");
			return newDeviationElement;
		}
		
		String oldDistributionType = oldDistributionElement.getAttributeValue("type",
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));


		if (oldDistributionType.equals("am:BetaDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="10"/>
				<upperBound xsi:type="am:DoubleObject" value="20"/>
				<distribution xsi:type="am:BetaDistribution" alpha="0.0" beta="0.0"/>
			</deviation>
			*/

			setType(newDeviationElement, "am:ContinuousValueBetaDistribution");

			String alpha = oldDistributionElement.getAttributeValue("alpha");
			if (alpha != null) {
				newDeviationElement.setAttribute(new Attribute("alpha", alpha));
			}

			String beta = oldDistributionElement.getAttributeValue("beta");
			if (beta != null) {
				newDeviationElement.setAttribute(new Attribute("beta", beta));
			}

		} else if (oldDistributionType.equals("am:Boundaries")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="0"/>
				<upperBound xsi:type="am:DoubleObject" value="0"/>
				<distribution xsi:type="am:Boundaries" samplingType="WorstCase"/>
			</deviation>
			*/

			setType(newDeviationElement, "am:ContinuousValueBoundaries");

			String samplingType = oldDistributionElement.getAttributeValue("samplingType");
			if (samplingType != null) {
				newDeviationElement.setAttribute(new Attribute("samplingType", samplingType));
			}

		} else if (oldDistributionType.equals("am:GaussDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="10"/>
				<upperBound xsi:type="am:DoubleObject" value="20"/>
				<distribution xsi:type="am:GaussDistribution">
					<sd xsi:type="am:DoubleObject" value="5"/>
					<mean xsi:type="am:DoubleObject" value="4"/>
				</distribution>
			</deviation>
			*/

			setType(newDeviationElement, "am:ContinuousValueGaussDistribution");

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			if (mean != null) {
				newDeviationElement.setAttribute(new Attribute("mean", mean));
			}

			String sd = HelperUtil.getValueFromChildElement(oldDistributionElement, "sd", "value");
			if (sd != null) {
				newDeviationElement.setAttribute(new Attribute("sd", sd));
			}

		} else if (oldDistributionType.equals("am:UniformDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="0"/>
				<upperBound xsi:type="am:DoubleObject" value="0"/>
				<distribution xsi:type="am:UniformDistribution"/>
			</deviation>
			*/

			setType(newDeviationElement, "am:ContinuousValueUniformDistribution");

		} else if (oldDistributionType.equals("am:WeibullEstimators")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="10"/>
				<upperBound xsi:type="am:DoubleObject" value="20"/>
				<distribution xsi:type="am:WeibullEstimators" pRemainPromille="5.0">
					<mean xsi:type="am:DoubleObject" value="20"/>
				</distribution>
			</deviation>
			*/

			setType(newDeviationElement, "am:ContinuousValueWeibullEstimatorsDistribution");

			String pRemainPromille = oldDistributionElement.getAttributeValue("pRemainPromille");
			if (pRemainPromille != null) {
				newDeviationElement.setAttribute(new Attribute("pRemainPromille", pRemainPromille));
			}

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			if (mean != null) {
				newDeviationElement.setAttribute(new Attribute("average", mean));
			}

		} else if (oldDistributionType.equals("am:WeibullParameters")) {

			LOGGER.error(
					"Deviation with \"WeibullParameters\" as distribution can not be migrated as the semantics of \"WeibullParameters\" were not clearly described");

			return null;
		}

		return newDeviationElement;
	}

	/**
	 * - This method is used to migrate the data of Deviation element (and the
	 * corresponding sub-element : Distribution, both referring to Time as value)
	 * into the equivalent semantics as per 0.9.3 <br>
	 * <b>Note:</b><br>
	 * Below are the mappings for distribution<br>
	 *
	 * <table>
	 * 		<th>0.9.2</th>					<th>0.9.3</th>
	 * <tr>	<td>Boundaries</td>				<td>TimeBoundaries</td> </tr>
	 * <tr>	<td>UniformDistribution</td>	<td>TimeUniformDistribution</td> </tr>
	 * <tr>	<td>BetaDistribution</td>		<td>TimeBetaDistribution</td> </tr>
	 * <tr>	<td>WeibullEstimators</td>		<td>TimeWeibullEstimatorsDistribution</td> </tr>
	 * <tr>	<td>GaussDistribution</td>		<td>TimeGaussDistribution</td> </tr>
	 * </table>
	 * 
	 * @param oldDeviationElement
	 * @param newElementName
	 * @return
	 */

	public static Element migrateDeviationElement_Containing_TimeValue(Element oldDeviationElement, String newElementName) {

		if (oldDeviationElement == null) return null;
		
		Element newDeviationElement = new Element(newElementName);

		Element lowerBound = oldDeviationElement.getChild("lowerBound");
		if (lowerBound != null) {
			Element lowerBoundElement = new Element("lowerBound");
			newDeviationElement.addContent(lowerBoundElement);
			
			String valueString = lowerBound.getAttributeValue("value");
			if (valueString != null) {
				lowerBoundElement.setAttribute("value", valueString);
			}

			String unitString = lowerBound.getAttributeValue("unit");
			if (unitString != null) {
				lowerBoundElement.setAttribute("unit", unitString);
			}
		}

		Element upperBound = oldDeviationElement.getChild("upperBound");
		if (upperBound != null) {
			Element upperBoundElement = new Element("upperBound");
			newDeviationElement.addContent(upperBoundElement);
			
			String valueString = upperBound.getAttributeValue("value");
			if (valueString != null) {
				upperBoundElement.setAttribute("value", valueString);
			}

			String unitString = upperBound.getAttributeValue("unit");
			if (unitString != null) {
				upperBoundElement.setAttribute("unit", unitString);
			}
		}

		Element oldDistributionElement = oldDeviationElement.getChild("distribution");
		if (oldDistributionElement == null) {
			// fallback if no distribution is specified
			setType(newDeviationElement, "am:TimeBoundaries");
			return newDeviationElement;
		}
		
		String oldDistributionType = oldDistributionElement.getAttributeValue("type",
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));


		if (oldDistributionType.equals("am:BetaDistribution")) {
			/*-
			<nextOccurrence>
				<lowerBound xsi:type="am:Time" value="66" unit="us"/>
				<upperBound xsi:type="am:Time" value="2" unit="ms"/>
				<distribution xsi:type="am:BetaDistribution" alpha="0.0" beta="0.0"/>
			</nextOccurrence>
			*/

			setType(newDeviationElement, "am:TimeBetaDistribution");

			String alpha = oldDistributionElement.getAttributeValue("alpha");
			if (alpha != null) {
				newDeviationElement.setAttribute(new Attribute("alpha", alpha));
			}
			
			String beta = oldDistributionElement.getAttributeValue("beta");
			if (beta != null) {
				newDeviationElement.setAttribute(new Attribute("beta", beta));
			}

		} else if (oldDistributionType.equals("am:Boundaries")) {
			/*-
			<nextOccurrence>
				<lowerBound xsi:type="am:Time" value="0"/>
				<upperBound xsi:type="am:Time" value="0" unit="s"/>
				<distribution xsi:type="am:Boundaries" samplingType="BestCase"/>
			</nextOccurrence>
			*/

			setType(newDeviationElement, "am:TimeBoundaries");

			String samplingType = oldDistributionElement.getAttributeValue("samplingType");
			if (samplingType != null) {
				newDeviationElement.setAttribute(new Attribute("samplingType", samplingType));
			}

		} else if (oldDistributionType.equals("am:GaussDistribution")) {
			/*-
			<nextOccurrence>
				<lowerBound xsi:type="am:Time" value="0"/>
				<upperBound xsi:type="am:Time" value="0" unit="s"/>
				<distribution xsi:type="am:GaussDistribution">
					<sd xsi:type="am:Time" value="55" unit="us"/>
					<mean xsi:type="am:Time" value="0"/>
				</distribution>
			</nextOccurrence>
			*/

			setType(newDeviationElement, "am:TimeGaussDistribution");

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			String mean_unit = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "unit");
			String sd = HelperUtil.getValueFromChildElement(oldDistributionElement, "sd", "value");
			String sd_unit = HelperUtil.getValueFromChildElement(oldDistributionElement, "sd", "unit");

			if (mean != null) {
				Element meanElement = new Element("mean");
				meanElement.setAttribute("value", mean);
				if (mean_unit != null) {
					meanElement.setAttribute("unit", mean_unit);
				}
				newDeviationElement.addContent(meanElement);
			}

			if (sd != null) {
				Element sdElement = new Element("sd");
				sdElement.setAttribute("value", sd);
				if (sd_unit != null) {
					sdElement.setAttribute("unit", sd_unit);
				}
				newDeviationElement.addContent(sdElement);
			}

		} else if (oldDistributionType.equals("am:UniformDistribution")) {
			/*-
			<deviation>
				<lowerBound xsi:type="am:DoubleObject" value="0"/>
				<upperBound xsi:type="am:DoubleObject" value="0"/>
				<distribution xsi:type="am:UniformDistribution"/>
			 </deviation>
			 */

			setType(newDeviationElement, "am:TimeUniformDistribution");

		} else if (oldDistributionType.equals("am:WeibullEstimators")) {

			/*-
			<nextOccurrence>
				<lowerBound xsi:type="am:Time" value="0"/>
				<upperBound xsi:type="am:Time" value="0"/>
				<distribution xsi:type="am:WeibullEstimators" pRemainPromille="10.0">
					<mean xsi:type="am:Time" value="55" unit="us"/>
				</distribution>
			</nextOccurrence>
			*/

			setType(newDeviationElement, "am:TimeWeibullEstimatorsDistribution");

			String pRemainPromille = oldDistributionElement.getAttributeValue("pRemainPromille");

			if (pRemainPromille != null) {
				newDeviationElement.setAttribute(new Attribute("pRemainPromille", pRemainPromille));
			}

			String mean = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "value");
			String mean_unit = HelperUtil.getValueFromChildElement(oldDistributionElement, "mean", "unit");

			if (mean != null) {
				Element averageElement = new Element("average");
				averageElement.setAttribute(new Attribute("value", mean));
				if (mean_unit != null) {
					averageElement.setAttribute(new Attribute("unit", mean_unit));
				}

				newDeviationElement.addContent(averageElement);
			}

		} else if (oldDistributionType.equals("am:WeibullParameters")) {

			LOGGER.error(
					"Deviation with \"WeibullParameters\" as distribution can not be migrated as the semantics of \"WeibullParameters\" were not clearly described");

			return null;
		}

		return newDeviationElement;
	}

	public static String getValueAfterApplyingIPC(String value, double ipcValue) {
		if (ipcValue != 0) {
			try {
				Double result = Double.parseDouble(value) / ipcValue;
				return result.longValue() + "";
			} catch (Exception e) {
				LOGGER.error("error on IPC value conversion", e);
			}
		}
		return value;
	}

	private static void setType(Element deviation, String distributionType) {
		Attribute value_TypeAttribute = new Attribute("type", distributionType, AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
		deviation.getAttributes().add(value_TypeAttribute);
	}
}
