/**
 ********************************************************************************
 * Copyright (c) 2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */
package org.eclipse.app4mc.amalthea.converters.common;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.eclipse.app4mc.amalthea.converters.common.utils.ModelVersion;
import org.jdom2.Namespace;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MigrationHelper {

	private static final Logger LOGGER = LoggerFactory.getLogger(MigrationHelper.class);

	public static final String LINE_SEPARATOR = System.getProperty("line.separator");

	private MigrationHelper() {
		// empty default constructor
	}

	public static Map<File, MigrationInputFile> populateModels(
			List<File> inputModelFiles,
			MigrationSettings migrationSettings) throws Exception {

		HashMap<File, MigrationInputFile> modelFilesMap = new HashMap<>();

		for (File inputFile : inputModelFiles) {

			MigrationInputFile migModelFile = new MigrationInputFile();
			migModelFile.setFile(inputFile.getCanonicalFile(), migrationSettings.getProject());

			// identify if the model file that is selected by the user
			migModelFile.setSelectedFile(inputModelFiles.contains(inputFile));

			// set the model file version
			migModelFile.setModelVersion(getModelVersion(inputFile));

			modelFilesMap.put(inputFile, migModelFile);
		}

		return modelFilesMap;
	}

	/**
	 * Save the converted documents carried in the given {@link MigrationSettings}.
	 *
	 * @param settings The {@link MigrationSettings} that contains the references to
	 *                 the converted Documents to save.
	 * @throws IOException If an error occurred on saving the output file.
	 */
	public static void saveFiles(MigrationSettings settings) throws IOException {
		Set<File> keySet = settings.getMigModelFilesMap().keySet();

		boolean updateFileNames = false;
		if (settings.getInputModelVersion().equals("itea.103")
				|| settings.getInputModelVersion().equals("itea.110")) {

			List<String> allVersions = ModelVersion.getAllSupportedVersions();

			if (!settings.getMigrationModelVersion().equals("itea.103")
					&& !settings.getMigrationModelVersion().equals("itea.110")
					&& allVersions.contains(settings.getMigrationModelVersion())) {
				updateFileNames=true;
			}
		}

		for (File inputFile : keySet) {

			String outputDirectoryLocation = settings.getOutputDirectoryLocation();

			String convertedFileName = inputFile.getName();

			if (updateFileNames) {
				// add amxmi extension to file names which do not have it (and are migrated to versions 1.1.1 or higher)
				int indexOfDot = convertedFileName.lastIndexOf(".");
				String extension = convertedFileName.substring(indexOfDot + 1);
				if (extension.startsWith("amxmi") && !extension.equals("amxmi")) {
					convertedFileName = convertedFileName + ".amxmi";
				}
			}

			File outputFile = null;

			if (outputDirectoryLocation != null && !outputDirectoryLocation.equals("")) {
				String location = outputDirectoryLocation + File.separator + convertedFileName;
				HelperUtil.saveFile(settings.getMigModelFilesMap().get(inputFile), location, true, true);

				outputFile = new File(location);
			}
			else {
				String location = inputFile.getParentFile().getAbsolutePath() + File.separator + convertedFileName;
				HelperUtil.saveFile(settings.getMigModelFilesMap().get(inputFile), location, true, true);

				outputFile = new File(location);
			}

			LOGGER.info("Migrated model file saved @ : {}", outputFile.getAbsolutePath());
		}
	}

	/**
	 * This method is used to generate list of model migration steps which are required to migrate from one version to
	 * other. <br>
	 * <br>
	 * <b>Example</b>: Below are the steps which are required to migrate from itea.103 to 0.7.2:<br>
	 * <br>
	 *
	 * <table style="height: 65px;" width="120">
	 * <tbody>
	 * <tr>
	 * <td style="text-align:right;">itea.103 --&gt;</td>
	 * <td>itea.110</td>
	 * </tr>
	 * <tr>
	 * <td style="text-align:right;">itea.110--&gt;</td>
	 * <td>itea.111</td>
	 * </tr>
	 * <tr>
	 * <td style="text-align:right;">itea.111--&gt;</td>
	 * <td>0.7.0</td>
	 * </tr>
	 * <tr>
	 * <td style="text-align:center;">0.7.0--&gt;</td>
	 * <td>0.7.1</td>
	 * </tr>
	 * <tr>
	 * <td style="text-align:center;">0.7.1--&gt;</td>
	 * <td>0.7.2</td>
	 * </tr>
	 * </tbody>
	 * </table>
	 * <p>
	 * &nbsp;
	 * </p>
	 *
	 * @param inputModelVersion
	 *            This is the version present in input AMALTHEA model file (e.g: itea.103)
	 * @param outputModelVersion
	 *            AMALTHEA model file should be migrated to this version (e.g: 0.7.2)
	 * @return
	 */
	public static Map<String, String> generateMigrationSteps(String inputModelVersion, String outputModelVersion) {

		LinkedHashMap<String, String> migStepEntries = new LinkedHashMap<>();

		/*
		 * Note: These are the various AMALTHEA model versions which are released.
		 *        Order of the below list should be same as the release order
		 *        -> as based on this order, model migration steps are prepared
		 */
		List<String> versions = ModelVersion.getAllSupportedVersions();

		int inputModelVersionIndex = versions.indexOf(inputModelVersion);
		int outputModelVersionIndex = versions.indexOf(outputModelVersion);

		if (inputModelVersionIndex != -1 && outputModelVersionIndex != -1) {

			for (int i = inputModelVersionIndex;
					(i <= outputModelVersionIndex) && ((i + 1) <= outputModelVersionIndex);
					i++) {

				migStepEntries.put(versions.get(i), versions.get(i + 1));
			}
		}

		return migStepEntries;
	}

	/**
	 * Verifies the model version of the input files set to the provided
	 * {@link MigrationSettings} and sets the input model version if the
	 * verification succeeds.
	 *
	 * @param migrationSettings The MigrationSettings that contain the input model
	 *                          file references and is used for migration
	 *                          processing.
	 * @return <code>true</code> if the model versions of the input files are valid,
	 *         <code>false</code> if the model versions of the input files are not
	 *         equal.
	 * @throws MigrationException if input model files contain ITEA model versions
	 *                            or invalid model versions that are not defined in
	 *                            {@link ModelVersion}.
	 */
	public static boolean isInputModelVersionValid(MigrationSettings migrationSettings) {
		List<MigrationInputFile> modelsWithLegacyVersionInfo = getInputModelsOfLegacyVersions(migrationSettings);
		List<MigrationInputFile> modelsWithInvalidVersionInfo = getModelsWithInvalidVersionInfo(migrationSettings);

		if (modelsWithLegacyVersionInfo.size() > 0) {
			StringBuilder builder = new StringBuilder();

			builder.append("Can not migrate legacy Amalthea models of versions : itea.103 or itea.110 or itea.111");
			builder.append(LINE_SEPARATOR);
			builder.append(LINE_SEPARATOR);

			builder.append("From APP4MC 0.9.3, support for migrating legacy model versions (itea.103, itea.110, itea.111) is removed");
			builder.append(LINE_SEPARATOR);
			builder.append(LINE_SEPARATOR);

			builder.append("** Legacy model versions found for the below files ->");
			builder.append(LINE_SEPARATOR);
			builder.append(LINE_SEPARATOR);

			for (MigrationInputFile migrationFile : modelsWithInvalidVersionInfo) {
				builder
					.append(migrationFile.getFile().getAbsolutePath())
					.append(" Version: ")
					.append(migrationFile.getModelVersion());

				builder.append(LINE_SEPARATOR);
				builder
					.append(" -------- > Version: ")
					.append(migrationFile.getModelVersion());
			}
			builder.append(LINE_SEPARATOR);
			builder.append(LINE_SEPARATOR);

			builder.append(
					"Use one of the previous version of APP4MC (till 0.9.2) for migrating these legacy Amalthea models to regular versions (0.7.0 and above)");

			throw new MigrationException(builder.toString());

		} else if (modelsWithInvalidVersionInfo.size() > 0) {

			StringBuilder builder = new StringBuilder();
			builder.append("Invalid model versions found for the below files ->");

			builder.append(LINE_SEPARATOR);
			builder.append(LINE_SEPARATOR);

			for (MigrationInputFile migrationFile : modelsWithInvalidVersionInfo) {
				builder.append(migrationFile.getFile().getAbsolutePath());
				builder.append(LINE_SEPARATOR);
			}

			throw new MigrationException(builder.toString());

		} else {
			// try to identify and set the input model version
			boolean inputModelVersionEqual = true;

			String inputModelVersion = null;
			for (MigrationInputFile migModelFile : migrationSettings.getMigModelFiles()) {
				if (inputModelVersion == null) {
					inputModelVersion = migModelFile.getModelVersion();
				} else if (!inputModelVersion.equals(migModelFile.getModelVersion())) {
					migModelFile.setVersionDifferent(true);
					inputModelVersionEqual = false;
				}
			}

			if (inputModelVersionEqual) {
				migrationSettings.setInputModelVersion(inputModelVersion);
			}
			return inputModelVersionEqual;
		}
	}

	/**
	 * This method is used to get all the input model files which are belonging to
	 * ITEA versions of AMALTHEA (e.g: itea.103, itea.110, itea.111)
	 *
	 * @param migrationSettings The settings that contain the input model file
	 *                          references.
	 * @return List of all input model file references that belong to ITEA versions
	 *         of AMALTHEA.
	 */
	private static List<MigrationInputFile> getInputModelsOfLegacyVersions(MigrationSettings migrationSettings) {
		List<MigrationInputFile> list = new ArrayList<>();

		for (MigrationInputFile migModelFile : migrationSettings.getMigModelFiles()) {
			String inputModelVersion = migModelFile.getModelVersion();
			if (((inputModelVersion != null)
					&& (inputModelVersion.equals("itea.103")
						|| inputModelVersion.equals("itea.110")
						|| inputModelVersion.equals("itea.111")

			))) {
				list.add(migModelFile);
			}
		}

		return list;
	}

	/**
	 * This method is used to build list of model files for which version is
	 * invalid. Valid versions are described in {@link ModelVersion}.
	 *
	 * @param migrationSettings The settings that contain the input model file
	 *                          references.
	 * @return List of all input model files with invalid version information.
	 */
	private static List<MigrationInputFile> getModelsWithInvalidVersionInfo(MigrationSettings migrationSettings) {
		List<MigrationInputFile> list = new ArrayList<>();

		for (MigrationInputFile migModelFile : migrationSettings.getMigModelFiles()) {
			String inputModelVersion = migModelFile.getModelVersion();

			if (!ModelVersion.isValidVersion(inputModelVersion)) {
				list.add(migModelFile);
			}
		}

		return list;
	}

	/**
	 * Creates a backup file of the provided input file. The name of the backup file
	 * is defined in {@link #getBackupFileName(MigrationInputFile)}.
	 *
	 * @param migrationInputFile The input file for which a backup should be
	 *                           created.
	 * @return <code>true</code> if the backup file could be created and the backup
	 *         file exists, <code>false</code> if the file does not exist.
	 */
	public static boolean createBackupFile(MigrationInputFile migrationInputFile) {
		// Rename or copy the original file to filename_currentversion.amxmi
		String newFileName = MigrationHelper.getBackupFileName(migrationInputFile);
		try {
			String filebackupName = migrationInputFile.getFile().getParent() + File.separator + newFileName;
			Files.copy(migrationInputFile.getFile().toPath(), new File(filebackupName).toPath(), StandardCopyOption.REPLACE_EXISTING);
			LOGGER.info( "Original model file saved as {}", filebackupName);
			if (new File(filebackupName).exists()) {
				return true;
			}
		} catch (IOException e) {
			LOGGER.error(e.getMessage(), e);
		}
		return false;
	}

	/**
	 * Try to construct the file name for the source file backup which should be
	 * backed up before migration. Intension is not to migrate the source file
	 * without making a copy of it. Source file name is
	 * <i>&lt;origfilename_withoutextension&gt;_&lt;input_amlatheaversion&gt;.amxmi</i>
	 * <p>
	 * Examples:<br>
	 * model.axmi(orig) -> model_0.9.4.amxmi
	 * </p>
	 * <p>
	 * If file name exists then the model version is appended before the extension e.g.,
	 * model_0.9.4_0.9.4.amxi
	 * </p>
	 *
	 * @param migrationInputFile The input file for which the backup file name is
	 *                           requested.
	 * @return filename of the input file backup.
	 */
	public static String getBackupFileName(MigrationInputFile migrationInputFile) {
		String fileName = migrationInputFile.getFile().getName();
		String newFileName = fileName.substring(0, fileName.lastIndexOf("."));
		String suffixString = "_" + migrationInputFile.getModelVersion();
		String fileExtension = ".amxmi";

		newFileName = newFileName + suffixString + fileExtension;
		// check if file with newly constructed name already exists then append model
		// version at end of name
		if (new File(migrationInputFile.getFile().getParent() + File.separator + newFileName).exists()) {
			newFileName = newFileName.replace(suffixString + fileExtension, suffixString + suffixString + fileExtension);
		}

		return newFileName;
	}

	/**
	 * Extract the model version of the given input model file.
	 * 
	 * @param inputModelFile The model file for which the model version is
	 *                       requested.
	 * @return The model version of the given input model file or <i>invalid</i> if
	 *         an error occurs on parsing or the model version is invalid.
	 */
	public static String getModelVersion(File inputModelFile) {
		String result = "";
		
		if (inputModelFile != null && inputModelFile.exists() && inputModelFile.getName().endsWith(".amxmi")) {
			try (FileInputStream input = new FileInputStream(inputModelFile)) {
				XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance();
				XMLEventReader reader = xmlInputFactory.createXMLEventReader(input);
				while (reader.hasNext()) {
					XMLEvent nextEvent = reader.nextEvent();
					if (nextEvent.isStartElement()) {
					    StartElement startElement = nextEvent.asStartElement();
					    if ("am".equals(startElement.getName().getPrefix())) {
					    	
					    	String url = startElement.getNamespaceURI("am");
					    	Namespace namespace = Namespace.getNamespace("am", url);
					    	ModelVersion version = AmaltheaNamespaceRegistry.getModelVersion(namespace);
							if (version != null) {
								result = version.getVersion();
							} else {
								result = "invalid";
							}
							// we found the root tag and parsed the model version, now we can stop parsing
							break;
					    }
					}
				}
			} catch (IOException | XMLStreamException e) {
				LOGGER.error("Error on parsing input model file for model version", e);
				result = "invalid";
			}
		}
		
		return result;
	}
}
