/**
 ********************************************************************************
 * Copyright (c) 2020 Robert Bosch GmbH.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 *******************************************************************************/
package org.eclipse.app4mc.amalthea.visualization.hw;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.Charset;

import javax.inject.Named;

import org.eclipse.app4mc.amalthea.model.HWModel;
import org.eclipse.app4mc.amalthea.visualization.hw.templates.HWBlockDiagramCreator;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.Preference;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sourceforge.plantuml.FileFormat;
import net.sourceforge.plantuml.FileFormatOption;
import net.sourceforge.plantuml.SourceStringReader;
import net.sourceforge.plantuml.core.DiagramDescription;
import net.sourceforge.plantuml.eclipse.utils.PlantumlConstants;

@SuppressWarnings("restriction")
public class VisualizationHandler {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(VisualizationHandler.class);
	
	@Execute
	public void execute(
			Shell shell,
			UISynchronize sync,
			@Optional
			@Preference(
					nodePath = "net.sourceforge.plantuml.eclipse", 
					value = PlantumlConstants.GRAPHVIZ_PATH) 
			String dotPath,
			@Named(IServiceConstants.ACTIVE_SELECTION) IStructuredSelection selection) {
		
		// **** Check selection and set HW model
		
		if (!selection.isEmpty() && !(selection.getFirstElement() instanceof HWModel)) {
			LOGGER.info("No HWModel selected!");
			return;
		}

		HWModel hwModel = ((HWModel) selection.getFirstElement());
		DiagramResource diagramResource = getDiagramResource(hwModel);

		execute(shell, sync, dotPath, hwModel, diagramResource, true, null);
	}

	/**
	 * 
	 * @param shell The active Shell needed for info/error dialog creation.
	 * @param dotPath The path to Graphviz dot.exe needed for visualization.
	 * @param hwModel The {@link HWModel} that should be visualized.
	 * @param listener optional listener that is notified on diagram generation job changes.
	 * @return The path to the SVG.
	 */
	public void execute(
			Shell shell,
			UISynchronize sync,
			String dotPath,
			HWModel hwModel,
			DiagramResource diagramResource,
			boolean showSuccessInfo,
			IJobChangeListener listener) {

		// **** Check dot path and set GRAPHVIZ_DOT property
		
		if (dotPath == null || dotPath.equals("")) {
			showErrorDialog(shell, sync, "Missing Graphviz dot.exe location."
						+ "\nPlease specify location via Window - Preferences - PlantUML - Path to the dot executable of Graphviz.");
			return;
		}
		
		final File dotFile = new File(dotPath);
		if (!dotFile.canExecute()) {
			showErrorDialog(shell, sync, "Invalid Graphviz dot.exe location: " + dotFile.getAbsolutePath()
						+ "\nPlease set location via Window - Preferences - PlantUML - Path to the dot executable of Graphviz.");
			return;
		}
		
		System.setProperty("GRAPHVIZ_DOT", dotFile.getAbsolutePath());

		LOGGER.info("GRAPHVIZ_DOT: " + System.getProperty("GRAPHVIZ_DOT"));
			
		
		// **** Generate Plant-UML file
		Job job = Job.create("Generate diagram", monitor -> {
			
			ModelToTextResult umlOutput = HWBlockDiagramCreator.generatePlantUML(hwModel);
			if (umlOutput.error()) {
				showErrorDialog(shell, sync, "Errors on generating HW Visualization for selected AMALTHEA model file: " + umlOutput.getErrorMessage());
				return Status.CANCEL_STATUS;
			}
			
			// **** Write to UML to file

			try (BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(diagramResource.plantUmlFilePath))) {
				bufferedWriter.write(umlOutput.getOutput().toString());
			} catch (IOException e) {
				return Status.CANCEL_STATUS;
			}
			
			LOGGER.info("diagram.plantuml created: {}", umlOutput.getOutput());
			
			// **** Render SVG output
			
			try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
					FileOutputStream fileStream = new FileOutputStream(diagramResource.diagramFilePath)) {
				
				SourceStringReader reader = new SourceStringReader(umlOutput.getOutput().toString());
				DiagramDescription description = reader.outputImage(outputStream, new FileFormatOption(FileFormat.SVG));
				
				byte[] svgBytes = outputStream.toByteArray();
				String svgString = new String(svgBytes, Charset.forName("UTF-8"));
				
				fileStream.write(svgBytes);
				fileStream.flush();
				
				if (description == null || svgString.contains("An error has occured")) {
					showErrorDialog(shell, sync, "SVG rendering failed."
							+ "\nCheck Graphviz dot.exe location via Window - Preferences - PlantUML - Path to the dot executable of Graphviz..");
					return Status.CANCEL_STATUS;
				}
				
				LOGGER.info("diagram.svg created");
				
				if (showSuccessInfo) {
					showInfoDialog(shell, sync, "Successfully generated HW Visualization for amalthea model file.");
				}
			} catch (Exception e) {
				showErrorDialog(shell, sync, "Unable to generate HW Visualization for selected AMALTHEA model file.");
				LOGGER.error("Unable to generate HW Visualization for selected AMALTHEA model file.", e);
			} finally {
				try {
					diagramResource.diagramFolder.refreshLocal(1, new NullProgressMonitor());
				} catch (CoreException ex) {
					// no action required
				}
			}
			
			return Status.OK_STATUS;

		});
		
		if (listener != null) {
			job.addJobChangeListener(listener);
		}
		
		job.schedule();
	}

	
	public DiagramResource getDiagramResource(HWModel hwModel) {
		
		// **** Determine parent folder

		Resource eResource = hwModel.eResource();
		if (eResource == null) {
			return null;
		}
		
		URI uri = eResource.getURI();
		IPath path = new Path(uri.toPlatformString(true));
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		
		IContainer parentFolder = file.getParent();
		if (parentFolder == null) {
			return null;
		}

		// **** Find visualization folder (create if necessary)

		IFolder folder = parentFolder.getFolder(new Path("hwModelVisualization"));
		
		if (!folder.exists()) {
			try {
				folder.create(IResource.NONE, true, null);
			} catch (CoreException e) {
				return null;
			}
		}
		
		String absPath = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString()
				+ folder.getFullPath().toString();
		String diagramFileName = "Diagram.svg";
		String plantumlFileName = "Diagram_Description.plantuml";
		
		String amaltheaModelName = file.getName();
		if (amaltheaModelName != null) {
			amaltheaModelName = amaltheaModelName.replaceFirst("[.][^.]+$", "");
			diagramFileName = amaltheaModelName+".svg";
			plantumlFileName = amaltheaModelName+".plantuml";
		}
		
		String plantumlFilePath = absPath + "/" + plantumlFileName;
		String diagramFilePath = absPath + "/" + diagramFileName;
		
		LOGGER.info("Generate plantUML file {} and SVG file {}", plantumlFilePath, diagramFilePath);

		return new DiagramResource(folder, plantumlFilePath, diagramFilePath);
	}
	
	private void showInfoDialog(Shell shell, UISynchronize sync, String message) {
		sync.asyncExec(() -> {
			MessageDialog.openInformation(shell,
					"AMALTHEA HW Visualization",
					message);
		});
	}
	
	private void showErrorDialog(Shell shell, UISynchronize sync, String message) {
		sync.asyncExec(() -> {
			MessageDialog.openError(shell,
					"AMALTHEA HW Visualization",
					message);
		});
	}

	public static class DiagramResource {
		
		public final IFolder diagramFolder;
		public final String plantUmlFilePath;
		public final String diagramFilePath;
		
		public DiagramResource(IFolder diagramFolder, String plantUmlFilePath, String diagramFilePath) {
			this.diagramFolder = diagramFolder;
			this.plantUmlFilePath = plantUmlFilePath;
			this.diagramFilePath = diagramFilePath;
		}
	}
}
