/**
 ********************************************************************************
 * Copyright (c) 2019, 2020 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.validation.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.inject.Inject;

import org.eclipse.app4mc.validation.core.IProfile;
import org.eclipse.app4mc.validation.util.CachedProfile;
import org.eclipse.app4mc.validation.util.ProfileManager;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.Service;
import org.eclipse.jface.dialogs.IDialogSettings;

@SuppressWarnings("restriction")
public class ProfileDialogSettings {

	private final String KEY_SELECTED_PROFILE_CLASSES = "Validation_SelectedProfileClasses";
	
	// dialog input
	private Set<String> selectedProfileClassNames = new HashSet<>();

	// dialog result
	boolean dialogSuccess = false;
	public List<Class<? extends IProfile>>dialogResults = null;
	
	// dialog setup
	
	private ProfileManager profileManager;
	
	@Inject
	@Optional
	public void setProfileManager(@Service ProfileManager manager) {
		this.profileManager = manager;
	}

	// methods required by tree view

	public Object[] getProfiles() {
		return this.profileManager.getRegisteredValidationProfiles().values().stream()
				.sorted((p1, p2) -> p1.getName().compareTo(p2.getName()))
				.toArray();
	}

	public CachedProfile[] getSelectedProfiles() {
		List<CachedProfile> allNodes = new ArrayList<>();
		
		for (CachedProfile rootNode : this.profileManager.getRegisteredValidationProfiles().values()) {
			allNodes.add(rootNode);
			collectSubNodes(rootNode, allNodes);
		}
		
		return allNodes.stream()
				.filter( cache -> this.selectedProfileClassNames.contains(cache.getProfileClass().getName()) )
				.toArray(CachedProfile[]::new);
	}
	
	private void collectSubNodes(CachedProfile profile, List<CachedProfile> list) {
		if (!profile.getCachedProfiles().isEmpty()) {
			for (CachedProfile subProfile : profile.getCachedProfiles().values()) {
				list.add(subProfile);
				collectSubNodes(subProfile, list);
			}
		}
	}

	public void setSelectedProfileClassNames(List<String> classNames) {
		this.selectedProfileClassNames.clear();
		this.selectedProfileClassNames.addAll(classNames);
	}

	// dialog settings store
	
	public void loadFrom(IDialogSettings store) {
		if (store == null) return;
		
		// KEY_SELECTED_PROFILE_CLASSES
		String[] classNames = store.getArray(KEY_SELECTED_PROFILE_CLASSES);
		if (classNames != null) {
			setSelectedProfileClassNames(Arrays.asList(classNames));			
		}
	}

	public void saveTo(IDialogSettings store) {
		if (store == null) return;
		if (!dialogSuccess || dialogResults == null) return;
		
		// KEY_SELECTED_PROFILE_CLASSES
		String[] classNames = dialogResults.stream().map(cls -> cls.getName()).toArray(String[]::new);
		store.put(KEY_SELECTED_PROFILE_CLASSES, classNames);
	}

}
