/**
 ********************************************************************************
 * Copyright (c) 2019 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */
package org.eclipse.app4mc.amalthea.converters.common.utils;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jdom2.Namespace;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AmaltheaNamespaceRegistry {

	private static final Logger LOGGER = LoggerFactory.getLogger(AmaltheaNamespaceRegistry.class);

	private static final LinkedHashMap<String, AmaltheaNamespace> REGISTRY = new LinkedHashMap<>();
	private static final String GENERIC_NAMESPACE = "generic";

	static {
		AmaltheaNamespace generic = new AmaltheaNamespace();
		generic.registerMapping("xsi", "http://www.w3.org/2001/XMLSchema-instance");
		generic.registerMapping("xmi", "http://www.omg.org/XMI");
		REGISTRY.put(GENERIC_NAMESPACE, generic);

		AmaltheaNamespace ns103 = new AmaltheaNamespace();
		ns103.registerMapping("central", "http://www.amalthea.itea2.org/model/1.1.0/central");
		ns103.registerMapping("common", "http://www.amalthea.itea2.org/model/1.1.0/common");
		ns103.registerMapping("hw", "http://www.amalthea.itea2.org/model/1.1.0/hw");
		ns103.registerMapping("os", "http://www.amalthea.itea2.org/model/1.1.0/os");
		ns103.registerMapping("stimuli", "http://www.amalthea.itea2.org/model/1.1.0/stimuli");
		ns103.registerMapping("sw", "http://www.amalthea.itea2.org/model/1.1.0/sw");
		ns103.registerMapping("propertyconstraints", "http://www.amalthea.itea2.org/model/1.1.0/propertyconstraints");
		ns103.registerMapping("mapping", "http://www.amalthea.itea2.org/model/1.1.0/mapping");
		ns103.registerMapping("events", "http://www.amalthea.itea2.org/model/1.1.0/events");
		ns103.registerMapping("constraints", "http://www.amalthea.itea2.org/model/1.1.0/constraints");
		ns103.registerMapping("config", "http://www.amalthea.itea2.org/model/1.1.0/config");
		ns103.registerMapping("components", "http://amalthea.itea2.org/model/1.1.0/components");
		REGISTRY.put(ModelVersion._103.getVersion(), ns103);

		AmaltheaNamespace ns110 = new AmaltheaNamespace();
		ns110.registerMapping("central", "http://www.amalthea.itea2.org/model/1.2.0/central");
		ns110.registerMapping("common", "http://www.amalthea.itea2.org/model/1.2.0/common");
		ns110.registerMapping("hw", "http://www.amalthea.itea2.org/model/1.2.0/hw");
		ns110.registerMapping("os", "http://www.amalthea.itea2.org/model/1.2.0/os");
		ns110.registerMapping("stimuli", "http://www.amalthea.itea2.org/model/1.2.0/stimuli");
		ns110.registerMapping("sw", "http://www.amalthea.itea2.org/model/1.2.0/sw");
		ns110.registerMapping("propertyconstraints", "http://www.amalthea.itea2.org/model/1.2.0/propertyconstraints");
		ns110.registerMapping("mapping", "http://www.amalthea.itea2.org/model/1.2.0/mapping");
		ns110.registerMapping("events", "http://www.amalthea.itea2.org/model/1.2.0/events");
		ns110.registerMapping("constraints", "http://www.amalthea.itea2.org/model/1.2.0/constraints");
		ns110.registerMapping("config", "http://www.amalthea.itea2.org/model/1.2.0/config");
		ns110.registerMapping("components", "http://amalthea.itea2.org/model/1.2.0/components");
		REGISTRY.put(ModelVersion._110.getVersion(), ns110);

		AmaltheaNamespace ns111 = new AmaltheaNamespace();
		ns111.registerMapping("central", "http://www.amalthea.itea2.org/model/1.3.0/central");
		ns111.registerMapping("common","http://www.amalthea.itea2.org/model/1.3.0/common");
		ns111.registerMapping("hw", "http://www.amalthea.itea2.org/model/1.3.0/hw");
		ns111.registerMapping("os", "http://www.amalthea.itea2.org/model/1.3.0/os");
		ns111.registerMapping("stimuli", "http://www.amalthea.itea2.org/model/1.3.0/stimuli");
		ns111.registerMapping("sw", "http://www.amalthea.itea2.org/model/1.3.0/sw");
		ns111.registerMapping("propertyconstraints", "http://www.amalthea.itea2.org/model/1.3.0/propertyconstraints");
		ns111.registerMapping("mapping", "http://www.amalthea.itea2.org/model/1.3.0/mapping");
		ns111.registerMapping("events", "http://www.amalthea.itea2.org/model/1.3.0/events");
		ns111.registerMapping("constraints", "http://www.amalthea.itea2.org/model/1.3.0/constraints");
		ns111.registerMapping("config", "http://www.amalthea.itea2.org/model/1.3.0/config");
		ns111.registerMapping("components", "http://amalthea.itea2.org/model/1.3.0/components");
		REGISTRY.put(ModelVersion._111.getVersion(), ns111);

		AmaltheaNamespace ns070 = new AmaltheaNamespace();
		ns070.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.7.0");
		REGISTRY.put(ModelVersion._070.getVersion(), ns070);

		AmaltheaNamespace ns071 = new AmaltheaNamespace();
		ns071.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.7.1");
		REGISTRY.put(ModelVersion._071.getVersion(), ns071);

		AmaltheaNamespace ns072 = new AmaltheaNamespace();
		ns072.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.7.2");
		REGISTRY.put(ModelVersion._072.getVersion(), ns072);

		AmaltheaNamespace ns080 = new AmaltheaNamespace();
		ns080.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.8.0");
		REGISTRY.put(ModelVersion._080.getVersion(), ns080);

		AmaltheaNamespace ns081 = new AmaltheaNamespace();
		ns081.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.8.1");
		REGISTRY.put(ModelVersion._081.getVersion(), ns081);

		AmaltheaNamespace ns082 = new AmaltheaNamespace();
		ns082.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.8.2");
		REGISTRY.put(ModelVersion._082.getVersion(), ns082);

		AmaltheaNamespace ns083 = new AmaltheaNamespace();
		ns083.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.8.3");
		REGISTRY.put(ModelVersion._083.getVersion(), ns083);

		AmaltheaNamespace ns090 = new AmaltheaNamespace();
		ns090.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.0");
		REGISTRY.put(ModelVersion._090.getVersion(), ns090);

		AmaltheaNamespace ns091 = new AmaltheaNamespace();
		ns091.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.1");
		REGISTRY.put(ModelVersion._091.getVersion(), ns091);

		AmaltheaNamespace ns092 = new AmaltheaNamespace();
		ns092.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.2");
		REGISTRY.put(ModelVersion._092.getVersion(), ns092);

		AmaltheaNamespace ns093 = new AmaltheaNamespace();
		ns093.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.3");
		REGISTRY.put(ModelVersion._093.getVersion(), ns093);

		AmaltheaNamespace ns094 = new AmaltheaNamespace();
		ns094.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.4");
		REGISTRY.put(ModelVersion._094.getVersion(), ns094);

		AmaltheaNamespace ns095 = new AmaltheaNamespace();
		ns095.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.5");
		REGISTRY.put(ModelVersion._095.getVersion(), ns095);

		AmaltheaNamespace ns096 = new AmaltheaNamespace();
		ns096.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.6");
		REGISTRY.put(ModelVersion._096.getVersion(), ns096);

		AmaltheaNamespace ns097 = new AmaltheaNamespace();
		ns097.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.7");
		REGISTRY.put(ModelVersion._097.getVersion(), ns097);
		
		AmaltheaNamespace ns098 = new AmaltheaNamespace();
		ns098.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.8");
		REGISTRY.put(ModelVersion._098.getVersion(), ns098);

		AmaltheaNamespace ns099 = new AmaltheaNamespace();
		ns099.registerMapping("am", "http://app4mc.eclipse.org/amalthea/0.9.9");
		REGISTRY.put(ModelVersion._099.getVersion(), ns099);
	}

	/**
	 *
	 * @param namespace The namespace for which the ModelVersion is searched.
	 * @return The {@link ModelVersion} for the given {@link Namespace}.
	 */
	public static ModelVersion getModelVersion(Namespace namespace) {
		ModelVersion result = null;

		if (isNamespaceAvailable(ModelVersion._103, namespace)) {
			result = ModelVersion._103;
		}
		else if (isNamespaceAvailable(ModelVersion._110, namespace)) {
			result = ModelVersion._110;
		}
		else if (isNamespaceAvailable(ModelVersion._111, namespace)) {
			result = ModelVersion._111;
		}
		else if (isNamespaceAvailable(ModelVersion._070, namespace)) {
			result = ModelVersion._070;
		}
		else if (isNamespaceAvailable(ModelVersion._071, namespace)) {
			result = ModelVersion._071;
		}
		else if (isNamespaceAvailable(ModelVersion._072, namespace)) {
			result = ModelVersion._072;
		}
		else if (isNamespaceAvailable(ModelVersion._080, namespace)) {
			result = ModelVersion._080;
		}
		else if (isNamespaceAvailable(ModelVersion._081, namespace)) {
			result = ModelVersion._081;
		}
		else if (isNamespaceAvailable(ModelVersion._082, namespace)) {
			result = ModelVersion._082;
		}
		else if (isNamespaceAvailable(ModelVersion._083, namespace)) {
			result = ModelVersion._083;
		}
		else if (isNamespaceAvailable(ModelVersion._090, namespace)) {
			result = ModelVersion._090;
		}
		else if (isNamespaceAvailable(ModelVersion._091, namespace)) {
			result = ModelVersion._091;
		}
		else if (isNamespaceAvailable(ModelVersion._092, namespace)) {
			result = ModelVersion._092;
		}
		else if (isNamespaceAvailable(ModelVersion._093, namespace)) {
			result = ModelVersion._093;
		}
		else if (isNamespaceAvailable(ModelVersion._094, namespace)) {
			result = ModelVersion._094;
		}
		else if (isNamespaceAvailable(ModelVersion._095, namespace)) {
			result = ModelVersion._095;
		}
		else if (isNamespaceAvailable(ModelVersion._096, namespace)) {
			result = ModelVersion._096;
		}
		else if (isNamespaceAvailable(ModelVersion._097, namespace)) {
			result = ModelVersion._097;
		}
		else if (isNamespaceAvailable(ModelVersion._098, namespace)) {
			result = ModelVersion._098;
		}
		else if (isNamespaceAvailable(ModelVersion._099, namespace)) {
			result = ModelVersion._099;
		}
		return result;
	}

	/**
	 *
	 * @param version The ModelVersion for the requested Namespace.
	 * @param prefix  The namespace prefix.
	 * @return The namespace for the given version and prefix.
	 */
	public static Namespace getNamespace(ModelVersion version, String prefix) {
		return getNamespace(version.getVersion(), prefix);
	}

	/**
	 *
	 * @param version The String representation of the ModelVersion for which the
	 *                Namespace is requested.
	 * @param prefix  The namespace prefix.
	 * @return The namespace for the given version and prefix.
	 */
	public static Namespace getNamespace(String version, String prefix) {
		AmaltheaNamespace ns = REGISTRY.get(version);
		if (ns != null) {
			return ns.namespaceFor(prefix);
		} else {
			LOGGER.warn("No namespace registered for version {}", version);
		}
		return Namespace.getNamespace("", "");
	}

	public static Namespace[] getAllNamespacesFor(ModelVersion version) {
		AmaltheaNamespace ns = REGISTRY.get(version.getVersion());
		if (ns != null) {
			return ns.getAllNamespaces();
		} else {
			LOGGER.warn("No namespace registered for version {}", version.getVersion());
		}
		return new Namespace[] { Namespace.getNamespace("", "") };
	}

	public static Namespace[] getAllNamespacesBefore(ModelVersion version, boolean include, boolean includeGeneric) {
		ModelVersion[] versionsBefore = ModelVersion.getVersionsBefore(version, include);
		List<Namespace> collect = Arrays.stream(versionsBefore)
			.map(AmaltheaNamespaceRegistry::getAllNamespacesFor)
			.flatMap(Arrays::stream)
			.collect(Collectors.toList());

		if (includeGeneric) {
			AmaltheaNamespace generic = REGISTRY.get(GENERIC_NAMESPACE);
			if (generic != null) {
				collect.addAll(0, Arrays.asList(generic.getAllNamespaces()));
			}
		}

		if (!collect.isEmpty()) {
			return collect.toArray(new Namespace[0]);
		}

		return new Namespace[] { Namespace.getNamespace("", "") };
	}

	public static Namespace[] getAllNamespaces() {
		List<Namespace> collect = REGISTRY.entrySet().stream()
			.map(Map.Entry::getValue)
			.map(AmaltheaNamespace::getAllNamespaces)
			.flatMap(Arrays::stream)
			.collect(Collectors.toList());

		if (!collect.isEmpty()) {
			return collect.toArray(new Namespace[0]);
		}
		return new Namespace[] { Namespace.getNamespace("", "") };
	}

	public static boolean isNamespaceAvailable(ModelVersion version, Namespace namespace) {
		if (namespace != null) {
			String prefix = namespace.getPrefix();
			String uri = namespace.getURI();

			AmaltheaNamespace ns = REGISTRY.get(version.getVersion());
			if (ns != null && uri.equals(ns.valueOf(prefix))) {
				return true;
			}
		}

		return false;
	}

	public static boolean isPrefixForVersion(ModelVersion version, String prefix) {
		AmaltheaNamespace ns = REGISTRY.get(version.getVersion());
		return (ns != null && ns.hasPrefix(prefix));
	}

	public static Namespace getGenericNamespace(String prefix) {
		return getNamespace(GENERIC_NAMESPACE, prefix);
	}

	private AmaltheaNamespaceRegistry() {
		// empty constructor for helper class
	}
}
