/**
 ********************************************************************************
 * Copyright (c) 2013-2020 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.sphinx.ui.editors.actions;

import java.util.regex.Pattern;

import org.eclipse.app4mc.amalthea.sphinx.ui.editors.SphinxSupportPlugin;
import org.eclipse.app4mc.amalthea.sphinx.ui.editors.editor.ExtendedBasicTransactionalFormEditor;
import org.eclipse.app4mc.amalthea.sphinx.ui.editors.messages.Messages;
import org.eclipse.app4mc.amalthea.sphinx.ui.editors.search.ModelHitCollector;
import org.eclipse.app4mc.amalthea.sphinx.ui.editors.search.handlers.SearchDialog;
import org.eclipse.app4mc.amalthea.sphinx.ui.editors.search.handlers.SearchDialogSettings;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ResourceLocator;
import org.eclipse.jface.window.Window;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

public class SearchAction extends Action {

	private final ExtendedBasicTransactionalFormEditor editor;

	public SearchAction(ExtendedBasicTransactionalFormEditor editor) {
		super(Messages.SearchAction_title, AS_PUSH_BUTTON);
		this.editor = editor;
		// image taken from
		// https://github.com/eclipse/mylyn.commons/blob/d95886e9e377efa663da5901661d20e5e0da9a49/org.eclipse.mylyn.commons.ui/icons/etool16/find.gif
		setImageDescriptor(ResourceLocator.imageDescriptorFromBundle(SphinxSupportPlugin.getPluginId(),
				"/icons/full/obj16/find.gif").orElse(null)); //$NON-NLS-1$
		setToolTipText(Messages.SearchGUI_title);
	}

	/**
	 * @see org.eclipse.jface.action.Action#run()
	 */
	@Override
	public void run() {
		final IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		final SearchDialogSettings settings = new SearchDialogSettings();
		
		// retrieve stored settings
		final IDialogSettings store = SphinxSupportPlugin.getDefault().getDialogSettings();
		settings.loadFrom(store);
		
		final SearchDialog dialog = new SearchDialog(window.getShell(), settings);
		int returnCode = dialog.open();
		
		if (returnCode == Window.OK) {
			final Pattern searchPattern = settings.computeSearchPattern();
			if (searchPattern == null) return;
			
			// store settings
			settings.saveTo(store);
			
			// try to get current open model
			final Object input = this.editor.getEditorInputObject();
			if (input == null) return;
			
			EObject model = null;
			if (input instanceof EObject) {
				model = (EObject) input;
			}
			else if (input instanceof Resource) {
				final Resource resource = (Resource) input;
				if (!resource.getContents().isEmpty()) {
					model = resource.getContents().get(0);
				}
			}
			if (model == null) return;
			
			final ModelHitCollector query = new ModelHitCollector (
					searchPattern,
					settings.computeFilterClass(),
					settings.isSearchRestrictedToFile(),
					model,
					this.editor.getEditorInput());
			
			NewSearchUI.runQueryInBackground(query);
		}
	}
}
