package prefuse.util.force;

import java.awt.geom.Line2D;

/**
 * Uses a gravitational force model to act as a "wall". Can be used to
 * construct line segments which either attract or repel items.
 *
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class WallForce extends AbstractForce {

    private static String[] pnames = new String[] { "GravitationalConstant" };
    
    public static final float DEFAULT_GRAV_CONSTANT = -0.1f;
    public static final float DEFAULT_MIN_GRAV_CONSTANT = -1.0f;
    public static final float DEFAULT_MAX_GRAV_CONSTANT = 1.0f;
    public static final int GRAVITATIONAL_CONST = 0;
    
    private float x1, y1, x2, y2;
    private float dx, dy;
    
    /**
     * Create a new WallForce.
     * @param gravConst the gravitational constant of the wall
     * @param x1 the first x-coordinate of the wall
     * @param y1 the first y-coordinate of the wall
     * @param x2 the second x-coordinate of the wall
     * @param y2 the second y-coordinate of the wall
     */
    public WallForce(float gravConst, 
        float x1, float y1, float x2, float y2) 
    {
        this.params = new float[] { gravConst };
        this.minValues = new float[] { DEFAULT_MIN_GRAV_CONSTANT };
        this.maxValues = new float[] { DEFAULT_MAX_GRAV_CONSTANT };
        
        this.x1 = x1; this.y1 = y1;
        this.x2 = x2; this.y2 = y2;
        this.dx = x2-x1;
        this.dy = y2-y1;
        float r = (float)Math.sqrt(this.dx*this.dx+this.dy*this.dy);
        if ( this.dx != 0.0 ) this.dx /= r;
        if ( this.dy != 0.0 ) this.dy /= r;
    }
    
    /**
     * Create a new WallForce with default gravitational constant.
     * @param x1 the first x-coordinate of the wall
     * @param y1 the first y-coordinate of the wall
     * @param x2 the second x-coordinate of the wall
     * @param y2 the second y-coordinate of the wall
     */
    public WallForce(float x1, float y1, float x2, float y2) {
        this(DEFAULT_GRAV_CONSTANT,x1,y1,x2,y2);
    }
    
    /**
     * Returns true.
     * @see prefuse.util.force.Force#isItemForce()
     */
    public boolean isItemForce() {
        return true;
    }
    
    /**
     * @see prefuse.util.force.AbstractForce#getParameterNames()
     */
    protected String[] getParameterNames() {
        return pnames;
    }
    
    /**
     * @see prefuse.util.force.Force#getForce(prefuse.util.force.ForceItem)
     */
    public void getForce(ForceItem item) {
        float[] n = item.location;
        int ccw = Line2D.relativeCCW(this.x1,this.y1,this.x2,this.y2,n[0],n[1]);
        float r = (float)Line2D.ptSegDist(this.x1,this.y1,this.x2,this.y2,n[0],n[1]);
        if ( r == 0.0 ) r = (float)Math.random() / 100.0f;
        float v = this.params[GRAVITATIONAL_CONST]*item.mass / (r*r*r);
        if ( n[0] >= Math.min(this.x1,this.x2) && n[0] <= Math.max(this.x1,this.x2) )
            item.force[1] += ccw*v*this.dx;
        if ( n[1] >= Math.min(this.y1,this.y2) && n[1] <= Math.max(this.y1,this.y2) )
            item.force[0] += -1*ccw*v*this.dy;
    }

} // end of class WallForce
