/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RBox.java,v 1.1 2007/03/18 09:07:03 lzhang Exp $
 */

package org.eclipse.eodm.owl.reasoner.structural;

import java.util.HashMap;
import java.util.List;
import java.util.Vector;

public class RBox {
    
    public RBox() {
        this(DEFAULT_INITIAL_CAPACITY);
    }
    
    public RBox(int initCapacity) {
        size = initCapacity;
        subPropertyMatrix = new boolean[size][size];
        properties = new String[size];
        topPropertyID = getNextID();
        bottomPropertyID = getNextID();
        subPropertyMatrix[topPropertyID][topPropertyID] = true;
        subPropertyMatrix[bottomPropertyID][topPropertyID] = true;
        subPropertyMatrix[bottomPropertyID][bottomPropertyID] = true;
    }
       
    public static final int DEFAULT_INITIAL_CAPACITY = 10;

    public int getTopPropertyID() {
        return topPropertyID;
    }

    public int getBottomPropertyID() {
        return bottomPropertyID;
    }

    public synchronized int getCreatePropertyID(Role p) {
        String name = p.getName();
        Integer id = (Integer) propertyMap.get(name);
        if (id == null) {
            id = new Integer(getNextID());
            subPropertyMatrix[id.intValue()][id.intValue()] = true;
            subPropertyMatrix[id.intValue()][topPropertyID] = true;
            subPropertyMatrix[bottomPropertyID][id.intValue()] = true;
            propertyMap.put(name, id);
            properties[id.intValue()] = name;
            return id.intValue();
        } else {
            return id.intValue();
        }
    }

    public synchronized int getPropertyID(Role p) {
        String name = p.getName();
        Integer id = (Integer) propertyMap.get(name);
        if (id == null) {
            return -1;
        } else {
            return id.intValue();
        }
    }

    /**
     * Set p1 as a sub property of p2
     * @param child
     * @param parent
     */
    public synchronized void addSubPropertyOfAxiom(int child, int parent) {
        if (child < 0 || parent < 0 || child >= nextPropertyID || parent >= nextPropertyID) {
            throw new ArrayIndexOutOfBoundsException();
        }
        if (subPropertyMatrix[child][parent]) {
            return;
        }
        for (int i = 0; i < nextPropertyID; i++) {
            if (subPropertyMatrix[i][child]) {
                for (int j = 0; j < nextPropertyID; j++) {
                    if (subPropertyMatrix[parent][j]) {
                        subPropertyMatrix[i][j] = true;
                    }
                }
            }
        }
    }

    /**
     * Set child as a sub property of parent
     * @param child
     * @param parent
     */
    public synchronized void addSubPropertyOfAxiom(Role child, Role parent) {
        addSubPropertyOfAxiom(getCreatePropertyID(child), getCreatePropertyID(parent));
    }

    /**
     * Test Whether p1 is subPropertyOf p2
     * 
     * @param p1
     * @param p2
     * @return true if p1 is subPropertyOf p2
     */
    public synchronized boolean isSubPropertyOf(int p1, int p2) {
        if (p1 < 0 || p2 < 0 || p1 >= nextPropertyID || p2 >= nextPropertyID) {
            throw new ArrayIndexOutOfBoundsException();
        }
        return subPropertyMatrix[p1][p2];
    }

    /**
     * Decide whether p1 is sub property of p2 
     * 
     * 
     * @param p 
     * @param q 
     * @return true if p1 is sub property of p2
     */
    public synchronized boolean isSubPropertyOf(Role p1, Role p2) {
        return isSubPropertyOf(getCreatePropertyID(p1), getCreatePropertyID(p2));
    }

    /**
     * Get Sub Properties of p
     * 
     * @param p
     *            ID of given property
     * @return URIs of all sub properties
     */
    public synchronized List getSubProperties(int p) {
        if (p < 0 || p >= nextPropertyID) {
            throw new ArrayIndexOutOfBoundsException();
        }

        Vector result = new Vector();
        for (int i = 0; i < nextPropertyID; i++) {
            if (subPropertyMatrix[i][p]
                && i != topPropertyID && i != bottomPropertyID) {
                result.add(properties[i]);
            }
        }

        return result;
    }

    /**
     * Get Sub Properties of p
     * 
     * @param p
     * @return URIs of all sub properties
     */
    public synchronized List getSubProperties(Role p) {
        return getSubProperties(getCreatePropertyID(p));
    }

    /**
     * Get Super Properties of p
     * 
     * @param p
     *            ID of given property
     * @return URIs of all super properties
     */
    public synchronized List getSuperProperties(int p) {
        if (p < 0 || p >= nextPropertyID) {
            throw new ArrayIndexOutOfBoundsException();
        }

        Vector result = new Vector();
        for (int i = 0; i < nextPropertyID; i++) {
            if (subPropertyMatrix[p][i]
                && i != topPropertyID && i != bottomPropertyID) {
                result.add(properties[i]);
            }
        }

        return result;
    }

    /**
     * Get Super Properties of p
     * 
     * @param p
     * @return URIs of all super properties
     */
    public synchronized List getSuperProperties(Role p) {
        return getSuperProperties(getCreatePropertyID(p));
    }

    private synchronized int getNextID() {
        if (nextPropertyID == size) {
            int newSize = size * 3 / 2;
            boolean[][] newMatrix = new boolean[newSize][newSize];
            String[] newPropertiesList = new String[newSize];

            for (int i = 0; i < size; i++) {
                System
                        .arraycopy(subPropertyMatrix[i], 0, newMatrix[i], 0,
                                size);
            }

            System.arraycopy(properties, 0, newPropertiesList, 0, size);

            size = newSize;
            subPropertyMatrix = newMatrix;
            properties = newPropertiesList;

        }
        return nextPropertyID++;
    }

    private HashMap propertyMap = new HashMap();

    private int nextPropertyID = 0;

    private int size;

    private boolean[][] subPropertyMatrix = null;

    private String[] properties = null;

    private int topPropertyID, bottomPropertyID;
    
}
