/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLDocumentImpl.java,v 1.2 2007/03/18 09:07:04 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource.parser.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedList;

import org.eclipse.eodm.owl.resource.parser.OWLDocument;
import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;

/**
 * OWLDocumentImpl - OWLDocument Implementation.
 * 
 */
public class OWLDocumentImpl implements OWLDocument {
    /** local file in defalut */
    private boolean localFile = true;

    /**
     * The local uri of the document
     */
    private String localURI = null;

    /**
     * The public uri of the document by which other documents can import the
     * document.
     */
    private String publicURI = null;

    /**
     * The encoding of the document.
     */
    private String charset = null;

    /**
     * Default base URI of the document
     */
    private String defaultBaseURI = null;

    /**
     * The input stream of the document
     */
    private InputStream inputStream = null;

    /**
     * URIs of ontologies which are imported by this document
     */
    private LinkedList incomingURIs = new LinkedList();

    /**
     * URIs of ontologies which imports this document
     */
    private LinkedList outgoingURIs = new LinkedList();

    /**
     * The flag to denote if the ontology has been parsed
     */
    private boolean parsingFlag = false;

    /**
     * The generated ontology graph from this document
     */
    OWLOntologyGraph ontologygraph;

    /**
     * Constructs an OWL Document.
     */
    public OWLDocumentImpl() {
        super();
    }

    /**
     * Construct an OWL Document.
     * 
     * @param localURI
     *            the document local uri
     * @param publicURI
     *            the document publid uri
     * @param localFile
     *            whether it is a local file
     * @return the document
     */
    public OWLDocumentImpl(String localURI, String publicURI, boolean localFile) {
        this.localURI = localURI;
        this.publicURI = publicURI;

        if (publicURI != null)
            addImportOntologyURIs(publicURI);
        this.localFile = localFile;

        if (localURI != null)
            try {
                this.localURI = new File(localURI).getCanonicalPath();
            } catch (IOException e) {
                throw new OWLParserException("The file "
                                             + localFile + " is not found.");
            }
    }

    /**
     * Construct OWLDocument by InputStream
     * 
     * @param stream
     *            the input stream
     */
    public OWLDocumentImpl(InputStream stream) {
        this.localURI = "OWLInputStream";
        this.publicURI = null;
        this.localFile = true;
        this.inputStream = stream;
    }

    /**
     * Get inputStream
     * 
     * @return inputStream
     */
    public InputStream getOWLInputStream() {
        return inputStream;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#isLocalFile()
     */
    public boolean isLocalFile() {
        return localFile;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getLocalURI()
     */
    public String getLocalURI() {
        return localURI;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getPublicURI()
     */
    public String getPublicURI() {
        return publicURI;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setPublicURI(java.lang.String)
     */
    public void setPublicURI(String URI) {
        this.publicURI = URI;
        addImportOntologyURIs(URI);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getCharset()
     */
    public String getCharset() {
        return charset;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setCharset(java.lang.String)
     */
    public void setCharset(String charset) {
        this.charset = charset;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#readInput()
     */
    public InputStream readInput() throws OWLParserException {
        if (inputStream != null)
            return inputStream;

        InputStream input = null;

        URL url = null;
        try {
            if (publicURI != null)
                url = new URL(publicURI);
        } catch (MalformedURLException e1) {
            throw new OWLParserException("The document uri "
                                         + publicURI + " is malformed.", e1);
        }
        // the document is a local file
        if (isLocalFile())
            try {
                input = new FileInputStream(localURI);
            } catch (FileNotFoundException e) {
                throw new OWLParserException("The file "
                                             + localURI + " is not found.", e);
            }
        else if (url != null && url.getProtocol().equalsIgnoreCase("file")) {
            try {
                String file = new File(url.getPath()).getCanonicalPath();
                input = new FileInputStream(file);
                this.localURI = file;
                this.localFile = true;
            } catch (IOException e) {
                e.printStackTrace();
                throw new OWLParserException("Cannot load the file", e);
            }
        }
        // the document is a web resource
        else
            try {
                input = new URL(publicURI).openStream();
            } catch (MalformedURLException e) {
                throw new OWLParserException("The document uri "
                                             + publicURI + " is malformed.", e);
            } catch (IOException e) {
                throw new OWLParserException(
                        "IO exception when open the input stream.", e);
            }

        return input;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getDefaultBaseURI()
     */
    public String getDefaultBaseURI() {
        return defaultBaseURI;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setDefaultBaseURI(java.lang.String)
     */
    public void setDefaultBaseURI(String baseURI) {
        this.defaultBaseURI = baseURI;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#isParsed()
     */
    public boolean isParsed() {
        return parsingFlag;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getImportOntologyURIs(boolean)
     */
    public LinkedList getImportOntologyURIs(boolean flag) {
        if (flag)
            return incomingURIs;
        else
            return outgoingURIs;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setParsed()
     */
    public void setParsed() {
        parsingFlag = true;

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#addImportOntologyURIs(java.lang.String)
     */
    public void addImportOntologyURIs(String str) {
        for (int i = 0; i < incomingURIs.size(); i++) {
            if (str.equals((String) incomingURIs.get(i)))
                return;
        }
        incomingURIs.addLast(str);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#getOntologyGraph()
     */
    public OWLOntologyGraph getOntologyGraph() {
        return ontologygraph;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setOntologyGraph(org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph)
     */
    public void setOntologyGraph(OWLOntologyGraph ontologygraph) {
        this.ontologygraph = ontologygraph;

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.eodm.owl.resource.parser.OWLDocument#setInputStream(java.io.InputStream)
     */
    public void setInputStream(InputStream input) {
        inputStream = null;

    }
}