/**
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * $Id: OWLFactory.java,v 1.3 2007/04/03 09:49:39 lzhang Exp $
 */
package org.eclipse.eodm;

import java.net.URISyntaxException;

import org.eclipse.eodm.owl.owlbase.AllValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.CardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.ComplementClass;
import org.eclipse.eodm.owl.owlbase.EnumeratedClass;
import org.eclipse.eodm.owl.owlbase.HasValueRestriction;
import org.eclipse.eodm.owl.owlbase.Individual;
import org.eclipse.eodm.owl.owlbase.IntersectionClass;
import org.eclipse.eodm.owl.owlbase.MaxCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.MinCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.OWLAllDifferent;
import org.eclipse.eodm.owl.owlbase.OWLAnnotationProperty;
import org.eclipse.eodm.owl.owlbase.OWLClass;
import org.eclipse.eodm.owl.owlbase.OWLDataRange;
import org.eclipse.eodm.owl.owlbase.OWLDatatypeProperty;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.owlbase.OWLObjectProperty;
import org.eclipse.eodm.owl.owlbase.OWLOntology;
import org.eclipse.eodm.owl.owlbase.OWLOntologyProperty;
import org.eclipse.eodm.owl.owlbase.SomeValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.Statement;
import org.eclipse.eodm.owl.owlbase.UnionClass;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfweb.Namespace;

/**
 * OWLFactory - An abstract factory interface for creating all kinds of OWL objects.
 * 
 * <p>The creation method for OWL resources has the following variants:
 * <ul>
 *  <li><code>createXYZ( RDFGraph graph, String uriRefString )</code>
 *  <li><code>createXYZ( RDFGraph graph, URIReferecne uriRef)</code>
 *  <li><code>createXYZ( RDFGraph graph, Namespace namespace, String localNameString )</code>
 * </ul>
 * All of the above methods return an object that uses the <code>graph</code> as its context
 * graph and corresponds to a resource identified by a RDF URI Reference. 
 * Both the identity and the context graph can not be changed after the object 
 * is returned from the above methods. However, a resource may have multiple
 * RDF URI References that are equal to each other.<p>
 * 
 * <p>To create an anonymous resource, you may first create a BlankNode using the 
 * method {@link org.eclipse.eodm.RDFFactory#createBlankNode(RDFGraph) createBlankNode(RDFGraph g)}
 * and force that blank node to the type of resource you need using the method
 * {@link org.eclipse.eodm.rdf.rdfbase.RDFSResource#asType(Class) asType(Class)}.
 * The factory also provides methods for creating some kind of anonymous resource,
 * such as sub-classes of OWLRestrictions</p>
 * 
 * <a name="uriRefString"></a>
 * <p>The <code>uriRefString</code> parameter in the first kind of method for creating
 * a non-anonymous resource must be a valid <strong>RDF URI Reference</strong>. 
 * Defined by <a href="http://www.w3.org/TR/2004/REC-rdf-concepts-20040210/#dfn-URI-reference">
 * W3C RDF Concepts and Abstract Syntaxes</a>, a RDF URI reference is a Unicode 
 * string that:
 * <ul>
 *   <li>does not contain any control characters ( #x00 - #x1F, #x7F-#x9F)
 *   <li>and would produce a valid URI character sequence (per RFC-2396, 
 *   sections 2.1) representing an absolute URI with optional fragment identifier 
 *   when subjected to the encoding described below.
 * </ul>
 * The encoding consists of:
 * <ul>
 *   <li>encoding the Unicode string as UTF-8 [RFC-2279], giving a sequence of octet values.
 *   <li>%-escaping octets that do not correspond to permitted US-ASCII characters.
 * </ul>
 * </p>
 *   
 * <p></p>
 * 
 * @eodmext  This factory interface is added by EODM.
 * @author zhanglei
 */
public interface OWLFactory {
	 OWLFactory eINSTANCE = new org.eclipse.eodm.impl.OWLFactoryImpl();
	 /**
	    * Returns an object of class '<em>OWLOntology</em>'.
	    * 
	    * @param graph the context graph of this OWLOntology.
	    * @param uriRefString the RDF URI Reference of the OWLOntology.
	    *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	    * @return an object of class '<em>OWLOntology</em>'.
	    * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	    *         valid RDF URI Reference. 
	    */
	    OWLOntology createOWLOntology( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	    * Returns an object of class '<em>OWLOntology</em>'.
	    * 
	    * @param graph  the context graph of this OWLOntology
	    * @param uriRef the RDF URI Reference of the OWLOntology.
	    * @return an object of class '<em>OWLOntology</em>'.
	    */
	    OWLOntology createOWLOntology( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLGraph</em>'. 
	     * 
	     * @param uriRefString  The string must be <code>null</code> or <a href="#uriRefString"> a valid RDF URI Reference</a>.
	     *        If it is a valid RDF URI Reference, it serves as the name of the graph
	     *        and can not be changed. If it is <code>null</code>, this graph then
	     *        does not have a name and can not be given a name later.
	     * @return an object of class '<em>OWLGraph</em>'.
	     * @throws URISyntaxException if the uriRefString is not <code>null</code> and
	     *         is not a valid RDF URI Reference.
	     */
	    OWLGraph createOWLGraph( String uriRefString )
	    throws URISyntaxException;

	    /**
	     * Returns an object of class '<em>Statement</em>'.
	     * If this statement is reified, it is an anonymous statement without a RDF URI
	     * reference and can not be given a URI reference.
	     *  
	     * @return an object of class '<em>Statement</em>'.
	     */
	    Statement createStatement(OWLGraph graph);

	    /**
	     * Returns an object of class '<em>Statement</em>'.
	     * If this statement is reified, the reified statement resource will use
	     * <code>uriRefString</code> as its RDF URI Reference and can not be changed.
	     * 
	     * @param uriRefString  <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>Statement</em>'.
	     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
	     */
	    Statement createStatement( OWLGraph graph,String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>Statement</em>'.
	     * 
	     * @param graph  the context graph of this Statement
	     * @param uriRef the RDF URI Reference of the Statement.
	     * @return an object of class '<em>Statement</em>'.
	     */
	    Statement createStatement( OWLGraph graph, URIReference uriRef );

	    
	    /**
	     * Returns an object of class '<em>Statement</em>'.
	     * 
	     * @param graph           the context graph of this Statement
	     * @param namespace       the namespace of the RDF URI Reference of the Statement
	     * @param localNameString the fragment identifier of the RDF URI Reference of the Statement
	     * @return an object of class '<em>Statement</em>'.
	     */
	    Statement createStatement( OWLGraph graph, Namespace namespace, String localNameString );


	    /**
	     * Returns an object of class '<em>OWLOntologyProperty</em>'.
	     * 
	     * @param graph the context graph of this OWLOntologyProperty.
	     * @param uriRefString the RDF URI Reference of the OWLOntologyProperty.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLOntologyProperty</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLOntologyProperty createOWLOntologyProperty( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLOntologyProperty</em>'.
	     * 
	     * @param graph  the context graph of this OWLOntologyProperty
	     * @param uriRef the RDF URI Reference of the OWLOntologyProperty.
	     * @return an object of class '<em>OWLOntologyProperty</em>'.
	     */
	    OWLOntologyProperty createOWLOntologyProperty( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLOntologyProperty</em>'.
	     * 
	     * @param graph           the context graph of this OWLOntologyProperty
	     * @param namespace       the namespace of the RDF URI Reference of the OWLOntologyProperty
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLOntologyProperty
	     * @return an object of class '<em>OWLOntologyProperty</em>'.
	     */
	    OWLOntologyProperty createOWLOntologyProperty( OWLGraph graph, Namespace namespace, String localNameString );


	    /**
	     * Returns an object of class '<em>OWLClass</em>'.
	     * 
	     * @param graph the context graph of this OWLClass.
	     * @param uriRefString the RDF URI Reference of the OWLClass.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLClass</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLClass createOWLClass( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLClass</em>'.
	     * 
	     * @param graph  the context graph of this OWLClass
	     * @param uriRef the RDF URI Reference of the OWLClass.
	     * @return an object of class '<em>OWLClass</em>'.
	     */
	    OWLClass createOWLClass( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLClass</em>'.
	     * 
	     * @param graph           the context graph of this OWLClass
	     * @param namespace       the namespace of the RDF URI Reference of the OWLClass
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLClass
	     * @return an object of class '<em>OWLClass</em>'.
	    */
	    OWLClass createOWLClass( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns an object of class '<em>OWLObjectProperty</em>'.
	     * 
	     * @param graph the context graph of this OWLObjectProperty.
	     * @param uriRefString the RDF URI Reference of the OWLObjectProperty.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLObjectProperty</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLObjectProperty createOWLObjectProperty( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLObjectProperty</em>'.
	     * 
	     * @param graph  the context graph of this OWLObjectProperty
	     * @param uriRef the RDF URI Reference of the OWLObjectProperty.
	     * @return an object of class '<em>OWLObjectProperty</em>'.
	     */
	    OWLObjectProperty createOWLObjectProperty( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLObjectProperty</em>'.
	     * 
	     * @param graph           the context graph of this OWLObjectProperty
	     * @param namespace       the namespace of the RDF URI Reference of the OWLObjectProperty
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLObjectProperty
	     * @return an object of class '<em>OWLObjectProperty</em>'.
	     */
	    OWLObjectProperty createOWLObjectProperty( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns an object of class '<em>OWLDatatypeProperty</em>'.
	     * 
	     * @param graph the context graph of this OWLDatatypeProperty.
	     * @param uriRefString the RDF URI Reference of the OWLDatatypeProperty.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLDatatypeProperty</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLDatatypeProperty createOWLDatatypeProperty( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLDatatypeProperty</em>'.
	     * 
	     * @param graph  the context graph of this OWLDatatypeProperty
	     * @param uriRef the RDF URI Reference of the OWLDatatypeProperty.
	     * @return an object of class '<em>OWLDatatypeProperty</em>'.
	     */
	    OWLDatatypeProperty createOWLDatatypeProperty( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLDatatypeProperty</em>'.
	     * 
	     * @param graph           the context graph of this OWLDatatypeProperty
	     * @param namespace       the namespace of the RDF URI Reference of the OWLDatatypeProperty
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLDatatypeProperty
	     * @return an object of class '<em>OWLDatatypeProperty</em>'.
	     */
	    OWLDatatypeProperty createOWLDatatypeProperty( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type Individual (in other words, 
	     * a anonymous individual). 
	     * 
	     * The individual can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this Individual
	     * @return a new unique blank node of type Individual
	     */
	    Individual createIndividual( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>Individual</em>'.
	     * 
	     * @param graph the context graph of this Individual.
	     * @param uriRefString the RDF URI Reference of the Individual.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>Individual</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    Individual createIndividual( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>Individual</em>'.
	     * 
	     * @param graph  the context graph of this Individual
	     * @param uriRef the RDF URI Reference of the Individual.
	     * @return an object of class '<em>Individual</em>'.
	     */
	    Individual createIndividual( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>Individual</em>'.
	     * 
	     * @param graph           the context graph of this Individual
	     * @param namespace       the namespace of the RDF URI Reference of the Individual
	     * @param localNameString the fragment identifier of the RDF URI Reference of the Individual
	     * @return an object of class '<em>Individual</em>'.
	     */
	    Individual createIndividual( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type OWLAllDifferent.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this OWLAllDifferent
	     * @return a new unique blank node of type OWLAllDifferent
	     */
	    OWLAllDifferent createOWLAllDifferent( OWLGraph graph );

	    /**
	     * Returns an object of class '<em>OWLAllDifferent</em>'.
	     * 
	     * @param graph the context graph of this OWLAllDifferent.
	     * @param uriRefString the RDF URI Reference of the OWLAllDifferent.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLAllDifferent</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLAllDifferent createOWLAllDifferent( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLAllDifferent</em>'.
	     * 
	     * @param graph  the context graph of this OWLAllDifferent
	     * @param uriRef the RDF URI Reference of the OWLAllDifferent.
	     * @return an object of class '<em>OWLAllDifferent</em>'.
	     */
	    OWLAllDifferent createOWLAllDifferent( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLAllDifferent</em>'.
	     * 
	     * @param graph           the context graph of this OWLAllDifferent
	     * @param namespace       the namespace of the RDF URI Reference of the OWLAllDifferent
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLAllDifferent
	     * @return an object of class '<em>OWLAllDifferent</em>'.
	     */
	    OWLAllDifferent createOWLAllDifferent( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type OWLDataRange.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this OWLDataRange
	     * @return a new unique blank node of type OWLDataRange
	     */
	    OWLDataRange createOWLDataRange( OWLGraph graph );

	    /**
	     * Returns an object of class '<em>OWLDataRange</em>'.
	     * 
	     * @param graph the context graph of this OWLDataRange.
	     * @param uriRefString the RDF URI Reference of the OWLDataRange.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLDataRange</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLDataRange createOWLDataRange( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLDataRange</em>'.
	     * 
	     * @param graph  the context graph of this OWLDataRange
	     * @param uriRef the RDF URI Reference of the OWLDataRange.
	     * @return an object of class '<em>OWLDataRange</em>'.
	     */
	    OWLDataRange createOWLDataRange( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLDataRange</em>'.
	     * 
	     * @param graph           the context graph of this OWLDataRange
	     * @param namespace       the namespace of the RDF URI Reference of the OWLDataRange
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLDataRange
	     * @return an object of class '<em>OWLDataRange</em>'.
	     */
	    OWLDataRange createOWLDataRange( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns an object of class '<em>OWLAnnotationProperty</em>'.
	     * 
	     * @param graph the context graph of this OWLAnnotationProperty.
	     * @param uriRefString the RDF URI Reference of the OWLAnnotationProperty.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>OWLAnnotationProperty</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    OWLAnnotationProperty createOWLAnnotationProperty( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>OWLAnnotationProperty</em>'.
	     * 
	     * @param graph  the context graph of this OWLAnnotationProperty
	     * @param uriRef the RDF URI Reference of the OWLAnnotationProperty.
	     * @return an object of class '<em>OWLAnnotationProperty</em>'.
	     */
	    OWLAnnotationProperty createOWLAnnotationProperty( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>OWLAnnotationProperty</em>'.
	     * 
	     * @param graph           the context graph of this OWLAnnotationProperty
	     * @param namespace       the namespace of the RDF URI Reference of the OWLAnnotationProperty
	     * @param localNameString the fragment identifier of the RDF URI Reference of the OWLAnnotationProperty
	     * @return an object of class '<em>OWLAnnotationProperty</em>'.
	     */
	    OWLAnnotationProperty createOWLAnnotationProperty( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type EnumeratedClass.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this EnumeratedClass
	     * @return a new unique blank node of type EnumeratedClass
	     */
	    EnumeratedClass createEnumeratedClass( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>EnumeratedClass</em>'.
	     * 
	     * @param graph the context graph of this EnumeratedClass.
	     * @param uriRefString the RDF URI Reference of the EnumeratedClass.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>EnumeratedClass</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    EnumeratedClass createEnumeratedClass( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>EnumeratedClass</em>'.
	     * 
	     * @param graph  the context graph of this EnumeratedClass
	     * @param uriRef the RDF URI Reference of the EnumeratedClass.
	     * @return an object of class '<em>EnumeratedClass</em>'.
	     */
	    EnumeratedClass createEnumeratedClass( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>EnumeratedClass</em>'.
	     * 
	     * @param graph           the context graph of this EnumeratedClass
	     * @param namespace       the namespace of the RDF URI Reference of the EnumeratedClass
	     * @param localNameString the fragment identifier of the RDF URI Reference of the EnumeratedClass
	     * @return an object of class '<em>EnumeratedClass</em>'.
	     */
	    EnumeratedClass createEnumeratedClass( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type IntersectionClass.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this IntersectionClass
	     * @return a new unique blank node of type IntersectionClass
	     */
	    IntersectionClass createIntersectionClass( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>IntersectionClass</em>'.
	     * 
	     * @param graph the context graph of this IntersectionClass.
	     * @param uriRefString the RDF URI Reference of the IntersectionClass.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>IntersectionClass</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    IntersectionClass createIntersectionClass( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>IntersectionClass</em>'.
	     * 
	     * @param graph  the context graph of this IntersectionClass
	     * @param uriRef the RDF URI Reference of the IntersectionClass.
	     * @return an object of class '<em>IntersectionClass</em>'.
	     */
	    IntersectionClass createIntersectionClass( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>IntersectionClass</em>'.
	     * 
	     * @param graph           the context graph of this IntersectionClass
	     * @param namespace       the namespace of the RDF URI Reference of the IntersectionClass
	     * @param localNameString the fragment identifier of the RDF URI Reference of the IntersectionClass
	     * @return an object of class '<em>IntersectionClass</em>'.
	     */
	    IntersectionClass createIntersectionClass( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type UnionClass.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this UnionClass
	     * @return a new unique blank node of type UnionClass
	     */
	    UnionClass createUnionClass( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>UnionClass</em>'.
	     * 
	     * @param graph the context graph of this UnionClass.
	     * @param uriRefString the RDF URI Reference of the UnionClass.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>UnionClass</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    UnionClass createUnionClass( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>UnionClass</em>'.
	     * 
	     * @param graph  the context graph of this UnionClass
	     * @param uriRef the RDF URI Reference of the UnionClass.
	     * @return an object of class '<em>UnionClass</em>'.
	     */
	    UnionClass createUnionClass( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>UnionClass</em>'.
	     * 
	     * @param graph           the context graph of this UnionClass
	     * @param namespace       the namespace of the RDF URI Reference of the UnionClass
	     * @param localNameString the fragment identifier of the RDF URI Reference of the UnionClass
	     * @return an object of class '<em>UnionClass</em>'.
	     */
	    UnionClass createUnionClass( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type ComplementClass.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this ComplementClass
	     * @return a new unique blank node of type ComplementClass
	     */
	    ComplementClass createComplementClass( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>ComplementClass</em>'.
	     * 
	     * @param graph the context graph of this ComplementClass.
	     * @param uriRefString the RDF URI Reference of the ComplementClass.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>ComplementClass</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    ComplementClass createComplementClass( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>ComplementClass</em>'.
	     * 
	     * @param graph  the context graph of this ComplementClass
	     * @param uriRef the RDF URI Reference of the ComplementClass.
	     * @return an object of class '<em>ComplementClass</em>'.
	     */
	    ComplementClass createComplementClass( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>ComplementClass</em>'.
	     * 
	     * @param graph           the context graph of this ComplementClass
	     * @param namespace       the namespace of the RDF URI Reference of the ComplementClass
	     * @param localNameString the fragment identifier of the RDF URI Reference of the ComplementClass
	     * @return an object of class '<em>ComplementClass</em>'.
	     */
	    ComplementClass createComplementClass( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type HasValueRestriction.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this HasValueRestriction
	     * @return a new unique blank node of type HasValueRestriction
	     */
	    HasValueRestriction createHasValueRestriction( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>HasValueRestriction</em>'.
	     * 
	     * @param graph the context graph of this HasValueRestriction.
	     * @param uriRefString the RDF URI Reference of the HasValueRestriction.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>HasValueRestriction</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    HasValueRestriction createHasValueRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>HasValueRestriction</em>'.
	     * 
	     * @param graph  the context graph of this HasValueRestriction
	     * @param uriRef the RDF URI Reference of the HasValueRestriction.
	     * @return an object of class '<em>HasValueRestriction</em>'.
	     */
	    HasValueRestriction createHasValueRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>HasValueRestriction</em>'.
	     * 
	     * @param graph           the context graph of this HasValueRestriction
	     * @param namespace       the namespace of the RDF URI Reference of the HasValueRestriction
	     * @param localNameString the fragment identifier of the RDF URI Reference of the HasValueRestriction
	     * @return an object of class '<em>HasValueRestriction</em>'.
	     */
	    HasValueRestriction createHasValueRestriction( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type AllValuesFromRestriction.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this AllValuesFromRestriction
	     * @return a new unique blank node of type AllValuesFromRestriction
	     */
	    AllValuesFromRestriction createAllValuesFromRestriction( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>AllValuesFromRestriction</em>'.
	     * 
	     * @param graph the context graph of this AllValuesFromRestriction.
	     * @param uriRefString the RDF URI Reference of the AllValuesFromRestriction.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>AllValuesFromRestriction</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    AllValuesFromRestriction createAllValuesFromRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>AllValuesFromRestriction</em>'.
	     * 
	     * @param graph  the context graph of this AllValuesFromRestriction
	     * @param uriRef the RDF URI Reference of the AllValuesFromRestriction.
	     * @return an object of class '<em>AllValuesFromRestriction</em>'.
	     */
	    AllValuesFromRestriction createAllValuesFromRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>AllValuesFromRestriction</em>'.
	     * 
	     * @param graph           the context graph of this AllValuesFromRestriction
	     * @param namespace       the namespace of the RDF URI Reference of the AllValuesFromRestriction
	     * @param localNameString the fragment identifier of the RDF URI Reference of the AllValuesFromRestriction
	     * @return an object of class '<em>AllValuesFromRestriction</em>'.
	     */
	    AllValuesFromRestriction createAllValuesFromRestriction( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type SomeValuesFromRestriction.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this SomeValuesFromRestriction
	     * @return a new unique blank node of type SomeValuesFromRestriction
	     */
	    SomeValuesFromRestriction createSomeValuesFromRestriction( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>SomeValuesFromRestriction</em>'.
	     * 
	     * @param graph the context graph of this SomeValuesFromRestriction.
	     * @param uriRefString the RDF URI Reference of the SomeValuesFromRestriction.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>SomeValuesFromRestriction</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    SomeValuesFromRestriction createSomeValuesFromRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>SomeValuesFromRestriction</em>'.
	     * 
	     * @param graph  the context graph of this SomeValuesFromRestriction
	     * @param uriRef the RDF URI Reference of the SomeValuesFromRestriction.
	     * @return an object of class '<em>SomeValuesFromRestriction</em>'.
	     */
	    SomeValuesFromRestriction createSomeValuesFromRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>SomeValuesFromRestriction</em>'.
	     * 
	     * @param graph           the context graph of this SomeValuesFromRestriction
	     * @param namespace       the namespace of the RDF URI Reference of the SomeValuesFromRestriction
	     * @param localNameString the fragment identifier of the RDF URI Reference of the SomeValuesFromRestriction
	     * @return an object of class '<em>SomeValuesFromRestriction</em>'.
	     */
	    SomeValuesFromRestriction createSomeValuesFromRestriction( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type CardinalityRestriction.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this CardinalityRestriction
	     * @return a new unique blank node of type CardinalityRestriction
	     */
	    CardinalityRestriction createCardinalityRestriction( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>CardinalityRestriction</em>'.
	     * 
	     * @param graph the context graph of this CardinalityRestriction.
	     * @param uriRefString the RDF URI Reference of the CardinalityRestriction.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>CardinalityRestriction</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    CardinalityRestriction createCardinalityRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>CardinalityRestriction</em>'.
	     * 
	     * @param graph  the context graph of this CardinalityRestriction
	     * @param uriRef the RDF URI Reference of the CardinalityRestriction.
	     * @return an object of class '<em>CardinalityRestriction</em>'.
	     */
	    CardinalityRestriction createCardinalityRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	    * Returns an object of class '<em>CardinalityRestriction</em>'.
	    * 
	    * @param graph           the context graph of this CardinalityRestriction
	    * @param namespace       the namespace of the RDF URI Reference of the CardinalityRestriction
	    * @param localNameString the fragment identifier of the RDF URI Reference of the CardinalityRestriction
	    * @return an object of class '<em>CardinalityRestriction</em>'.
	    */
	    CardinalityRestriction createCardinalityRestriction( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	     * Returns a new unique blank node of type MaxCardinalityRestriction.
	     * The resource can not be given a URI Reference later.
	     *
	     * @param graph the context graph of this MaxCardinalityRestriction
	     * @return a new unique blank node of type MaxCardinalityRestriction
	     */
	    MaxCardinalityRestriction createMaxCardinalityRestriction( OWLGraph graph );
	    
	    /**
	     * Returns an object of class '<em>MaxCardinalityRestriction</em>'.
	     * 
	     * @param graph the context graph of this MaxCardinalityRestriction.
	     * @param uriRefString the RDF URI Reference of the MaxCardinalityRestriction.
	     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	     * @return an object of class '<em>MaxCardinalityRestriction</em>'.
	     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	     *         valid RDF URI Reference. 
	     */
	    MaxCardinalityRestriction createMaxCardinalityRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	     * Returns an object of class '<em>MaxCardinalityRestriction</em>'.
	     * 
	     * @param graph  the context graph of this MaxCardinalityRestriction
	     * @param uriRef the RDF URI Reference of the MaxCardinalityRestriction.
	     * @return an object of class '<em>MaxCardinalityRestriction</em>'.
	     */
	    MaxCardinalityRestriction createMaxCardinalityRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	     * Returns an object of class '<em>MaxCardinalityRestriction</em>'.
	     * 
	     * @param graph           the context graph of this MaxCardinalityRestriction
	     * @param namespace       the namespace of the RDF URI Reference of the MaxCardinalityRestriction
	     * @param localNameString the fragment identifier of the RDF URI Reference of the MaxCardinalityRestriction
	     * @return an object of class '<em>MaxCardinalityRestriction</em>'.
	     */
	    MaxCardinalityRestriction createMaxCardinalityRestriction( OWLGraph graph, Namespace namespace, String localNameString );

	    /**
	    * Returns a new unique blank node of type MinCardinalityRestriction.
	    * The resource can not be given a URI Reference later.
	    *
	    * @param graph the context graph of this MinCardinalityRestriction
	    * @return a new unique blank node of type MinCardinalityRestriction
	    */
	    MinCardinalityRestriction createMinCardinalityRestriction( OWLGraph graph );
	    
	    /**
	    * Returns an object of class '<em>MinCardinalityRestriction</em>'.
	    * 
	    * @param graph the context graph of this MinCardinalityRestriction.
	    * @param uriRefString the RDF URI Reference of the MinCardinalityRestriction.
	    *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
	    * @return an object of class '<em>MinCardinalityRestriction</em>'.
	    * @throws URISyntaxException if the <code>uriRefString</code> is not a 
	    *         valid RDF URI Reference. 
	    */
	    MinCardinalityRestriction createMinCardinalityRestriction( OWLGraph graph, String uriRefString )
	    throws URISyntaxException;
	    
	    /**
	    * Returns an object of class '<em>MinCardinalityRestriction</em>'.
	    * 
	    * @param graph  the context graph of this MinCardinalityRestriction
	    * @param uriRef the RDF URI Reference of the MinCardinalityRestriction.
	    * @return an object of class '<em>MinCardinalityRestriction</em>'.
	    */
	    MinCardinalityRestriction createMinCardinalityRestriction( OWLGraph graph, URIReference uriRef );
	    
	    /**
	    * Returns an object of class '<em>MinCardinalityRestriction</em>'.
	    * 
	    * @param graph           the context graph of this MinCardinalityRestriction
	    * @param namespace       the namespace of the RDF URI Reference of the MinCardinalityRestriction
	    * @param localNameString the fragment identifier of the RDF URI Reference of the MinCardinalityRestriction
	    * @return an object of class '<em>MinCardinalityRestriction</em>'.
	    */
	    MinCardinalityRestriction createMinCardinalityRestriction( OWLGraph graph, Namespace namespace, String localNameString );
}
