/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: URIReference.java,v 1.2 2007/03/18 08:39:02 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.element;

import java.net.URI;

import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;

/**
 * URIReference
 */
public class URIReference implements RDFResourceElement {
    public static final String POUND = "#";

    private String namespace = null;

    private String localName = null;

    private String query = null;

    private URI uriRef = null;

    /**
     * Construct a URI reference
     * 
     * @param uri
     *            URI string
     * @throws IllegalArgumentException
     */
    public URIReference(String uri) throws IllegalArgumentException {
        if (uri == null) {
            throw new ParserException("URI must be not null.");
        }
        createURI(uri);
        namespace = namespace.intern();
        localName = localName.intern();
    }

    /**
     * Construct a URI reference
     * 
     * @param URI
     *            string
     * @throws IllegalArgumentException
     */
    public URIReference(URI uri) throws IllegalArgumentException {
        if (uri == null) {
            throw new ParserException("URI must be not null.");
        }

        uriRef = uri;
        createURI();
        namespace = namespace.intern();
        localName = localName.intern();
    }

    /**
     * Construct a URI reference
     * 
     * @param namespace
     *            the namespace of URI
     * @param localName
     *            the localname of URI
     */

    public URIReference(String ns, String ln) throws ParserException {
        if (ns == null || ln == null) {
            throw new ParserException(
                    "URI namespace and local name must not be null.");
        }

        createURI(ns + ln);
        namespace = namespace.intern();
        localName = localName.intern();
    }

    /**
     * Construct a valid URI
     * 
     * @param uri
     *            The input URI
     * @throws ParserException
     *             if the uri is invalid, parser exception will be thrown out.
     */
    private void createURI(String uri) throws IllegalArgumentException {
        uri = convertToValidURI(uri);
        uriRef = URI.create(uri);
        createURI();
    }

    /**
     * Construct a valid URI
     * 
     * @param uri
     *            The input URI
     * @throws ParserException
     *             if the uri is invalid, parser exception will be thrown out.
     */
    private void createURI() throws IllegalArgumentException {
        query = uriRef.getQuery();

        if (uriRef.getFragment() != null) {
            namespace = uriRef.getScheme()
                        + ":" + uriRef.getSchemeSpecificPart() + POUND;
            localName = uriRef.getFragment();
            namespace = convertToValidURI(namespace);
            localName = convertToValidURI(localName);
            return;
        }

        namespace = uriRef.getScheme() + ":" + uriRef.getSchemeSpecificPart();
        localName = "";
        if (uriRef.getPath() != null) {
            int lim = uriRef.getPath().lastIndexOf('/');
            if (lim >= 0) {
                localName = uriRef.getSchemeSpecificPart().substring(
                        uriRef.getSchemeSpecificPart().lastIndexOf('/') + 1);
                namespace = namespace.substring(0,
                        namespace.lastIndexOf('/') + 1);
            } else {
                // path=""
                localName = query == null ? "" : ("?" + query);
                namespace = uriRef.getScheme()
                            + ":"
                            + uriRef.getSchemeSpecificPart().substring(
                                    0,
                                    uriRef.getSchemeSpecificPart().lastIndexOf(
                                            localName));
            }
        }
        namespace = convertToValidURI(namespace);
        localName = convertToValidURI(localName);
    }

    /**
     * Check whether the uri is an absolute uri.
     * 
     * @return true if the uri is absolute.
     */
    public boolean isAbsolute() {
        return uriRef.isAbsolute();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.rdfs.resource.parser.element.RDFResourceElement#getNamespace()
     */
    public String getNamespace() {
        return namespace;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.rdfs.resource.parser.element.RDFResourceElement#getLocalName()
     */
    public String getLocalName() {
        return localName;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.rdfs.resource.parser.element.RDFResourceElement#getFullURI()
     */
    public String getFullURI() {
        return namespace.concat(localName);
    }

    /**
     * relativeURI = ( net_path | abs_path | rel_path ) [ "?" query ] 1. A
     * relative reference beginning with two slash characters is termed a
     * network-path reference, as defined by <net_path>in Section 3. Such
     * references are rarely used. 2. A relative reference beginning with a
     * single slash character is termed an absolute-path reference, as defined
     * by <abs_path>in Section 3. 3. A relative reference that does not begin
     * with a scheme name or a slash character is termed a relative-path
     * reference.
     * 
     * @param relPath
     *            the input uri
     * @return the relative path
     */
    public URIReference relative(String relPath) throws ParserException {
        relPath = convertToValidURI(relPath);

        if (!isAbsolute())
            throw new ParserException("The uri "
                                      + getFullURI()
                                      + " is not an absolute uri.");

        URI relURI = null;
        relURI = URI.create(relPath);

        if (relURI.isAbsolute()) {
            return new URIReference(relURI);
        }

        if (relPath.startsWith(POUND)) {
            String uri = getNamespace();
            if (uri.endsWith(POUND))
                uri = uri + relPath.substring(1);
            else if (uri.endsWith("/"))
                uri = uri + localName + relPath;
            else
                uri = uri + relPath;
            return new URIReference(uri);
        }

        // 1 net_path, eg. //g
        if (relPath.startsWith("//"))
            return new URIReference(uriRef.getScheme() + ":" + relPath);

        // 2 abs_path
        if (relPath.startsWith("/")) {
            String prefix = uriRef.getSchemeSpecificPart().substring(
                    0,
                    uriRef.getSchemeSpecificPart()
                            .lastIndexOf(uriRef.getPath()));
            String absPath = uriRef.getScheme() + ":" + prefix;
            return new URIReference(absPath + relPath);
        }

        // 3 rel_path
        // 3.1 . | ./ | .. | ../
        if (relPath.startsWith(".")
            || relPath.startsWith("./") || relPath.startsWith("..")
            || relPath.startsWith("../")) {
            String basePath = uriRef.getScheme()
                              + ":"
                              + uriRef.getSchemeSpecificPart().substring(
                                      0,
                                      uriRef.getSchemeSpecificPart()
                                              .lastIndexOf(uriRef.getPath()));
            String absPath = getRelativePath(
                    uriRef.getPath().startsWith("/") ? uriRef.getPath()
                            .substring(0, uriRef.getPath().lastIndexOf('/'))
                            : "", relPath);
            return new URIReference(basePath + absPath);
        }
        // 3.2 other
        else {
            String curPath = uriRef.toString();
            if (uriRef.getFragment() != null) {
                curPath = curPath.substring(0, curPath.length()
                                               - uriRef.getFragment().length()
                                               - 1);
            }

            if (relPath.length() == 0)
                return new URIReference(curPath);
            else {
                if (uriRef.getPath() == null || uriRef.getPath().length() == 0) {
                    return new URIReference(curPath + "/" + relPath);
                } else {
                    return new URIReference(curPath.substring(0, curPath
                            .lastIndexOf('/') + 1)
                                            + relPath);
                }
            }
        }
    }

    /**
     * Construct an absolute path through a base path and a relative path.
     * 
     * @param originalPath
     *            the original path, eg. /dir/dir1, /dir
     * @param relativePath
     *            the relative path, eg. . .. ./ ../ ../refFile, ./refFile
     *            ../../refFile
     * @return the full path, eg. /refFile, /dir/refFile
     */
    private String getRelativePath(String originalPath, String relativePath) {
        if (".".equals(relativePath))
            return originalPath;

        /**
         * "./" can appear in the relative uri only once(just supported in
         * RDFXMLParser).
         */
        if (relativePath.startsWith("./")) {
            String relPath = relativePath.substring(2);
            if (!relPath.startsWith("/"))
                relPath = "/" + relPath;
            return originalPath + relPath;
        }

        if ("..".equals(relativePath)) {
            int slashPos = originalPath.lastIndexOf('/');
            // originalPath=""
            if (slashPos < 0)
                return "/" + relativePath;
            else
                return originalPath.substring(0, slashPos);
        }

        /**
         * "../" can appear in the relative uri muliple times, but it must be
         * present as the prefix of the path. And also it must be a valid
         * relative uri obviously.
         */
        String result = originalPath;
        String tmp = relativePath;
        while (tmp.startsWith("../")) {
            int frag = result.lastIndexOf('/');
            if (frag < 0)
                break;
            result = result.substring(0, frag);
            tmp = tmp.substring(3);
        }
        if (!tmp.startsWith("/"))
            tmp = "/" + tmp;
        return result + tmp;
    }

    /**
     * To filter some reserved charactors percent-encoded " <" & ">"
     * 
     * @param uri
     *            the source uri string
     * @return the target uri string
     */
    private String convertToValidURI(String uri) {
        if (uri == null || uri.length() == 0)
            return uri;

        String result = uri.replaceAll("<", "%3C");
        result = result.replaceAll(">", "%3E");
        result = result.replaceAll(" ", "%20");
        return result;
    }

    public String toString() {
        return getFullURI();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (!(o instanceof URIReference)) {
            return false;
        }

        URIReference uriReference = (URIReference) o;

        /** both the namespace and local name must be equivalent */
        // The local strings are interned, so == comparisons should work.
        if (namespace != uriReference.namespace) {
            return false;
        }

        if (localName != uriReference.localName) {
            return false;
        }

        return true;
    }

    public int hashCode() {
        int result;
        result = namespace.hashCode();
        result = 29 * result + localName.hashCode();
        return result;
    }
}