/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.natures;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jst.j2ee.common.CommonFactory;
import org.eclipse.jst.j2ee.common.EnvEntry;
import org.eclipse.jst.j2ee.common.EnvEntryType;
import org.eclipse.jst.j2ee.common.ParamValue;
import org.eclipse.jst.j2ee.web.componentcore.util.WebArtifactEdit;
import org.eclipse.jst.j2ee.webapplication.Servlet;
import org.eclipse.jst.j2ee.webapplication.ServletMapping;
import org.eclipse.jst.j2ee.webapplication.ServletType;
import org.eclipse.jst.j2ee.webapplication.WebApp;
import org.eclipse.jst.j2ee.webapplication.WebapplicationFactory;
import org.eclipse.wst.common.componentcore.resources.IVirtualComponent;

public class ArtifactHandlerUtil {

	public static void addEnvEntry(IVirtualComponent component, String name, EnvEntryType type, String value) {
		
		WebArtifactEdit webEdit = null;
		
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			EList envProps = webApp.getEnvironmentProperties();
						
			EnvEntry newEntry = CommonFactory.eINSTANCE.createEnvEntry();
			newEntry.setName(name);
			newEntry.setType(type);
			newEntry.setValue(value);
			if(!envEntryExists(envProps, newEntry))
				envProps.add(newEntry);
			
			webEdit.saveIfNecessary(new NullProgressMonitor());
		}
		finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	public static void removeEnvEntry(IVirtualComponent component, String name) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			EList envProps = webApp.getEnvironmentProperties();
						
			int index = getEnvEntryIndex(envProps, name);
			if(index != -1 && name != null) 
				envProps.remove(index);
			webEdit.saveIfNecessary(new NullProgressMonitor());
		}
		finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	private static boolean envEntryExists(EList envProps, EnvEntry entry) {
		
		for(int i=0; i<envProps.size(); i++) {
			EnvEntry e = (EnvEntry)envProps.get(i);
			String eName = e.getName();
			if(eName != null && eName.equals(entry.getName()))
				return true;
		}
		return false;
	}
	
	private static int getEnvEntryIndex(EList envProps, String name) {
		
		for(int i=0; i<envProps.size(); i++) {
			EnvEntry e = (EnvEntry)envProps.get(i);
			String eName = e.getName();
			if(eName != null && eName.equals(name))
				return i;
		}
		return -1;
	}
	
	public static void addContextParam(IVirtualComponent component, String name, String value) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList contextParams = webApp.getContextParams();
			ParamValue contextParam = CommonFactory.eINSTANCE.createParamValue();
			contextParam.setName(name);
			contextParam.setValue(value);
			if(!contextParamExists(contextParams, contextParam))
				contextParams.add(contextParam);
			
			webEdit.saveIfNecessary(new NullProgressMonitor());
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	public static void removeContextParam(IVirtualComponent component, String name) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList contextParams = webApp.getContextParams();
			int index = getContextParamIndex(contextParams, name);
			if(index != -1 && name != null) {
				contextParams.remove(index);
			}
			webEdit.saveIfNecessary(new NullProgressMonitor());
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	private static boolean contextParamExists(EList contextParams, ParamValue contextParam) {
		
		for(int i=0; i<contextParams.size(); i++) {
			ParamValue p = (ParamValue)contextParams.get(i);
			String pName = p.getName();
			if(pName != null && pName.equals(contextParam.getName()))
				return true;
		}
		return false;
		
	}
	
	private static int getContextParamIndex(EList contextParams, String name) {
		
		for(int i=0; i<contextParams.size(); i++) {
			ParamValue p = (ParamValue)contextParams.get(i);
			String pName = p.getName();
			if(pName != null && pName.equals(name))
				return i;
		}
		return -1;
	}
	
	public static void addServlet(IVirtualComponent component, String name, String classname) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList servlets = webApp.getServlets();
			Servlet servlet = WebapplicationFactory.eINSTANCE.createServlet();
			servlet.setServletName(name);
			ServletType servletType = WebapplicationFactory.eINSTANCE.createServletType();
			servletType.setClassName(classname);
			servlet.setWebType(servletType);
			if(!servletExists(servlets, servlet))
				servlets.add(servlet);
			
			webEdit.saveIfNecessary(new NullProgressMonitor());
			
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	public static void removeServlet(IVirtualComponent component, String name) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList servlets = webApp.getServlets();
			int index = getServletIndex(servlets, name);
			if(index != -1 && name != null) {
				servlets.remove(index);
			}
			webEdit.saveIfNecessary(new NullProgressMonitor());
			
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	private static boolean servletExists(EList servlets, Servlet servlet) {
		
		for(int i=0; i<servlets.size(); i++) {
			String name = ((Servlet)servlets.get(i)).getServletName();
			if(name != null && name.equals(servlet.getServletName()))
				return true;
		}
		return false;
	}
	
	private static int getServletIndex(EList servlets, String name) {
		
		for(int i=0; i<servlets.size(); i++) {
			String sName = ((Servlet)servlets.get(i)).getServletName();
			if(sName != null && sName.equals(name))
				return i;
		}
		return -1;
	}
	
	private static Servlet getServlet(EList servlets, String sName) {
		
		for(int i=0; i<servlets.size(); i++) {
			Servlet servlet = (Servlet)servlets.get(i);
			String name = servlet.getServletName();
			if(sName != null && name.equals(sName))
				return servlet;
		}
		return null;
	}
	
	public static void addServletMapping(IVirtualComponent component, String name, String url) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList mappings = webApp.getServletMappings();
			EList servlets = webApp.getServlets();
			Servlet servlet = getServlet(servlets, name);
			if(servlet == null)
				return;
			ServletMapping mapping = WebapplicationFactory.eINSTANCE.createServletMapping();
			mapping.setName(name);
			mapping.setUrlPattern(url);
			mapping.setServlet(servlet);
			if(!servletMappingExists(mappings, mapping))
				mappings.add(mapping);
			
			webEdit.saveIfNecessary(new NullProgressMonitor());
			
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
		
	}
	
	public static void removeServletMapping(IVirtualComponent component, String name) {
		
		WebArtifactEdit webEdit = null;
		try {
			webEdit = WebArtifactEdit.getWebArtifactEditForWrite(component);
			WebApp webApp = webEdit.getWebApp();
			
			EList mappings = webApp.getServletMappings();
			int index = getServletMappingIndex(mappings, name);
			if(index != -1 && name != null) 
				mappings.remove(index);
			
			webEdit.saveIfNecessary(new NullProgressMonitor());
			
		} finally {
			if(webEdit != null)
				webEdit.dispose();
		}
	}
	
	private static boolean servletMappingExists(EList mappings, ServletMapping mapping) {
		
		for(int i=0; i<mappings.size(); i++) {
			String name = ((ServletMapping)mappings.get(i)).getServlet().getServletName();
			if(name != null && name.equals(mapping.getName()))
				return true;
		}
		return false;
	}
	
	private static int getServletMappingIndex(EList mappings, String name) {
		
		for(int i=0; i<mappings.size(); i++) {
			String mName = ((ServletMapping)mappings.get(i)).getServlet().getServletName();
			if(mName != null && mName.equals(name))
				return i;
		}
		return -1;
	}
	
	/**
	 * Copies the directory tree at sourceDir to the targetPath in the web project.  Files that already exist will not be overwritten.
	 * 
	 * @param sourceDir
	 * @param targetPath
	 */
	public static void copyTree(File sourceDir, IPath targetPath, IWorkspaceRoot workspaceRoot) {
		
		if(sourceDir == null || targetPath == null || workspaceRoot == null)
			return;
		try {
			if(sourceDir.exists()) { // && sourceDir.isDirectory()) {
				File[] srcFiles;
				if(sourceDir.isDirectory()) {
					srcFiles = sourceDir.listFiles();
				} else {
					srcFiles = new File[] { sourceDir };
				}

				// do the copy
				for (int i = 0; i < srcFiles.length; i++) {
					File srcFile = srcFiles[i];
					if(srcFile.isFile()) {
						IFile copyFile = workspaceRoot.getFile(targetPath.append(srcFile.getName()));
						FileInputStream sourceStream = new FileInputStream(srcFile);
						if (!(copyFile.exists())) {
							copyFile.create(sourceStream, true, null);
						} else {
							copyFile.setContents(sourceStream, true, false, null);
						}
						sourceStream.close();
					} else if(srcFile.isDirectory()) {
						IFolder packageDirFolder = workspaceRoot.getFolder(targetPath.append(srcFile.getName()));
						if(!packageDirFolder.exists()) {
							packageDirFolder.create(true, true, null);
						}
						
						ArtifactHandlerUtil.copyTree(srcFile, targetPath.append(srcFile.getName()), workspaceRoot);
					}
				}			
			}			
		} catch (IOException ioe) {
			CorePlugin.log(ioe);
		} catch (CoreException ce) {
			CorePlugin.log(ce);
		}
	}
			
}
