/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.runtime.installer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.atf.adapter.IWebResourceLocator;
import org.eclipse.atf.core.CorePlugin;
import org.eclipse.atf.natures.ArtifactHandlerUtil;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jst.j2ee.common.EnvEntryType;
import org.eclipse.wst.common.componentcore.ComponentCore;
import org.eclipse.wst.common.componentcore.resources.IVirtualComponent;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;


public class WARRuntimeInstaller implements IRuntimeInstaller{
	
	
	private static final int ADDVALUES = 0;
	private static final int REMOVEVALUES = 1;
	/* Overwrite this in extended classes depending on J2EE config */
	private static String[] EXCLUDE_LIST = {"WEB-INF","META-INF","web.xml","MANIFEST.MF"};
	
	public void install(IRuntimeInstance runtimeInstance, IProject project, IProgressMonitor monitor) throws CoreException{
		monitor.beginTask( "Handler installing runtime " +runtimeInstance.getType().getName() + " into project " + project.getName() + "...", 100 );
		IWebResourceLocator adapter = (IWebResourceLocator)project.getAdapter( IWebResourceLocator.class );
		if( adapter == null ) return;
		IContainer runtimeInstallRoot = adapter.getWebResourceContainer();
		try {
			
			//in the case that the WebContent directory is not yet created
			if(!runtimeInstallRoot.exists()){
				project.getFolder( runtimeInstallRoot.getProjectRelativePath() ).create(true, true, null);
			}
			
			//resolving the URL set as the RuntimeInstance location to a Direcory somewhere in the system
			//current support for file: URLs and eclipse: URLs
			URL instanceLocationURL = new URL( runtimeInstance.getLocation());
			if( !"file".equalsIgnoreCase(instanceLocationURL.getProtocol()) ){
				//have the Platform resolve the URL to a local file url
				instanceLocationURL = FileLocator.toFileURL( instanceLocationURL );
			}
			monitor.worked( 10 );
			
			SubProgressMonitor sub = new SubProgressMonitor( monitor, 90 );
			sub.setTaskName( "Copying runtime assets..." );
			mergeTree( new File(instanceLocationURL.getPath()), runtimeInstallRoot, sub );
			IVirtualComponent componentHandle = ComponentCore.createComponent(project);
			mergeDeploymentDescriptors(new File(instanceLocationURL.getPath() +
									   System.getProperty("file.separator") + 
									   "WEB-INF"+ 
									   System.getProperty("file.separator") + 
									   "web.xml"),componentHandle);
		} catch (MalformedURLException me) {
			me.printStackTrace();
			throw new CoreException( new Status(
					IStatus.ERROR,
					CorePlugin.PLUGIN_ID,
					IStatus.OK,
					"Error installing runtime instance! Location <"+runtimeInstance.getLocation()+"> is not a valid URL.", me) );
		} catch (IOException ioe) {
			ioe.printStackTrace();
			throw new CoreException( new Status(
					IStatus.ERROR,
					CorePlugin.PLUGIN_ID,
					IStatus.OK,
					"Error installing runtime instance! Location <"+runtimeInstance.getLocation()+"> is not a valid platform URL.", ioe) );
		}
		finally{
			monitor.done();
		}			
	}
	public void uninstall(IRuntimeInstance runtimeInstance, IProject project, IProgressMonitor monitor) throws CoreException{
		
		//need to determine the correct location to install based in the type of project
		IWebResourceLocator adapter = (IWebResourceLocator)project.getAdapter( IWebResourceLocator.class );
		
		if( adapter == null ) return;
			
		monitor.beginTask( "Handler uninstalling runtime " +runtimeInstance.getType().getName() + " from project " + project.getName() + "...", 100 );
			
		//find the destionation folder within the project to install runtime
		//IContainer runtimeInstallRoot = adapter.getWebResourceContainer();
		//IFolder instanceInstallRoot = runtimeInstallRoot.getFolder( new Path(getRuntimeRootDirName()) );
		if( adapter == null ) return;
		IContainer runtimeInstallRoot = adapter.getWebResourceContainer();
		IPath installDestinationPath = runtimeInstallRoot.getFullPath();	
		monitor.worked( 10 );
			
			try {
				URL instanceLocationURL = new URL( runtimeInstance.getLocation());
			//	IFolder installDestionationFolder = project.getWorkspace().getRoot().getFolder( installDestinationPath );
				
				if( !"file".equalsIgnoreCase(instanceLocationURL.getProtocol()) ){
					//have the Platform resolve the URL to a local file url
					instanceLocationURL = FileLocator.toFileURL( instanceLocationURL );
				}
				SubProgressMonitor sub = new SubProgressMonitor( monitor, 80 );
				IVirtualComponent componentHandle = ComponentCore.createComponent(project);
				uninstallDeploymentDescriptors(new File(instanceLocationURL.getPath() + 
						   System.getProperty("file.separator") + 
						   "WEB-INF"+ 
						   System.getProperty("file.separator") + 
						   "web.xml"),componentHandle);
				sub.setTaskName( "Deleting runtime assets..." );
				removeRuntimeTree( new File(instanceLocationURL.getPath()), installDestinationPath, project.getWorkspace().getRoot(), sub );
				
				sub = new SubProgressMonitor( monitor, 10 );
				sub.setTaskName( "Refresing workspace..." );
				runtimeInstallRoot.refreshLocal( IResource.DEPTH_ZERO, sub );
			}catch(IOException ex){
				CorePlugin.log(ex);
			}finally{
				monitor.done();
			}
			
			
		
	}
	
	protected void removeRuntimeTree(File sourceDir, IPath targetPath, IWorkspaceRoot workspaceRoot, IProgressMonitor monitor ) {
		if(sourceDir == null || workspaceRoot == null){
			monitor.done();
			return;
		}
		try {
			if(sourceDir.exists()) { // && sourceDir.isDirectory()) {
				File[] srcFiles;
				if(sourceDir.isDirectory()) {
					srcFiles = sourceDir.listFiles();
				} else {
					srcFiles = new File[] { sourceDir };
				}

				monitor.beginTask( "Deleting files from runtime" + sourceDir.getName() + "...", srcFiles.length );
				// do the delete
				for (int i = 0; i < srcFiles.length; i++) {
					File srcFile = srcFiles[i];
					boolean isWebDescriptor = srcFile.getName().toLowerCase().equals("web.xml");
					if(srcFile.isFile() && !isWebDescriptor) {	
						monitor.subTask( "Deleting file " + srcFile.getName() + "..." );
						if(! shouldExclude(srcFile.getName()))
							workspaceRoot.getFile(targetPath.append(srcFile.getName())).delete(true, monitor);
					}else if(srcFile.isDirectory() ) {
						SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
						removeRuntimeTree(srcFile, targetPath.append(srcFile.getName()), workspaceRoot, sub );
						/* Code would go here to remove empty directories */
						IFolder tf = workspaceRoot.getFolder(targetPath.append(srcFile.getName()));
						/* Delete but don't force */
						if(! shouldExclude(tf.getName())) tf.delete(false, sub);						
					}
				}			
			}
		} catch (CoreException ce) {
			CorePlugin.log(ce);
		}
		finally{
			monitor.done();
		}
	}	
    protected void mergeTree(File sourceDir, IContainer targetContainer, IProgressMonitor monitor ) {
		
		if(sourceDir == null || targetContainer == null ){
			monitor.done();
			return;
		}
		
		try {
			if(sourceDir.exists()) { // && sourceDir.isDirectory()) {
				File[] srcFiles;
				if(sourceDir.isDirectory()) {
					srcFiles = sourceDir.listFiles();
				} else {
					srcFiles = new File[] { sourceDir };
				}

				monitor.beginTask( "Copying files from " + sourceDir.getName() + "...", srcFiles.length );
				// do the copy
				for (int i = 0; i < srcFiles.length; i++) {
					File srcFile = srcFiles[i];
					if(srcFile.isFile()) {
						
						copyFile(srcFile, targetContainer, monitor);
						
					} else if(srcFile.isDirectory()) {
						
						IFolder packageDirFolder = targetContainer.getProject().getFolder( targetContainer.getProjectRelativePath().append(srcFile.getName()) );
						//IFolder packageDirFolder = workspaceRoot.getFolder(targetPath.append(srcFile.getName()));
						
						if(!packageDirFolder.exists()) {
							packageDirFolder.create(true, true, new NullProgressMonitor() );
						}
						
						SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
						mergeTree(srcFile, packageDirFolder, sub );
						
					}
				}			
			}			
		} catch (IOException ioe) {
			CorePlugin.log(ioe);
		} catch (CoreException ce) {
			CorePlugin.log(ce);
		}
		finally{
			monitor.done();
		}
	}
	protected void copyFile(File srcFile, IContainer targetContainer, IProgressMonitor monitor) throws FileNotFoundException, CoreException, IOException {
		monitor.subTask( "Copying file " + srcFile.getName() + "..." );
		
		IFile copyFile = targetContainer.getProject().getFile( targetContainer.getProjectRelativePath().append(srcFile.getName()) );
			
		FileInputStream sourceStream = new FileInputStream(srcFile);
		
		SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
		sub.setTaskName( "Creating file in project..." );
		boolean isWebDescriptor = copyFile.getName().toLowerCase().equals("web.xml");
		if (!(copyFile.exists()) && ! isWebDescriptor     ) {
			copyFile.create(sourceStream, true, sub);
		} else if(!isWebDescriptor){
			copyFile.setContents(sourceStream, true, false, sub);		
		}
		sourceStream.close();
	}
    
    protected void mergeDeploymentDescriptors(File WebDescriptorFile, IVirtualComponent component) throws  CoreException{
    	DocumentBuilder builder = null;
    	try{
    		builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
    	}catch(Exception ex){ return; }
    	
    	Document document=null;
    	
    	try{
    		document = builder.parse( new FileInputStream(WebDescriptorFile));
    	}catch(Exception ex){ return; } 
    	
    	updateServlets(document, component,  ADDVALUES);
    	updateContextParams(document, component,  ADDVALUES);
    	updateServletMapping(document, component,  ADDVALUES);
    	updateEnvEntries(document, component,  ADDVALUES); 	
    }
    
    protected void uninstallDeploymentDescriptors(File WebDescriptorFile, IVirtualComponent component) throws  CoreException{
    	DocumentBuilder builder = null;
    	try{
    		builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
    	}catch(Exception ex){ return; }
    	
    	Document document=null;
    	
    	try{
    		document = builder.parse( new FileInputStream(WebDescriptorFile));
    	}catch(Exception ex){ return; } 
    	
    	updateServlets(document, component,  REMOVEVALUES);
    	updateContextParams(document, component,  REMOVEVALUES);
    	updateServletMapping(document, component,  REMOVEVALUES);
    	updateEnvEntries(document, component,  REMOVEVALUES); 	
    }
    
    private void updateServlets(Document document, IVirtualComponent component, int action) throws CoreException{
      	/* Retrieve servlet */
    	Element root = document.getDocumentElement();

    	NodeList servletNodes = root.getElementsByTagName("servlet");
    	// Loops through servlets
    	for(int i = 0; i<servletNodes.getLength();i++){ 		
    		NodeList s  = servletNodes.item(i).getChildNodes();
    		String nName = null;
    		String nClass = null;
    		for(int k = 0;k<s.getLength();k++){
    			// Loops through servlets nodes (servlet-name, servlet-class)
    			Node t = s.item(k);
    			if(t.getNodeName().equalsIgnoreCase("servlet-name"))
    				nName = ((Text)t.getFirstChild()).getData().trim();
    			if(t.getNodeName().equalsIgnoreCase("servlet-class"))
    				nClass = ((Text)t.getFirstChild()).getData().trim();	
    		}
    		if(nClass!=null && nName!=null ){	
    			switch(action){
    				case ADDVALUES:
    					ArtifactHandlerUtil.addServlet(component,nName,nClass);
    					break;
    				case REMOVEVALUES:
    					ArtifactHandlerUtil.removeServlet(component,nName);
    					break;	
    			}
    		}
    	}	
    	
    }
    private void updateServletMapping(Document document, IVirtualComponent component, int action) throws CoreException{
      	/* Retrieve servlet */
    	Element root = document.getDocumentElement();

    	NodeList servletMapNodes = root.getElementsByTagName("servlet-mapping");
    	// Loops through servlets
    	for(int i = 0; i<servletMapNodes.getLength();i++){ 		
    		NodeList s  = servletMapNodes.item(i).getChildNodes();
    		String nName = null;
    		String nURLPatern = null;
    		for(int k = 0;k<s.getLength();k++){
    			// Loops through servlets nodes (servlet-name, servlet-class)
    			Node t = s.item(k);
    			if(t.getNodeName().equalsIgnoreCase("servlet-name"))
    				nName = ((Text)t.getFirstChild()).getData().trim();
    			if(t.getNodeName().equalsIgnoreCase("url-pattern"))
    				nURLPatern = ((Text)t.getFirstChild()).getData().trim();	
    		}
    		if(nURLPatern!=null && nName!=null ){	
    			switch(action){
    				case ADDVALUES:
    					ArtifactHandlerUtil.addServletMapping(component,nName,nURLPatern);
    					break;
    				case REMOVEVALUES:
    					ArtifactHandlerUtil.removeServletMapping(component,nName);
    					break;	
    			}
    		}
    	}	
    	
    }
    private void updateContextParams(Document document, IVirtualComponent component, int action) throws CoreException{
      	/* Retrieve servlet */
    	Element root = document.getDocumentElement();

    	NodeList contextParamsNodes = root.getElementsByTagName("context-param");
    	// Loops through servlets
    	for(int i = 0; i<contextParamsNodes.getLength();i++){ 		
    		NodeList s  = contextParamsNodes.item(i).getChildNodes();
    		String nName = null;
    		String nValue = null;
    		for(int k = 0;k<s.getLength();k++){
    			// Loops through servlets nodes (servlet-name, servlet-class)
    			Node t = s.item(k);
    			if(t.getNodeName().equalsIgnoreCase("param-name"))
    				nName = ((Text)t.getFirstChild()).getData().trim();
    			if(t.getNodeName().equalsIgnoreCase("param-value"))
    				nValue = ((Text)t.getFirstChild()).getData().trim();	
    		}
    		if(nValue!=null && nName!=null ){	
    			switch(action){
    				case ADDVALUES:
    					ArtifactHandlerUtil.addContextParam(component,nName,nValue);
    					break;
    				case REMOVEVALUES:
    					ArtifactHandlerUtil.removeContextParam(component,nName);
    					break;	
    			}
    		}
    	}	
    } 
    
    private void updateEnvEntries(Document document, IVirtualComponent component, int action) throws CoreException{
      	/* Retrieve servlet */
    	Element root = document.getDocumentElement();

    	NodeList envEntriesNodes = root.getElementsByTagName("env-entry");
    	// Loops through servlets
    	for(int i = 0; i<envEntriesNodes.getLength();i++){ 		
    		NodeList s  = envEntriesNodes.item(i).getChildNodes();
    		String nName = null;
    		String nValue = null;
    		EnvEntryType nType = null;
    		for(int k = 0;k<s.getLength();k++){
    			// Loops through servlets nodes (servlet-name, servlet-class)
    			Node t = s.item(k);
    			if(t.getNodeName().equalsIgnoreCase("env-entry-name"))
    				nName = ((Text)t.getFirstChild()).getData().trim();
    			if(t.getNodeName().equalsIgnoreCase("env-entry-value"))
    				nValue = ((Text)t.getFirstChild()).getData().trim();	
    			if(t.getNodeName().equalsIgnoreCase("env-entry-type")){
    				String nt = (((Text)t.getFirstChild()).getData()).toUpperCase();
    				/* Need to use this data to determine the EnvEntryType */
	    			if(nt.indexOf("STRING") >= 0) nType = EnvEntryType.STRING_LITERAL;
	    			if(nt.indexOf("INTEGER") >= 0) nType = EnvEntryType.INTEGER_LITERAL;
	    			if(nt.indexOf("BOOLEAN") >= 0) nType = EnvEntryType.BOOLEAN_LITERAL;
	    			if(nt.indexOf("DOUBLE") >= 0) nType = EnvEntryType.DOUBLE_LITERAL;
	    			if(nt.indexOf("BYTE") >= 0) nType = EnvEntryType.BYTE_LITERAL;
	    			if(nt.indexOf("SHORT") >= 0) nType = EnvEntryType.SHORT_LITERAL;
	    			if(nt.indexOf("LONG") >= 0) nType = EnvEntryType.LONG_LITERAL;
	    			if(nt.indexOf("FLOAT") >= 0) nType = EnvEntryType.FLOAT_LITERAL;
	    			if(nt.indexOf("CHARACTER") >= 0) nType = EnvEntryType.CHARACTER_LITERAL;
    			}
    		}
    		if(nValue!=null && nName!=null ){	
    			switch(action){
    				case ADDVALUES:
    					ArtifactHandlerUtil.addEnvEntry(component,nName,nType,nValue);
    					break;
    				case REMOVEVALUES:
    					ArtifactHandlerUtil.removeEnvEntry(component,nName);
    					break;	
    			}
    		}
    	}	
    	
    } 
    
    private boolean shouldExclude(String fileName){
    	for(int i = 0;i<EXCLUDE_LIST.length;i++){
    		if(EXCLUDE_LIST[i].equalsIgnoreCase(fileName)) return true;	
    	}
   
    	return false; 	
    }
    
	public IPath getProjectInstallPath( IRuntimeInstance runtimeInstance )
	{
		throw new RuntimeException( "not implemented" );
	}
}
