/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.natures;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;

public class ATFNature implements IProjectNature
{
	public final static String ATF_NATURE_ID = "org.eclipse.atf.core.ATFNature";
	protected final static String RUNTIME_PROP = "runtime";

	private static final String PLUGIN_POINT = "jsdtClasspathInitializer";
	private static final Object TAG_DELEGATE = "delegate";

	private IProject project;

	public void configure() throws CoreException
	{
		IClasspathInitializerDelegate delegate = classpathInitDelegate();
		if ( delegate != null )
			delegate.configure( project );
	}

	public void deconfigure() throws CoreException
	{
		IClasspathInitializerDelegate delegate = classpathInitDelegate();
		if ( delegate != null )
			delegate.deconfigure( project );
	}

	private IClasspathInitializerDelegate classpathInitDelegate()
	{
		// see if a JSDT classpath initializer is present
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint( CorePlugin.PLUGIN_ID, PLUGIN_POINT );
		if ( point == null )
			return null;
		
		IExtension[] extensions = point.getExtensions();
		if ( extensions.length == 0 )
			return null;
		
		IConfigurationElement[] elements = extensions[ 0 ].getConfigurationElements();
		for ( int i = 0; i < elements.length; i++ )
		{
			if ( elements[ i ].getName().equals( TAG_DELEGATE ) )
			{
				try
				{
					return (IClasspathInitializerDelegate) elements[ i ].createExecutableExtension( "class" );
				}
				catch ( CoreException e )
				{
					// failed to get initializer delegate class
					return null;
				}
			}
		}
		
		return null;
	}

	public IProject getProject()
	{
		return project;
	}

	public void setProject( IProject project )
	{
		this.project = project;
	}

	public static void addNature( IProject project, IProgressMonitor monitor )
			throws CoreException
	{
		if ( project.hasNature( ATF_NATURE_ID ) )
			return;

		IProjectDescription description = project.getDescription();
		String[] prevNatures = description.getNatureIds();
		String[] newNatures = new String[ prevNatures.length + 1 ];
		System.arraycopy( prevNatures, 0, newNatures, 0, prevNatures.length );
		newNatures[ prevNatures.length ] = ATF_NATURE_ID;
		description.setNatureIds( newNatures );

		try
		{
			project.setDescription( description, monitor );
		}
		catch ( CoreException e )
		{
			// restore to previous configuration. Eclipse is not rolling back
			// when an
			// error ocurrs when adding a nature. The memory list of natures
			// will contain
			// natures that may have caused this exception when configuring.
			description.setNatureIds( prevNatures );
			project.setDescription( description, monitor );

			throw e;
		}
	}

	public static void removeNature( IProject project, IProgressMonitor monitor )
			throws CoreException
	{
		if ( !project.hasNature( ATF_NATURE_ID ) )
			return;

		IProjectDescription description = project.getDescription();
		String[] prevNatures = description.getNatureIds();
		String[] newNatures = new String[ prevNatures.length - 1 ];

		int index = 0;
		for ( int i = 0; i < prevNatures.length; i++ )
		{
			if ( prevNatures[ i ].equals( ATF_NATURE_ID ) )
				continue;

			newNatures[ index ] = prevNatures[ i ];
			index++;
		}

		description.setNatureIds( newNatures );
		project.setDescription( description, monitor );
	}
}
