/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.natures;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;



public class NatureUtil {

	public static void removeNature(IProject project, String natureId, IProgressMonitor monitor) throws CoreException {
		
		if (project.hasNature(natureId)) {
			IProjectDescription description = project.getDescription();
			String[] prevNatures= description.getNatureIds();
			String[] newNatures= new String[prevNatures.length - 1];
			int index = 0;
			for(int i=0; i<prevNatures.length; i++) {
				if(prevNatures[i].equals(natureId))
					continue;
				newNatures[index] = prevNatures[i];
				index++;
			}
			description.setNatureIds(newNatures);
			project.setDescription(description, monitor);
		} 
	}
	
	public static void addNature(IProject project, String natureId, IProgressMonitor monitor) throws CoreException {
		if (!project.hasNature(natureId)) {
			IProjectDescription description = project.getDescription();
			String[] prevNatures= description.getNatureIds();
			String[] newNatures= new String[prevNatures.length + 1];
			System.arraycopy(prevNatures, 0, newNatures, 0, prevNatures.length);
			newNatures[prevNatures.length]= natureId;
			description.setNatureIds(newNatures);
			
			try{
				project.setDescription(description, monitor);
			}
			catch( CoreException e ){
				//restore to previous configuration. Eclipse is not rolling back when an
				//error ocurrs when adding a nature. The memory list of natures will contain
				//natures that may have caused this exception when configuring.
				description.setNatureIds(prevNatures);
				project.setDescription( description, monitor );
				
				throw e;
			}
			
		} 
	}

	
	public static void addBuilder(IProject project, String builderId) throws CoreException {
		IProjectDescription description = project.getDescription();
		int builderIndex = getBuilderIndex(description.getBuildSpec(), builderId);

		if (builderIndex == -1) {
			// Add a new command to the build spec
			ICommand command = description.newCommand();
			command.setBuilderName(builderId);
			ICommand[] oldBuildSpec = description.getBuildSpec();
			int oldImageBuilderIndex = getBuilderIndex(oldBuildSpec, builderId);
			ICommand[] newCommands;

			if (oldImageBuilderIndex == -1) {
				// Add a new build spec
				newCommands = new ICommand[oldBuildSpec.length + 1];
				System.arraycopy(oldBuildSpec, 0, newCommands, 0, oldBuildSpec.length);
				newCommands[newCommands.length - 1] = command;
			} else {
			    oldBuildSpec[oldImageBuilderIndex] = command;
				newCommands = oldBuildSpec;
			}
			// Commit the spec change into the project
			description.setBuildSpec(newCommands);
			project.setDescription(description, null);
		}
	}
	
	public static void removeBuilder(IProject project, String builderId) throws CoreException {
		
		IProjectDescription description = project.getDescription();
		int builderIndex = getBuilderIndex(description.getBuildSpec(), builderId);

		if (builderIndex != -1) {
			//Remove the command from the build spec
			ICommand[] buildSpecs = description.getBuildSpec();
			ICommand[] newBuildSpecs = new ICommand[buildSpecs.length - 1];
			int newIndex = 0;
			for(int i=0; i<buildSpecs.length; i++) {
				if(i == builderIndex)
					continue;
				newBuildSpecs[newIndex] = buildSpecs[i];
				newIndex++;
			}
			// Commit the spec change into the project
			description.setBuildSpec(newBuildSpecs);
			project.setDescription(description, null);
		}
	}
	
	protected static int getBuilderIndex(ICommand[] buildSpec, String builderId) {
		for (int i = 0; i < buildSpec.length; ++i) {
			if (buildSpec[i].getBuilderName().equals(builderId)) {
				return i;
			}
		}
		return -1;
	}
	
	protected static void setBuilderCommand(IProject project, IProjectDescription description, ICommand newCommand, String builderId) throws CoreException {
		
		ICommand[] oldBuildSpec = description.getBuildSpec();
		int oldImageBuilderIndex = getBuilderIndex(oldBuildSpec, builderId);
		ICommand[] newCommands;

		if (oldImageBuilderIndex == -1) {
			// Add a new build spec
			newCommands = new ICommand[oldBuildSpec.length + 1];
			System.arraycopy(oldBuildSpec, 0, newCommands, 0, oldBuildSpec.length);
			newCommands[newCommands.length - 1] = newCommand;
		} else {
		    oldBuildSpec[oldImageBuilderIndex] = newCommand;
			newCommands = oldBuildSpec;
		}
		// Commit the spec change into the project
		description.setBuildSpec(newCommands);
		project.setDescription(description, null);
	}
	
}
