/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.project;

import java.util.ArrayList;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.wst.common.componentcore.ComponentCore;
import org.eclipse.wst.common.componentcore.internal.StructureEdit;
import org.eclipse.wst.common.componentcore.internal.WorkbenchComponent;
import org.eclipse.wst.common.componentcore.internal.util.ComponentUtilities;
import org.eclipse.wst.common.componentcore.resources.IVirtualComponent;
import org.eclipse.wst.common.componentcore.resources.IVirtualFolder;


public class FlexibleProjectUtils {

	private static final String[] J2EE_PROJECT_NATURE_IDS = {"org.eclipse.wst.common.project.facet.core.nature",
																"org.eclipse.jdt.core.javanature",
																"org.eclipse.wst.common.modulecore.ModuleCoreNature",
																"org.eclipse.jem.workbench.JavaEMFNature"};
	public static final String PHP_NATURE_ID =  "org.eclipse.php.core.PHPNature";
	
	public static String[] getModuleNames(IProject project) throws CoreException {
		if(project.getNature("org.eclipse.wst.common.modulecore.ModuleCoreNature") == null)
			return null;
		StructureEdit moduleCore = StructureEdit.getStructureEditForRead(project);
		ArrayList list = new ArrayList();
		try {
			WorkbenchComponent[] components = moduleCore.getWorkbenchModules();
			
			for(int i=0; i<components.length; i++) {
				list.add(components[i].getName());
			}
		} finally {
			moduleCore.dispose();
		}
		
		String[] modules = new String[list.size()];
		modules = (String[])list.toArray(modules);
		return modules;
		
	}
	
	public static String[] getModuleNames(String projectName) throws CoreException {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		if(project == null)
			return null;
		if(project.getNature("org.eclipse.wst.common.modulecore.ModuleCoreNature") == null)
			return null;
		StructureEdit moduleCore = StructureEdit.getStructureEditForRead(project);
		ArrayList list = new ArrayList();
		try {
			WorkbenchComponent[] components = moduleCore.getWorkbenchModules();
			for(int i=0; i<components.length; i++) {
				list.add(components[i].getName());
			}
		} finally {
			moduleCore.dispose();
		}
		
		String[] modules = new String[list.size()];
		modules = (String[])list.toArray(modules);
		return modules;
	}
	
	public static WorkbenchComponent[] getModules(IProject project) throws CoreException {
		if(project.getNature("org.eclipse.wst.common.modulecore.ModuleCoreNature") == null)
			return null;
		StructureEdit moduleCore = StructureEdit.getStructureEditForRead(project);
		ArrayList list = new ArrayList();
		try {
			WorkbenchComponent[] components = moduleCore.getWorkbenchModules();
			for(int i=0; i<components.length; i++) {
				list.add(components[i].getName());
			}
		} finally {
			moduleCore.dispose();
		}
		
		WorkbenchComponent[] modules = new WorkbenchComponent[list.size()];
		modules = (WorkbenchComponent[])list.toArray(modules);
		return modules;
	}
		
	public static IContainer getWebContentFolder(IProject project) {
		try {
			if( project.hasNature(PHP_NATURE_ID) ) {
				return project;
			}
		} catch (CoreException e) {}
		
		IVirtualComponent comp = ComponentCore.createComponent(project);
		if(comp != null) {
			IVirtualFolder rootFolder = comp.getRootFolder();
			return rootFolder.getUnderlyingFolder();
		}
		return null;
	}
	
	/**
	 * Returns project relative WebContent sub path.
	 * @param component
	 * @return
	 */
	public static IPath getWebContentPath(IProject project, String webContentRelativePath) {
		
		IVirtualComponent comp = ComponentCore.createComponent(project);
		if(comp != null) {
			IVirtualFolder rootFolder = comp.getRootFolder();
			if(rootFolder.exists()) {
				IPath path = (rootFolder.getProjectRelativePath()).append(webContentRelativePath);
//				System.out.println("WebContent: " + path.toString());
				return path;
			} 
		} 
		return project.getFolder(webContentRelativePath).getProjectRelativePath();
		
	}
	
	public static String getWebContentRelativePath(IResource resource) {
		
		if(!isResourceInWebContentDir(resource))
			return null;
		
		IVirtualComponent comp = ComponentCore.createComponent(resource.getProject());
		if(comp != null) {
			IVirtualFolder rootFolder = comp.getRootFolder();
			if(rootFolder.exists()) {
				return  resource.getProjectRelativePath().removeFirstSegments(rootFolder.getProjectRelativePath().segmentCount()).makeRelative().toString();
			}
		} else {
			return resource.getProjectRelativePath().makeRelative().toString();
		}
		return null;
		
	}
		
	public static ArrayList getFilePathsForType(IResource resource, String extension, ArrayList paths) throws CoreException {
		
		if(!(resource instanceof IContainer))
			return paths;
		IContainer folder = (IContainer)resource;
		IResource[] members = folder.members();
		for(int i=0; i<members.length; i++) {
			IResource member = members[i];
			if(member instanceof IContainer) {
				getFilePathsForType(member, extension, paths);
			}
			else if(member instanceof IFile && ((IFile)member).getFileExtension().equals(extension)){
				paths.add(member.getFullPath());
			}
		}
		return paths;
	}
	
	public static boolean isResourceInWebContentDir(IResource resource) {
		
		IVirtualComponent comp = ComponentCore.createComponent(resource.getProject());
		if(comp != null) {
			IVirtualFolder rootFolder = comp.getRootFolder();
			if(resource.getProjectRelativePath().matchingFirstSegments(rootFolder.getProjectRelativePath()) == rootFolder.getProjectRelativePath().segmentCount())
				return true;
		} else
			try {
				if( resource.getProject().hasNature(PHP_NATURE_ID) ) {
					return true;
				}
			} catch (CoreException e) {}
		return false;
				
	}
	
	public static boolean isPathInSelectedFolderTree(IFolder folder, IPath path) {
			
		IPath webContentRelFolder;
		IPath webContentRelPath = path.removeFirstSegments(2);
		
		webContentRelFolder = folder.getProjectRelativePath().removeFirstSegments(1);
		if(webContentRelFolder.isPrefixOf(webContentRelPath))
			return true;
		return false;
								
	}
	
	/**
	 * Returns boolean indicating whether the specified IContainer is inside
	 * the WebContent directory.
	 * 
	 * @param container
	 * @param project
	 * @return
	 */
	public static boolean isContainerInsideWebContent(IContainer container) {
		
		IVirtualComponent comp = ComponentCore.createComponent(container.getProject());
		IVirtualFolder rootFolder = comp.getRootFolder();
		if(container.getProjectRelativePath().matchingFirstSegments(rootFolder.getProjectRelativePath()) == rootFolder.getProjectRelativePath().segmentCount())
			return true;
		return false;
			
	}
	
	
	public static String getModuleName(IFile file) {
		
		return file.getProject().getName();
		
	}
	
	public static String getModuleName(String project, IPath projRelativePath) {
		if(project == null || project.equals(""))
			return null;
		IProject p = ResourcesPlugin.getWorkspace().getRoot().getProject(project);
		if(p == null)
			return null;
		return p.getName();
		
	}
	
	public static String getWebAppPath(IFile file) {
		String module = getModuleName(file);
		if(module == null)
			return null;
		String webContentRelPath = getWebContentRelativePath(file);
		if(webContentRelPath == null)
			return null;
		return "/" + module + "/" + webContentRelPath;
	}
	
	public static String getWebAppPath(String project, IPath projRelativePath) {
		String module = getModuleName(project, projRelativePath);
		if(module == null)
			return null;
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getProject(project).getFile(projRelativePath);
		String webContentRelPath = getWebContentRelativePath(file);
		if(webContentRelPath == null)
			return null;
		return "/" + module + "/" + webContentRelPath;
	}
	
	public static String getProjectRelativePath(String projectName, String moduleName, IPath webContentRelPath) {
		if(projectName == null || moduleName == null || moduleName.equals(""))
			return null;
		IProject p = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		if(p == null)
			return null;
		IContainer wcf = getWebContentFolder(p);
		String projectRelPath = wcf.getProjectRelativePath().makeRelative().toString();
		if( projectRelPath.equals("") ) {
			return webContentRelPath.makeRelative().toString();
		} else {
			return projectRelPath+"/"+ webContentRelPath.makeRelative().toString();
		}
		
	}
	
	public static boolean isJ2EEProject(IProject project) throws CoreException {
		
		for(int i=0; i<J2EE_PROJECT_NATURE_IDS.length; i++) {
			if(!project.hasNature(J2EE_PROJECT_NATURE_IDS[i]))
				return false;
		}
		return true;
	}
	
	public static String getServerContextRoot(IProject project) throws CoreException {
		String contextRoot = null;
		if( project.hasNature(PHP_NATURE_ID) ) {
			contextRoot =  project.getName();
		} else {
			contextRoot = ComponentUtilities.getServerContextRoot(project);
		}
		return contextRoot;
	}
}
