/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.runtime;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.atf.runtime.installer.FileCopyRuntimeInstaller;
import org.eclipse.atf.runtime.installer.IRuntimeInstaller;
import org.eclipse.atf.runtime.validator.IRuntimeValidator;
import org.eclipse.atf.runtime.version.IVersionFinder;

public class Runtime implements IRuntime {

	/*
	 * Unique identification for the Runtime
	 */
	protected String id = null;
	
	/*
	 * Display name for the Runtime
	 */
	protected String name = null;
	
	/*
	 * Strategy used to install the RuntimeInstance into a project
	 */
	protected IRuntimeInstaller installer = null;
	
	/*
	 * Strategy used to validate the RuntimeInstance location
	 */
	protected IRuntimeValidator validator = null;
	
	/*
	 * Strategy used to inspect the version from an instance of this Runtime
	 */
	protected IVersionFinder versionFinder = null;
	
	/*
	 * Determines if the users are allowed to add new instances of the runtime
	 * 
	 * default is true
	 */
	protected boolean allowUserInstances = true;
	
	/*
	 * Determines if the user created this IRuntime
	 * 
	 * default is false
	 */
	protected boolean isUser = false;
	
	/*
	 * List of all the Instances available for the Runtime.
	 * 
	 * Each instance represents a different copy of the runtime
	 * as it exists either in a plugin or in the file system.
	 * (could extend to network locations)
	 */
	protected List instanceList = new ArrayList(2);
	
	/**
	 * Creates an instance of Runtime which assumes simple File copy
	 * installation.
	 *
	 * @param id - fully qualified id based on the plugin extension or a generated string or user defined
	 * @param name - pretty name use for display
	 */
	public Runtime( String id, String name ){
		this.id = id;
		this.name = name;
		
		this.installer = new FileCopyRuntimeInstaller();
		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getId()
	 */
	public String getId() {
		return id;
	}
	
	public void setId( String id ){
		if( isUser )
			this.id = id;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getName()
	 */
	public String getName() {
		return name;
	}
	
	public void setName( String name ){
		if( isUser )
			this.name = name;
	}
	
	public void addInstance( RuntimeInstance instance ){
		instance.setType( this );
		instanceList.add( instance );
	}
	
	public void removeInstance( RuntimeInstance instance ){
		instanceList.remove( instance );
		
		instance.setType( null );
	}

	public IRuntimeInstance[] getRuntimeInstances() {
		return (IRuntimeInstance []) instanceList.toArray( new IRuntimeInstance[0] );
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getInstaller()
	 */
	public IRuntimeInstaller getInstaller() {
		return installer;
	}

	/*
	 * 
	 */
	public void setInstaller(IRuntimeInstaller installer) {
		this.installer = installer;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getValidator()
	 */
	public IRuntimeValidator getValidator() {
		return validator;
	}

	/*
	 * 
	 */
	public void setValidator(IRuntimeValidator validator) {
		this.validator = validator;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getVersionFinder()
	 */
	public IVersionFinder getVersionFinder() {
		return versionFinder;
	}

	/*
	 * 
	 */
	public void setVersionFinder(IVersionFinder versionFinder) {
		this.versionFinder = versionFinder;
	}

	public boolean isAllowUserInstances() {
		return allowUserInstances;
	}

	public void setAllowUserInstances(boolean allowUserInstances) {
		this.allowUserInstances = allowUserInstances;
	}
	
	public boolean isUser(){
		return isUser;
	}
	
	public void setIsUser( boolean isUser ){
		this.isUser = isUser;
	}
}
