/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.runtime.installer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.atf.adapter.IWebResourceLocator;
import org.eclipse.atf.core.CorePlugin;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;

/*
 * This class provides a simple implementation of the RuntimeHandler
 * interface. It does a simple copy of all the resources in the RuntimeInstance
 * location to the specified destination.
 * 
 * The destination path is specified through the extension point.
 * 
 * &lt;destination path="path to destionation" /&gt; - path relative to the Projects Web Content container
 * 
 */
public class FileCopyRuntimeInstaller implements IRuntimeInstaller, IExecutableExtension {

	protected IConfigurationElement config = null;
	protected URL instanceLocationURL;
	
	public void install(IRuntimeInstance runtimeInstance, IProject project, IProgressMonitor monitor) throws CoreException{
		
		monitor.beginTask( "Handler installing runtime " +runtimeInstance.getType().getName() + " into project " + project.getName() + "...", 100 );
		//need to determine the correct location to install based in the type of project
		IWebResourceLocator adapter = (IWebResourceLocator)project.getAdapter( IWebResourceLocator.class );
		if( adapter != null ){
			//find the destionation folder within the project to install runtime
			IContainer runtimeInstallRoot = adapter.getWebResourceContainer();
						
			IPath installDestinationPath = runtimeInstallRoot.getFullPath().append( getProjectInstallPath(runtimeInstance) );
			
			try {
				IFolder installDestionationFolder = project.getWorkspace().getRoot().getFolder( installDestinationPath );
				if(!installDestionationFolder.exists())
					installDestionationFolder.create(true, true, null);
						
				//resolving the URL set as the RuntimeInstance location to a Direcory somewhere in the system
				//current support for file: URLs and eclipse: URLs
				instanceLocationURL = new URL( runtimeInstance.getLocation() );
				
				if( !"file".equalsIgnoreCase(instanceLocationURL.getProtocol()) ){
					//have the Platform resolve the URL to a local file url
					instanceLocationURL = FileLocator.toFileURL( instanceLocationURL );
				}
				monitor.worked( 10 );
				
				SubProgressMonitor sub = new SubProgressMonitor( monitor, 90 );
				sub.setTaskName( "Copying runtime assets..." );
				copyTree( new File(instanceLocationURL.getPath()) , installDestinationPath, project.getWorkspace().getRoot(), sub );
			} catch (MalformedURLException me) {
				me.printStackTrace();
				throw new CoreException( new Status(
						IStatus.ERROR,
						CorePlugin.PLUGIN_ID,
						IStatus.OK,
						"Error installing runtime instance! Location <"+runtimeInstance.getLocation()+"> is not a valid URL.", me) );
			} catch (IOException ioe) {
				ioe.printStackTrace();
				throw new CoreException( new Status(
						IStatus.ERROR,
						CorePlugin.PLUGIN_ID,
						IStatus.OK,
						"Error installing runtime instance! Location <"+runtimeInstance.getLocation()+"> is not a valid platform URL.", ioe) );
			}
			finally{
				monitor.done();
			}
		}
		
	}

	public void uninstall(IRuntimeInstance runtimeInstance, IProject project, IProgressMonitor monitor) throws CoreException{
		
		//need to determine the correct location to install based in the type of project
		IWebResourceLocator adapter = (IWebResourceLocator)project.getAdapter( IWebResourceLocator.class );
		
		if( adapter != null ){
			
			monitor.beginTask( "Handler uninstalling runtime " +runtimeInstance.getType().getName() + " from project " + project.getName() + "...", 100 );
			
			//find the destionation folder within the project to install runtime
			IContainer runtimeInstallRoot = adapter.getWebResourceContainer();
			IFolder instanceInstallRoot = runtimeInstallRoot.getFolder( getProjectInstallPath(runtimeInstance) );
			
			if(!instanceInstallRoot.exists()){
				monitor.done();
				return; // no need to delete
			}
			
			monitor.worked( 10 );
			
			try {
				
				SubProgressMonitor sub = new SubProgressMonitor( monitor, 80 );
				sub.setTaskName( "Deleting runtime assets..." );
				instanceInstallRoot.delete( true, sub );
				
				sub = new SubProgressMonitor( monitor, 10 );
				sub.setTaskName( "Refresing workspace..." );
				runtimeInstallRoot.refreshLocal( IResource.DEPTH_ZERO, sub );
				
			}
			finally{
				monitor.done();
			}
			
			
		}
	}
	
	protected void copyTree(File sourceDir, IPath targetPath, IWorkspaceRoot workspaceRoot, IProgressMonitor monitor ) {
		
		if(sourceDir == null || targetPath == null || workspaceRoot == null){
			monitor.done();
			return;
		}
		
		try {
			if(sourceDir.exists()) { // && sourceDir.isDirectory()) {
				File[] srcFiles;
				if(sourceDir.isDirectory()) {
					srcFiles = sourceDir.listFiles();
				} else {
					srcFiles = new File[] { sourceDir };
				}

				monitor.beginTask( "Copying files from " + sourceDir.getName() + "...", srcFiles.length );
				// do the copy
				for (int i = 0; i < srcFiles.length; i++) {
					File srcFile = srcFiles[i];
					if(srcFile.isFile()) {
						
						monitor.subTask( "Copying file " + srcFile.getName() + "..." );
						
						IFile copyFile = workspaceRoot.getFile(targetPath.append(srcFile.getName()));
						FileInputStream sourceStream = new FileInputStream(srcFile);
						
						SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
						sub.setTaskName( "Creating file in project..." );
						if (!(copyFile.exists())) {
							copyFile.create(sourceStream, true, sub);
						} else {
							copyFile.setContents(sourceStream, true, false, sub);
						}
						sourceStream.close();
					} else if(srcFile.isDirectory()) {
						IFolder packageDirFolder = workspaceRoot.getFolder(targetPath.append(srcFile.getName()));
						
						if(!packageDirFolder.exists()) {
							packageDirFolder.create(true, true, new NullProgressMonitor() );
						}
						
						SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
						copyTree(srcFile, targetPath.append(srcFile.getName()), workspaceRoot, sub );
						
					}
				}			
			}			
		} catch (IOException ioe) {
			CorePlugin.log(ioe);
		} catch (CoreException ce) {
			CorePlugin.log(ce);
		}
		finally{
			monitor.done();
		}
	}
	
	/*
	 * This methid will try to contruct a relative path of where the RuntimeInstance assests
	 * should be copy into the project.
	 * 
	 * It first looks if there is something configured in the plugin
	 * then it uses the id. If the id is a '.' separated string, it will use the last part
	 */
	public IPath getProjectInstallPath( IRuntimeInstance runtimeInstance ){
		
		//the destination path can be configured or the last part of the Runtime id is used
		IPath configuredProjectInstallPath = getConfiguredProjectInstallPath();
		if( configuredProjectInstallPath != null )
			return configuredProjectInstallPath;

		String runtimeId = runtimeInstance.getType().getId();
		int idxLastDot = runtimeId.lastIndexOf( '.' );
		String lastIdPart = runtimeId;
		if( idxLastDot != -1 ){
			lastIdPart = runtimeId.substring( idxLastDot+1 );
		}
		
		return new Path(lastIdPart);
	}

	/*
	 * Read the destination path from the config element
	 */
	protected IPath getConfiguredProjectInstallPath(){
		
		if( config != null ){
			IConfigurationElement [] destElts = config.getChildren( "destination" );
			
			if( destElts.length > 0 ){
				IConfigurationElement destElt = destElts[0];
				return new Path( destElt.getAttribute("path") );
			}
		}
		
		return null;
	}

	/**
	 * This implementation expects the following XML under the IConfigurationElement
	 * 
	 * &lt;destination path="path to destionation" /&gt; - path relative to the Projects Web Content container
	 * 
	 * @see IExecutableExtension
	 */
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		//save the config element relevant to versioning
		this.config = config.getChildren("installer")[0];
	}
}
