/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.runtime.validator;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

/**
 * This implementation of IRuntimeValidator checks that the file path specified exist in the
 * RuntimeInstance location. The paths are specified in the extension point.
 * 
 * &lt;file path="file path to check" /&gt; - path relative to the location of the runtime that need to exist
 * 
 * @author Gino Bustelo
 *
 */
public class FileCheckRuntimeValidator implements IRuntimeValidator, IExecutableExtension {

	protected IConfigurationElement config = null;
	
	/**
	 * 
	 * This method will check all file path specified in the extension point, relative to the instance
	 * location. If any of the paths are missing, the location is determined to be invalid.
	 * 
	 */
	public IStatus validate(IRuntimeInstance instance) {
		
		//resolving to a real file url
		
		URL instanceLocationURL;
		try {
			instanceLocationURL = new URL( instance.getLocation() );
		
		} catch (MalformedURLException me) {
			IStatus status= new Status(
					IStatus.ERROR,
					CorePlugin.PLUGIN_ID,
					IStatus.ERROR,
					"RuntimeInstance location <"+instance.getLocation()+"> is not a valid URL.",
					me);
			CorePlugin.getDefault().getLog().log( status );
			return status;
		}
		
		if( !"file".equalsIgnoreCase(instanceLocationURL.getProtocol()) ){
			//have the Platform resolve the URL to a local file url
			try {
				instanceLocationURL = FileLocator.toFileURL( instanceLocationURL );
			
			} catch (IOException ioe) {
				IStatus status= new Status(
						IStatus.ERROR,
						CorePlugin.PLUGIN_ID,
						IStatus.ERROR,
						"Could not convert location URL <"+instanceLocationURL+"> into a file:// URL.",
						ioe);
				CorePlugin.getDefault().getLog().log( status );
				return status;
			}
		}
		
		/*
		 * Check that all path specified exist in the instance location
		 */
		IConfigurationElement [] fileElts = config.getChildren( "file" );
		File instanceLocation = new File( instanceLocationURL.getPath() );
		
		for (int i = 0; i < fileElts.length; i++) {
			String pathToCheck = fileElts[i].getAttribute("path");
			File fileToCheck = new File( instanceLocation, pathToCheck );
			
			/*
			 * If the file does not exist, then return the error status
			 */
			if( !fileToCheck.exists() ){
				IStatus status= new Status(
						IStatus.ERROR,
						CorePlugin.PLUGIN_ID,
						IStatus.ERROR,
						"Missing runtime file <"+fileToCheck.toString()+">.",
						null);
				return status;
				
			}
		}
		
		return Status.OK_STATUS;
	}
	
	/**
	 * This implementation expects the following XML under the IConfigurationElement
	 * 
	 * &lt;file path="file path to check" /&gt; - path relative to the location of the runtime that need to exist
	 * 
	 * @see IExecutableExtension
	 */
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		//save the config element relevant to versioning
		this.config = config.getChildren("validator")[0];
	}

}
