/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.runtime.version;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;

import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.EcmaError;
import org.mozilla.javascript.JavaScriptException;
import org.mozilla.javascript.Scriptable;

/**
 * This implementation of IVersionFinder uses Rhino to load the specified JavaScript files
 * and execute an expression that returns a version String.
 * 
 * The file(s) to load and the expression to evaluate are all provided through the 
 * extension point.
 * 
 * $lt;load file="file name" /&gt;  - is used to load a file in Rhino (There will be Errors but those are ignored)
 * $lt;eval expr="javascript expression" /&gt; - gives the JavaScript expression that when evaluated, returns the version string
 * 
 * @author Gino Bustelo
 */
public class JSVersionFinder implements IVersionFinder, IExecutableExtension {

	protected IConfigurationElement config = null;
	/**
	 * @see IVersionFinder
	 */
	public String findVersion(IRuntimeInstance instance) {

		String result = null;

		Context cx = Context.enter();
		try {
			// Set version to JavaScript1.2 so that we get object-literal style
			// printing instead of "[object Object]"
			cx.setLanguageVersion(Context.VERSION_1_6);

			// Initialize the standard objects (Object, Function, etc.)
			// This must be done before scripts can be executed.
			Scriptable scope = cx.initStandardObjects();

			//load all the files in the config and eval the expression
			IConfigurationElement[] loadConfigs = config.getChildren( "load" );
			
			//getting the root location of files
			File rootLocation = null;
			try{
				//resolving the URL set as the RuntimeInstance location to a Direcory somewhere in the system
				//current support for file: URLs and eclipse: URLs
				URL instanceLocationURL = new URL( instance.getLocation() );
				
				if( !"file".equalsIgnoreCase(instanceLocationURL.getProtocol()) ){
					//have the Platform resolve the URL to a local file url
					instanceLocationURL = FileLocator.toFileURL( instanceLocationURL );
				}
				rootLocation = new File(instanceLocationURL.getPath());
			}
			catch( Exception e ){
				e.printStackTrace();
				return null;
			}
			
			//loading the files
			for (int i = 0; i < loadConfigs.length; i++) {
				String filename = loadConfigs[i].getAttribute("file");
				File fileToLoad = new File( rootLocation, filename );
				System.out.println( "Loading <"+fileToLoad.getAbsolutePath()+">" );
								
				FileReader in = null;
				try {
					in = new FileReader(fileToLoad);
					
					//compile the file in
					cx.evaluateReader( scope, in, filename, 1, null );
					
					
				} catch (FileNotFoundException ex) {
					ex.printStackTrace();
					return null;
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
					return null;
				}
				catch( EcmaError error ){
					//ignore error
					error.printStackTrace();
				}
				catch( JavaScriptException jse ){
					jse.printStackTrace();
				}				
				finally{
					if( in != null )
						try {
							in.close();
						} catch (IOException e) {}
				}
			}
			
			//evaluating expression
			try{
				String evalString = config.getChildren("eval")[0].getAttribute("expr");
				System.out.println( "Evaluating <"+evalString+">...");
				result = (String)cx.evaluateString( scope, evalString, "Eval_Expression", 1, null );
				
			}
			catch( EcmaError e ){
				e.printStackTrace();
			}
		}
		catch( Exception e ){
			e.printStackTrace();
		}
		finally {
			Context.exit();
		}

		return result;
	}

	/**
	 * This implementation expects the following XML under the
	 * IConfigurationElement <br/> 1-* &lt;load file|dir="" /&gt; <br/> 1
	 * &lt;eval expr="" /&gt;
	 * 
	 * @see IExecutableExtension
	 */
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		//save the config element relevant to versioning
		this.config = config.getChildren("version")[0];
	}

}
