/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.action;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.eclipse.atf.generator.model.BuilderInfo;
import org.eclipse.atf.generator.model.FacetInfo;
import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.NatureInfo;
import org.eclipse.atf.generator.model.RuntimeInfo;
import org.eclipse.atf.generator.model.RuntimeInstanceInfo;
import org.eclipse.atf.generator.model.Utils;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.osgi.service.pluginconversion.PluginConversionException;
import org.eclipse.osgi.service.pluginconversion.PluginConverter;
import org.eclipse.pde.core.build.IBuildEntry;
import org.eclipse.pde.core.plugin.IPluginBase;
import org.eclipse.pde.core.plugin.IPluginElement;
import org.eclipse.pde.core.plugin.IPluginExtension;
import org.eclipse.pde.core.plugin.IPluginModelFactory;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.ui.PDEUIMessages;
import org.eclipse.pde.internal.ui.wizards.IProjectProvider;
import org.eclipse.pde.internal.ui.wizards.plugin.PluginFieldData;
import org.eclipse.pde.ui.IFieldData;
import org.eclipse.pde.ui.IPluginContentWizard;
import org.eclipse.pde.ui.IPluginFieldData;
import org.osgi.framework.Constants;
import org.osgi.util.tracker.ServiceTracker;


public class CorePluginCreationOperation extends NewPluginProjectCreationOperation {

	MrCleanToolingInfo dataModel;
	
	public CorePluginCreationOperation(IFieldData data, IProjectProvider provider, MrCleanToolingInfo dataModel, IPluginContentWizard contentWizard) {
		super(data, provider, contentWizard);
		this.dataModel = dataModel;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.actions.WorkspaceModifyOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected void execute(IProgressMonitor monitor) throws CoreException,
			InvocationTargetException, InterruptedException {

		// start task
		monitor.beginTask(PDEUIMessages.NewProjectCreationOperation_creating,
				getNumberOfWorkUnits()); //$NON-NLS-1$
		monitor.subTask(PDEUIMessages.NewProjectCreationOperation_project); //$NON-NLS-1$

		// create project
		IProject project = createProject();
		monitor.worked(1);
		createContents(monitor, project);
		// set classpath if project has a Java nature
		if (project.hasNature(JavaCore.NATURE_ID)) {
			monitor
					.subTask(PDEUIMessages.NewProjectCreationOperation_setClasspath); //$NON-NLS-1$
			setClasspath(project, fData);
			monitor.worked(1);
		}

		if (fData instanceof IPluginFieldData) {
			IPluginFieldData data = (IPluginFieldData) fData;

			// generate top-level Java class if that option is selected
			if (data.doGenerateClass()) {
				generateTopLevelPluginClass(project, new SubProgressMonitor(
						monitor, 1));
			}
		}
		// generate the manifest file
		monitor.subTask(PDEUIMessages.NewProjectCreationOperation_manifestFile); //$NON-NLS-1$
		createManifest(project);
		monitor.worked(1);

		// generate the build.properties file
		monitor
				.subTask(PDEUIMessages.NewProjectCreationOperation_buildPropertiesFile); //$NON-NLS-1$
		createBuildPropertiesFile(project, true);
		monitor.worked(1);

		// generate content contributed by template wizards
		boolean contentWizardResult = true;
		if (fContentWizard != null) {
			contentWizardResult = fContentWizard.performFinish(project, fModel,
					new SubProgressMonitor(monitor, 1));
		}

		fModel.save();

		adjustManifests(monitor, project);
		/*if (fData.hasBundleStructure()) {
			openFile(project.getFile("META-INF/MANIFEST.MF")); //$NON-NLS-1$
		} else {
			openFile((IFile) fModel.getUnderlyingResource());
		}*/
		addExports();
		monitor.worked(1);

		fResult = contentWizardResult;
	}
	
	protected void addExports() {
		if(fData.hasBundleStructure()) {
			IFile file = fProjectProvider.getProject().getFile(JarFile.MANIFEST_NAME);
			if (file.exists()) {
				String naturePackage = dataModel.getNatureInfo().getPackageName();
				String facetPackage = dataModel.getFacetInfo().getPackageName();
				InputStream manifestStream = null;
				try {
					manifestStream = new FileInputStream(file.getLocation().toFile());
					Manifest manifest = new Manifest(manifestStream);
					Properties prop = manifestToProperties(manifest.getMainAttributes());
					prop.put(Constants.EXPORT_PACKAGE, naturePackage);
					prop.put(Constants.EXPORT_PACKAGE, facetPackage);
					ServiceTracker tracker = new ServiceTracker(PDECore.getDefault()
							.getBundleContext(), PluginConverter.class.getName(), null);
					tracker.open();
					PluginConverter converter = (PluginConverter) tracker.getService();
					converter.writeManifest(new File(file.getLocation().toOSString()), prop, false);
					file.refreshLocal(1, null);
					tracker.close();
				} catch (FileNotFoundException e) {
				} catch (IOException e) {
				} catch (PluginConversionException e) {
				} catch (CoreException e) {
				} finally {
					try {
						if (manifestStream != null)
							manifestStream.close();
					} catch (IOException e) {
					}
				}
			}
		}
	}
	
	private static Properties manifestToProperties(Attributes d) {
		Iterator iter = d.keySet().iterator();
		Properties result = new Properties();
		while (iter.hasNext()) {
			Attributes.Name key = (Attributes.Name) iter.next();
			result.put(key.toString(), d.get(key));
		}
		return result;
	}
	
	protected void fillBinIncludes(IProject project, IBuildEntry binEntry, boolean hasExtensions) throws CoreException {
		super.fillBinIncludes(project, binEntry, hasExtensions);
		binEntry.addToken(FileCreationUtils.ICONS_DIR+"/");
		
	}
	
	protected void createManifest(IProject project) throws CoreException {
		super.createManifest(project);
		addExtensions();
	}
		
	protected void addExtensions() throws CoreException {
		
		dataModel.generateIcons(fProjectProvider.getProject());
		addRuntimeExtensions();
		addNatureExtension();
		//addBuilderExtension();
		addFacetExtension();
		
	}
	
	/*
	 * <!-- Runtime Type Declaration for Dojo -->
	 *	<extension point="org.eclipse.atf.core.runtime">	
	 *		<runtime id="dojo" name="Dojo Toolkit">
	 *			<nature id="org.eclipse.atf.personality.dojo.core.dojoNature"/>
	 *			<handler class="org.eclipse.atf.personality.dojo.runtime.DojoRuntimeHandler"/>
	 *		</runtime>
	 *	</extension>
		
	 *	<!-- Embedded Runtime Instance Definition for Dojo -->
	 *	<extension point="org.eclipse.atf.core.runtimeInstance">
	 *			
	 *		<runtime type="dojo" 
	 *			name="dojo_0.3.1"
	 *			location="platform:/plugin/org.eclipse.atf.personality.dojo.runtime/resources"
	 *			version="0.3.1"/>
	 *		
	 *	</extension>
	 */
	protected void addRuntimeExtensions() throws CoreException {
		
		RuntimeInfo runtimeInfo = dataModel.getRuntimeInfo();
		
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		
		//Runtime extension
		IPluginExtension runtimeExtension = createExtension("org.eclipse.atf.core.runtime", true); //$NON-NLS-1$
		
		IPluginElement runtimeElt = factory.createElement(runtimeExtension);
		runtimeElt.setName( "runtime" );
		runtimeElt.setAttribute( "id", runtimeInfo.getRuntimeId() );
		runtimeElt.setAttribute("name", runtimeInfo.getRuntimeName() );
		
		
		IPluginElement natureElt = factory.createElement(runtimeElt);
		natureElt.setName("nature");
		natureElt.setAttribute("id", runtimeInfo.getNature() );
		runtimeElt.add(natureElt);
		
		IPluginElement handlerElt = factory.createElement(runtimeElt);
		handlerElt.setName("handler");
		handlerElt.setAttribute("class", runtimeInfo.getHandlerPackageName() + "." + runtimeInfo.getHandlerClassName() );
		runtimeElt.add(handlerElt);
		
		runtimeExtension.add(runtimeElt);
		
		if (!runtimeExtension.isInTheModel())
			plugin.add(runtimeExtension);
		
		//optionally create an instance
		RuntimeInstanceInfo instanceInfo = dataModel.getRuntimeInstanceInfo();
		if( instanceInfo != null ){
			//RuntimeInstance extension
			IPluginExtension runtimeInstanceExtension = createExtension("org.eclipse.atf.core.runtimeInstance", true); //$NON-NLS-1$
			
			IPluginElement runtimeElt2 = factory.createElement(runtimeInstanceExtension);
			runtimeElt2.setName( "runtime" );
			runtimeElt2.setAttribute( "type", instanceInfo.getRuntimeType() );
			runtimeElt2.setAttribute("name", instanceInfo.getRuntimeName() );
			runtimeElt2.setAttribute("location", instanceInfo.getRuntimeLocation() );
			runtimeElt2.setAttribute("version", instanceInfo.getRuntimeVersion() );
			
			runtimeInstanceExtension.add(runtimeElt2);
			
			if (!runtimeInstanceExtension.isInTheModel())
				plugin.add(runtimeInstanceExtension);
		}
	}
	
	protected void addNatureExtension() throws CoreException {
		
		BuilderInfo builderInfo = dataModel.getBuilderInfo();
		NatureInfo natureInfo = dataModel.getNatureInfo();
		
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		
		//Nature extension
		IPluginExtension natureExtension = createExtension("org.eclipse.core.resources.natures", true); //$NON-NLS-1$
		natureExtension.setId(natureInfo.getNatureID());
		natureExtension.setName(natureInfo.getNatureName());
		
		IPluginElement runtimeElement = factory.createElement(natureExtension);
		runtimeElement.setName("runtime");
		IPluginElement runElement = factory.createElement(runtimeElement);
		runElement.setName("run");
		runElement.setAttribute("class", natureInfo.getFullClassName());
		runtimeElement.add(runElement);
		natureExtension.add(runtimeElement);
		
		/*
		IPluginElement assocBuilderElement = factory.createElement(natureExtension);
		assocBuilderElement.setName("builder");
		assocBuilderElement.setAttribute("id", builderInfo.getFullBuilderID());
		natureExtension.add(assocBuilderElement);
		*/
		
		if (!natureExtension.isInTheModel())
			plugin.add(natureExtension);
	}
	
	/*
	protected void addBuilderExtension() throws CoreException {
		
		BuilderInfo builderInfo = dataModel.getBuilderInfo();
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		
		//Builder extension
		IPluginExtension builderExtension = createExtension("org.eclipse.core.resources.builders", true);
		builderExtension.setId(builderInfo.getBuilderID());
		builderExtension.setName(builderInfo.getName());
		
		IPluginElement builderElement = factory.createElement(builderExtension);
		builderElement.setName("builder");
		builderElement.setAttribute("hasNature", "false");
		IPluginElement builderRunElement = factory.createElement(builderElement);
		builderRunElement.setName("run");
		builderRunElement.setAttribute("class", builderInfo.getFullClassName());
		builderElement.add(builderRunElement);
		builderExtension.add(builderElement);
		
		if (!builderExtension.isInTheModel())
			plugin.add(builderExtension);
	}
	*/
	
	protected void addFacetExtension() throws CoreException {
		
		FacetInfo facetInfo = dataModel.getFacetInfo();
		String smallIconPath = dataModel.getSmallIconPath();
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		
		//Facet extension
		IPluginExtension facetExtension = createExtension("org.eclipse.wst.common.project.facet.core.facets", true);
		
		IPluginElement projectFacetElement = factory.createElement(facetExtension);
		projectFacetElement.setName("project-facet");
		projectFacetElement.setAttribute("id", dataModel.getCamelCasedTechnologyName());
		
		IPluginElement labelElement = factory.createElement(projectFacetElement);
		labelElement.setName("label");
		labelElement.setText(Utils.capitalize(dataModel.getTechnologyName()));
		projectFacetElement.add(labelElement);
		
		IPluginElement categoryElement = factory.createElement(projectFacetElement);
		categoryElement.setName("category");
		categoryElement.setText("atf.category");
		projectFacetElement.add(categoryElement);
		
		IPluginElement descriptionElement = factory.createElement(projectFacetElement);
		descriptionElement.setName("description");
		descriptionElement.setText("Adds support for developing Ajax web applications with the "+Utils.capitalize(dataModel.getTechnologyName())+" toolkit.");
		facetExtension.add(projectFacetElement);
		
		IPluginElement versionElement = factory.createElement(facetExtension);
		versionElement.setName("project-facet-version");
		versionElement.setAttribute("facet", dataModel.getCamelCasedTechnologyName());
		versionElement.setAttribute("version", "1.0");
		
		IPluginElement constraintElement = factory.createElement(versionElement);
		constraintElement.setName("constraint");
		
		IPluginElement orElement = factory.createElement(constraintElement);
		orElement.setName("or");
		
		IPluginElement andElement1 = factory.createElement(orElement);
		andElement1.setName("and");
		
		IPluginElement requiresElement1 = factory.createElement(andElement1);
		requiresElement1.setName("requires");
		requiresElement1.setAttribute("facet", "jst.web");
		requiresElement1.setAttribute("version", "[2.2");
		andElement1.add(requiresElement1);
		
		IPluginElement requiresElement2 = factory.createElement(andElement1);
		requiresElement2.setName("requires");
		requiresElement2.setAttribute("facet", "jst.java");
		requiresElement2.setAttribute("version", "[1.3");
		andElement1.add(requiresElement2);
		orElement.add(andElement1);
		
		IPluginElement andElement2 = factory.createElement(orElement);
		andElement2.setName("and");
		
		IPluginElement requiresElement3 = factory.createElement(andElement2);
		requiresElement3.setName("requires");
		requiresElement3.setAttribute("facet", "wst.web");
		requiresElement3.setAttribute("version", "[1.0");
		andElement2.add(requiresElement3);
		orElement.add(andElement2);
		constraintElement.add(orElement);
		versionElement.add(constraintElement);
		facetExtension.add(versionElement);
		
		IPluginElement installElement = factory.createElement(facetExtension);
		installElement.setName("action");
		installElement.setAttribute("type", "install");
		installElement.setAttribute("facet", dataModel.getCamelCasedTechnologyName());
		installElement.setAttribute("version", "1.0");
		
		IPluginElement installDelegateElement = factory.createElement(installElement);
		installDelegateElement.setName("delegate");
		installDelegateElement.setAttribute("class", facetInfo.getPackageName()+"."+facetInfo.getInstallDelegateClass());
		installElement.add(installDelegateElement);
		
		facetExtension.add(installElement);
		
		IPluginElement uninstallElement = factory.createElement(facetExtension);
		uninstallElement.setName("action");
		uninstallElement.setAttribute("type", "uninstall");
		uninstallElement.setAttribute("facet", dataModel.getCamelCasedTechnologyName());
		uninstallElement.setAttribute("version", "1.0");
		
		IPluginElement uninstallDelegateElement = factory.createElement(uninstallElement);
		uninstallDelegateElement.setName("delegate");
		uninstallDelegateElement.setAttribute("class", facetInfo.getPackageName()+"."+facetInfo.getUninstallDelegateClass());
		uninstallElement.add(uninstallDelegateElement);
		
		facetExtension.add(uninstallElement);
		if (!facetExtension.isInTheModel())
			plugin.add(facetExtension);
		
		//Facet image extension
		if(smallIconPath != null && !smallIconPath.equals("")) {
			IPluginExtension facetImageExtension = createExtension("org.eclipse.wst.common.project.facet.ui.images", true);
			IPluginElement imageElement = factory.createElement(facetImageExtension);
			imageElement.setName("image");
			imageElement.setAttribute("facet", dataModel.getCamelCasedTechnologyName());
			imageElement.setAttribute("path", smallIconPath);
			facetImageExtension.add(imageElement);
			if (!facetImageExtension.isInTheModel())
				plugin.add(facetImageExtension);
		}
		
		//Facet runtimes extension
		IPluginExtension facetRuntimeExtension = createExtension("org.eclipse.wst.common.project.facet.core.runtimes", true);
		IPluginElement supportedElement = factory.createElement(facetRuntimeExtension);
		supportedElement.setName("supported");
		IPluginElement runtimeElement = factory.createElement(supportedElement);
		runtimeElement.setName("runtime-component");
		runtimeElement.setAttribute("any", "true");
		supportedElement.add(runtimeElement);
		IPluginElement facetElement = factory.createElement(supportedElement);
		facetElement.setName("facet");
		facetElement.setAttribute("id", dataModel.getCamelCasedTechnologyName());
		facetElement.setAttribute("version", "1.0");
		supportedElement.add(facetElement);
		facetRuntimeExtension.add(supportedElement);
		if (!facetRuntimeExtension.isInTheModel())
			plugin.add(facetRuntimeExtension);
		
	}
	
	
	
	protected void generateTopLevelPluginClass(IProject project, IProgressMonitor monitor) throws CoreException {
		PluginFieldData data = (PluginFieldData) fData;
		fGenerator = new CorePluginClassCodeGenerator(project, data.getClassname(), data);
		fGenerator.generate(monitor);
		monitor.done();
	}
}
