/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.action;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;

import org.eclipse.atf.generator.model.ApplicationWizardInfo;
import org.eclipse.atf.generator.model.FileTemplateInfo;
import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.SnippetInfo;
import org.eclipse.atf.generator.model.Utils;
import org.eclipse.atf.generator.model.VariableInfo;
import org.eclipse.atf.generator.model.VariableViewPageInfo;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.pde.core.build.IBuildEntry;
import org.eclipse.pde.core.plugin.IPluginBase;
import org.eclipse.pde.core.plugin.IPluginElement;
import org.eclipse.pde.core.plugin.IPluginExtension;
import org.eclipse.pde.core.plugin.IPluginModelFactory;
import org.eclipse.pde.internal.ui.PDEUIMessages;
import org.eclipse.pde.internal.ui.wizards.IProjectProvider;
import org.eclipse.pde.internal.ui.wizards.plugin.PluginFieldData;
import org.eclipse.pde.ui.IFieldData;
import org.eclipse.pde.ui.IPluginContentWizard;
import org.eclipse.pde.ui.IPluginFieldData;


public class UiPluginCreationOperation extends NewPluginProjectCreationOperation {

	MrCleanToolingInfo dataModel;
	String wizardCategoryId;
	String smallIconPath;
	
	
	public UiPluginCreationOperation(IFieldData data, IProjectProvider provider, MrCleanToolingInfo dataModel, IPluginContentWizard contentWizard) {
		super(data, provider, contentWizard);
		this.dataModel = dataModel;
		
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.actions.WorkspaceModifyOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected void execute(IProgressMonitor monitor) throws CoreException,
			InvocationTargetException, InterruptedException {

		// start task
		monitor.beginTask(PDEUIMessages.NewProjectCreationOperation_creating,
				getNumberOfWorkUnits()); //$NON-NLS-1$
		monitor.subTask(PDEUIMessages.NewProjectCreationOperation_project); //$NON-NLS-1$

		// create project
		IProject project = createProject();
		monitor.worked(1);
		createContents(monitor, project);
		// set classpath if project has a Java nature
		if (project.hasNature(JavaCore.NATURE_ID)) {
			monitor
					.subTask(PDEUIMessages.NewProjectCreationOperation_setClasspath); //$NON-NLS-1$
			setClasspath(project, fData);
			monitor.worked(1);
		}

		if (fData instanceof IPluginFieldData) {
			IPluginFieldData data = (IPluginFieldData) fData;

			// generate top-level Java class if that option is selected
			if (data.doGenerateClass()) {
				generateTopLevelPluginClass(project, new SubProgressMonitor(
						monitor, 1));
			}
		}
		// generate the manifest file
		monitor.subTask(PDEUIMessages.NewProjectCreationOperation_manifestFile); //$NON-NLS-1$
		createManifest(project);
		monitor.worked(1);

		// generate the build.properties file
		monitor
				.subTask(PDEUIMessages.NewProjectCreationOperation_buildPropertiesFile); //$NON-NLS-1$
		createBuildPropertiesFile(project, true);
		monitor.worked(1);

		// generate content contributed by template wizards
		boolean contentWizardResult = true;
		if (fContentWizard != null) {
			contentWizardResult = fContentWizard.performFinish(project, fModel,
					new SubProgressMonitor(monitor, 1));
		}

		fModel.save();

		adjustManifests(monitor, project);
		/*if (fData.hasBundleStructure()) {
			openFile(project.getFile("META-INF/MANIFEST.MF")); //$NON-NLS-1$
		} else {
			openFile((IFile) fModel.getUnderlyingResource());
		}*/
		monitor.worked(1);

		fResult = contentWizardResult;
	}
	
	
	
	protected void fillBinIncludes(IProject project, IBuildEntry binEntry, boolean hasExtensions) throws CoreException {
		super.fillBinIncludes(project, binEntry, hasExtensions);
		binEntry.addToken(FileCreationUtils.ICONS_DIR+"/");
		binEntry.addToken(FileCreationUtils.TEMPLATES_DIR+"/");
	}

	protected void createManifest(IProject project) throws CoreException {
		super.createManifest(project);
		addExtensions();
	}
	
	protected void addExtensions() throws CoreException {
		
		dataModel.generateIcons(fProjectProvider.getProject());
		//addProjectWizardExtensions();
		addApplicationWizardExtensions();
		addSnippetExtensions();
		
	}
		
		
	protected void addApplicationWizardExtensions() throws CoreException {
		
		ApplicationWizardInfo wizardInfo = dataModel.getApplicationWizardInfo();
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		String techname = Utils.capitalize(dataModel.getCamelCasedTechnologyName());
		//String ajaxRoot = "WebContent/"+dataModel.getRuntimeArtifactsInfo().getRuntimeProjectFolder();
		//the path is always assumed to be under the root container of web resources
		String ajaxRoot = dataModel.getRuntimeInfo().getRuntimeInstallRoot();
		
		smallIconPath = dataModel.getSmallIconPath();
		wizardCategoryId = dataModel.getUiProjectName() + "wizardCategory";
		
		//Create wizard category
		IPluginExtension wizardExtension = createExtension("org.eclipse.ui.newWizards", true);
		IPluginElement categoryElement = factory.createElement(wizardExtension);
		categoryElement.setName("category");
		categoryElement.setAttribute("name", techname);
		categoryElement.setAttribute("id", wizardCategoryId);
		wizardExtension.add(categoryElement);
				
		//Create the Application Wizard extension
		String wizardId = wizardInfo.getPackageName()+"."+techname+"ApplicationWizard";
		
		IPluginElement wizardElement = factory.createElement(wizardExtension);
		wizardElement.setName("wizard");
		wizardElement.setAttribute("name", wizardInfo.getName());
		if(smallIconPath != null && !smallIconPath.equals("")) 
			wizardElement.setAttribute("icon", smallIconPath);
		wizardElement.setAttribute("category", wizardCategoryId);
		wizardElement.setAttribute("class", wizardInfo.getPackageName() + "." + wizardInfo.getClassName());
		wizardElement.setAttribute("id", wizardId);
		
		IPluginElement descElement = factory.createElement(wizardElement);
		descElement.setName("description");
		descElement.setText(wizardInfo.getDescription());
		
		wizardElement.add(descElement);
		wizardExtension.add(wizardElement);
		if(!wizardExtension.isInTheModel())
			plugin.add(wizardExtension);
		
		//Create the wizard model extension
		IPluginExtension modelExtension = createExtension("org.eclipse.atf.templates.wizardModel", true);
		
		IPluginElement modelElement = factory.createElement(modelExtension);
		modelElement.setName("model");
		modelElement.setAttribute("targetWizard", wizardId);
		modelElement.setAttribute("title", wizardInfo.getName());
		
		IPluginElement variablesElement = factory.createElement(modelElement);
		variablesElement.setName("variables");
		
		//Go through the defined variables from the application wizard model and add variable elements
		VariableInfo[] vars = wizardInfo.getVariableRefs();
		for(int i=0; i<vars.length; i++) {
			IPluginElement varElement = factory.createElement(variablesElement);
			varElement.setName("variable");
			varElement.setAttribute("name", vars[i].getName());
			varElement.setAttribute("required", vars[i].isRequired() ? "true" : "false");
			varElement.setAttribute("displayName", vars[i].getDisplayName());
			varElement.setAttribute("type", vars[i].getType());
			if(vars[i].getType().equals(VariableInfo.PROJECT_TYPE))
				varElement.setAttribute("defaultSelected", "true");
			if(vars[i].getType().equals(VariableInfo.RELATIVE_PATH_TYPE)){
				if( !"".equals(vars[i].getOtherAttributeValue("from")) )
					varElement.setAttribute("from", vars[i].getOtherAttributeValue("from"));
				varElement.setAttribute("to", ajaxRoot);
				
			}
			if(vars[i].getDefaultValue() != null && !vars[i].getDefaultValue().equals(""))
				varElement.setAttribute("value", vars[i].getDefaultValue());
			String optionsString = createOptionsString(vars[i].getOptions());
			if(optionsString != null)
				varElement.setAttribute("options", optionsString);
			if(vars[i].getType().equals(VariableInfo.PROJECT_TYPE) && vars[i].getFilterClasses() != null) {
				String[] filters = vars[i].getFilterClasses();
				for(int j=0; j<filters.length; j++) {
					IPluginElement filterElement = factory.createElement(varElement);
					filterElement.setName("projectFilter");
					filterElement.setAttribute("class", filters[j]);
					varElement.add(filterElement);
				}
			}
			variablesElement.add(varElement);
		}
		modelElement.add(variablesElement);
				
		//Go through the defined pages from the application wizard model and add page elements
		IPluginElement pagesElement = factory.createElement(modelElement);
		pagesElement.setName("pages");
		VariableViewPageInfo[] pages = wizardInfo.getViewPages();
		for(int i=0; i<pages.length; i++) {
			IPluginElement pageElement = factory.createElement(pagesElement);
			pageElement.setName("page");
			pageElement.setAttribute("name", pages[i].getName());
			pageElement.setAttribute("title", pages[i].getTitle());
			IPluginElement descriptionElement = factory.createElement(pageElement);
			descriptionElement.setName("description");
			descriptionElement.setText(pages[i].getDescription());
			pageElement.add(descriptionElement);
			VariableInfo[] varRefs = pages[i].getVariableRefs();
			for(int j=0; j<varRefs.length; j++) {
				IPluginElement varRefElement = factory.createElement(pageElement);
				varRefElement.setName("variableRef");
				varRefElement.setAttribute("name", varRefs[j].getName());
				pageElement.add(varRefElement);
			}
			pagesElement.add(pageElement);
		}
		modelElement.add(pagesElement);
		
		//Go through the defined file templates from the application wizard model and add creatFile elements
		IPluginElement finishElement = factory.createElement(modelElement);
		finishElement.setName("finish");
		
		//add an implicit operation for injecting the Runtime if it is not installed
		/*
		 * <operation
         *        type="InstallRuntime"
         *        project="${project}"
         *        runtime="<ajax runtime>">
         *  </operation>
		 */
		IPluginElement installRuntimeOperation = factory.createElement(finishElement);
		installRuntimeOperation.setName("operation");
		installRuntimeOperation.setAttribute("type", "InstallRuntime");
		installRuntimeOperation.setAttribute("project", "${project}");
		installRuntimeOperation.setAttribute("runtime", dataModel.getRuntimeInfo().getRuntimeName() );
		finishElement.add(installRuntimeOperation);
		
		FileTemplateInfo[] templates = wizardInfo.getFileTemplates();
		for(int i=0; i<templates.length; i++) {
			FileTemplateInfo info = templates[i];
			String[] conditionalStatements = info.getCreationConditions();
			if(conditionalStatements != null && conditionalStatements.length != 0) {
				String conditionString = "";
				String[] conditionStatements = info.getCreationConditions();
				for(int j=0; j<conditionStatements.length; j++) {
					conditionString += conditionStatements[j];
					if(conditionStatements.length > j+1)
						conditionString += " && ";
				}
				IPluginElement conditionOperationElement = factory.createElement(finishElement);
				conditionOperationElement.setName("operation");
				conditionOperationElement.setAttribute("type", "Conditional");
				conditionOperationElement.setAttribute("condition", conditionString);
				
				IPluginElement operationElement = factory.createElement(conditionOperationElement);
				operationElement.setName("operation");
				operationElement.setAttribute("type", "CreateFile");
				operationElement.setAttribute("name", info.getName());
				operationElement.setAttribute("location", info.getProjectRelativeLocation());
				operationElement.setAttribute("project", "${project}");
				
				IPluginElement templateElement = factory.createElement(operationElement);
				templateElement.setName("template");
				String templateLocation = FileCreationUtils.createTemplateFile(fProjectProvider.getProject(), 
																				info.getTemplate(), 
																				FileCreationUtils.FILE_TEMPLATE_TYPE, 
																				info.getName()+".template");
				templateElement.setAttribute("path", templateLocation);
				
				operationElement.add(templateElement);
				conditionOperationElement.add(operationElement);
				finishElement.add(conditionOperationElement);
			} else {
				IPluginElement operationElement = factory.createElement(finishElement);
				operationElement.setName("operation");
				operationElement.setAttribute("type", "CreateFile");
				operationElement.setAttribute("name", info.getName());
				operationElement.setAttribute("location", info.getProjectRelativeLocation());
				operationElement.setAttribute("project", "${project}");
				
				IPluginElement templateElement = factory.createElement(operationElement);
				templateElement.setName("template");
				String templateLocation = FileCreationUtils.createTemplateFile(fProjectProvider.getProject(), 
																				info.getTemplate(), 
																				FileCreationUtils.FILE_TEMPLATE_TYPE, 
																				info.getName()+".template");
				templateElement.setAttribute("path", templateLocation);
				
				operationElement.add(templateElement);
				finishElement.add(operationElement);
			}
			
		}
		modelElement.add(finishElement);
		
		modelExtension.add(modelElement);
		if(!modelExtension.isInTheModel())
			plugin.add(modelExtension);
			
	}
	
	protected void addSnippetExtensions() throws CoreException {
		
		SnippetInfo[] snippetInfos = dataModel.getSnippetInfo();
		if(snippetInfos == null || snippetInfos.length == 0)
			return;
		
		IPluginBase plugin = fModel.getPluginBase();
		IPluginModelFactory factory = fModel.getPluginFactory();
		String techname = Utils.capitalize(dataModel.getCamelCasedTechnologyName());
		String snippetCategoryId = dataModel.getUiProjectName() + ".snippetCategory";
		String snippetIdPrefix = dataModel.getUiProjectName()+".snippet.";
		
		//Create SnippetContribution extensions
		IPluginExtension snippetExtension = createExtension("org.eclipse.wst.common.snippets.SnippetContributions", true); 
		
		IPluginElement categoryElement = factory.createElement(snippetExtension);
		categoryElement.setName("category");
		categoryElement.setAttribute("label", techname);
		categoryElement.setAttribute("id", snippetCategoryId);
		if(dataModel.getSmallIconPath() != null && !dataModel.getSmallIconPath().equals(""))
			categoryElement.setAttribute("smallicon", dataModel.getSmallIconPath());
		categoryElement.setAttribute("contenttypes", "org.eclipse.jst.jsp.core.jspsource");
				
		for(int i=0; i<snippetInfos.length; i++) {
			IPluginElement itemElement = factory.createElement(categoryElement);
			itemElement.setName("item");
			itemElement.setAttribute("label", snippetInfos[i].getDisplayName());
			itemElement.setAttribute("id", snippetIdPrefix + "snippet" + (i+1));
			itemElement.setAttribute("class", "org.eclipse.atf.templates.util.TemplateInsertion2");
			categoryElement.add(itemElement);
		}
		snippetExtension.add(categoryElement);
		if(!snippetExtension.isInTheModel())
			plugin.add(snippetExtension);
		
		//Create SnippetModel extensions
		IPluginExtension modelExtension = createExtension("org.eclipse.atf.templates.snippetModel", true);
		
		for(int i=0; i<snippetInfos.length; i++) {
			IPluginElement modelElement = factory.createElement(modelExtension);
			modelElement.setName("model");
			String snippetId = "snippet"+(i+1);
			modelElement.setAttribute("targetSnippet", snippetIdPrefix + snippetId);
			VariableInfo[] varInfos = snippetInfos[i].getVariables();
			if(varInfos.length > 0) {
				IPluginElement variablesElement = factory.createElement(modelElement);
				variablesElement.setName("variables");
				for(int j=0; j<varInfos.length; j++) {
					IPluginElement varElement = factory.createElement(variablesElement);
					varElement.setName("variable");
					varElement.setAttribute("name", varInfos[j].getName());
					varElement.setAttribute("displayName", varInfos[j].getDisplayName());
					varElement.setAttribute("type", varInfos[j].getType());
					if(varInfos[j].getDefaultValue() != null && !varInfos[j].getDefaultValue().equals(""))
						varElement.setAttribute("value", varInfos[j].getDefaultValue());
					String optionsString = createOptionsString(varInfos[j].getOptions());
					if(optionsString != null)
						varElement.setAttribute("options", optionsString);
					variablesElement.add(varElement);
				}
				modelElement.add(variablesElement);
				
				IPluginElement dialogElement = factory.createElement(modelElement);
				dialogElement.setName("dialog");
				dialogElement.setAttribute("title", "Insert " + snippetInfos[i].getDisplayName());
				
				IPluginElement descElement = factory.createElement(dialogElement);
				descElement.setName("description");
				descElement.setText("Specify inputs for the code snippet");
				dialogElement.add(descElement);
				for(int j=0; j<varInfos.length; j++) {
					IPluginElement varRefElement = factory.createElement(dialogElement);
					varRefElement.setName("variableRef");
					varRefElement.setAttribute("name", varInfos[j].getName());
					dialogElement.add(varRefElement);
				}
				modelElement.add(dialogElement);
			}
			
			String templateFile = FileCreationUtils.createTemplateFile(fProjectProvider.getProject(), 
																		snippetInfos[i].getTemplate(), 
																		FileCreationUtils.SNIPPET_TEMPLATE_TYPE, 
																		snippetId+".template");
			IPluginElement templateElement = factory.createElement(modelElement);
			templateElement.setName("template");
			templateElement.setAttribute("path", templateFile);
			modelElement.add(templateElement);
			
			modelExtension.add(modelElement);
		}
		if(!modelExtension.isInTheModel())
			plugin.add(modelExtension);
		
		
	}
	
	private String createOptionsString(String[] options) {
		
		if(options != null && options.length > 0) {
			String optionsString = "";
			for(int j=0; j<options.length; j++) {
				optionsString += options[j];
				if(j < (options.length - 1))
					optionsString += "|";
			}
			return optionsString;
		}
		return null;
	}
	
	protected void generateTopLevelPluginClass(IProject project, IProgressMonitor monitor) throws CoreException {
		PluginFieldData data = (PluginFieldData) fData;
		fGenerator = new UiPluginClassCodeGenerator(project, data.getClassname(), data, getTechnologyProjectIncludes());
		fGenerator.generate(monitor);
		monitor.done();
	}
	
	private String[] getTechnologyProjectIncludes() {
		ArrayList list = new ArrayList();
		list.add(dataModel.getCoreProjectName());
		//if(dataModel.getProjectWizardInfo().doCreateJ2EEWizard())
		//	list.add(dataModel.getJ2EEProjectName());
		//if(dataModel.getProjectWizardInfo().doCreateNonJ2EEWizard())
		//	list.add(dataModel.getStaticWebProjectName());
		String[] includes = new String[list.size()];
		includes = (String[])list.toArray(includes);
		return includes;
	}
}
