/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui;

import java.util.Arrays;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

/**
 * A composite panel which contains a table for managing instances of ViewPage objects.
 * @author Don Sedota
 *
 */
public class PageTableViewerPanel extends Composite {

	private Table table;	//The table
	private TableViewer tableViewer;	//The table viewer
	private ViewPageList pageList;		//The page list to be viewed
	private TableInputListener tableInputListener; //Listener to be notified of table events.
	
	//Column name constants
	private static final String NAME_COL = "name";
	private static final String TITLE_COL = "title";
	private static final String DESCRIPTION_COL = "description";
	
	//Column names
	private static final String[] columnNames = new String[] {NAME_COL, TITLE_COL, DESCRIPTION_COL};
	
	/**
	 * Constructor.
	 * @param parent The parent Composite.
	 * @param style The SWT style.
	 */
	public PageTableViewerPanel(Composite parent, int style) {
		super(parent, style);
		setLayout(new GridLayout());
		createChildComponents();
	}
	
	/**
	 * Creates the sub components within the panel.
	 *
	 */
	protected void createChildComponents() {
		
		pageList = new ViewPageList();
		createTable();
		createTableViewer();
		
		tableViewer.setContentProvider(new ViewPageListContentProvider());
		tableViewer.setLabelProvider(new ViewPageListLabelProvider());
				
		tableViewer.setInput(pageList);
		
	}
	
	/**
	 * Creates the ViewPage table.
	 *
	 */
	protected void createTable() {
		
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
				SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
		
		table = new Table(this, style);
										
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
				
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// 1st column 
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);		
		column.setText("Name");
		column.setWidth(100);
		
		// 2nd column 
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Title");
		column.setWidth(150);
						
		// 3rd column 
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Description");
		column.setWidth(200);
				
		
	}
	
	/**
	 * Creates the table viewer used to view the ViewPage table.
	 *
	 */
	protected void createTableViewer() {
		
		tableViewer = new TableViewer(table); 
		tableViewer.setUseHashlookup(true);
				
		tableViewer.setColumnProperties(columnNames);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[columnNames.length];

		// Column 1 : Name (text)
		TextCellEditor textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[0] = textEditor;

		// Column 2 : Title (text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[1] = textEditor;

		// Column 3 : Description (Text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(200);
		editors[2] = textEditor;

		// Assign the cell editors to the viewer 
		tableViewer.setCellEditors(editors);
		// Set the cell modifier for the viewer
		tableViewer.setCellModifier(new ViewPageCellModifier(this));
		
	}
	
	public java.util.List getColumnNames() {
		return Arrays.asList(columnNames);
	}
	
	public ViewPageList getPageList() {
		return pageList;
	}
	
	public ViewPage[] getPages() {
		ViewPage[] pages = new ViewPage[pageList.getPages().size()];
		pages = (ViewPage[])pageList.getPages().toArray(pages);
		return pages;
	}
	
	public void removeSelectedPage() {
		ViewPage page = (ViewPage) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		if (page != null) {
			pageList.removePage(page);
		} 				
	}
	
	public void addNewPage() {
		pageList.addPage();
	}
	
	/**
	 * Adds to the list of listeners to notify when the table selection changes.
	 * @param listener
	 */
	public void addTableSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
	}
	
	/**
	 * Adds to the list of listeners to notify when the table's input changes.
	 * @param listener
	 */
	public void addTableInputListener(TableInputListener listener) {
		this.tableInputListener = listener;
	}
	
	public ViewPage getSelectedViewPage() {
		ViewPage page = (ViewPage) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		return page;
	}
	
	/**
	 * The table viewer's content provider.
	 * @author Don Sedota
	 *
	 */
	private class ViewPageListContentProvider implements IStructuredContentProvider, IViewPageListViewer {
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			if (newInput != null)
				((ViewPageList) newInput).addChangeListener(this);
			if (oldInput != null)
				((ViewPageList) oldInput).removeChangeListener(this);
		}

		public void dispose() {
			pageList.removeChangeListener(this);
		}

		// Return the variables as an array of Objects
		public Object[] getElements(Object parent) {
			return pageList.getPages().toArray();
		}

		/* (non-Javadoc)
		 * @see IViewPageListViewer#addPage(ViewPage)
		 */
		public void addPage(ViewPage page) {
			tableViewer.add(page);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(pageList);
		}

		/* (non-Javadoc)
		 * @see IViewPageListViewer#removePage(ViewPage)
		 */
		public void removePage(ViewPage page) {
			tableViewer.remove(page);		
			if(tableInputListener != null)
				tableInputListener.tableInputModified(pageList);
		}

		/* (non-Javadoc)
		 * @see IViewPageListViewer#updatePage(ViewPage)
		 */
		public void updatePage(ViewPage page) {
			tableViewer.update(page, null);	
			if(tableInputListener != null)
				tableInputListener.tableInputModified(pageList);
			
		}
	}
	
	/**
	 * The table viewer's ITableLabelProvider.
	 * @author Don Sedota
	 *
	 */
	private class ViewPageListLabelProvider extends LabelProvider implements ITableLabelProvider {

		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			String result = "";
			ViewPage page = (ViewPage) element;
			switch (columnIndex) {
				case 0:  
					result = page.getName();
					break;
				case 1 :
					result = page.getTitle();
					break;
				case 2 :
					result = page.getDescription();
					break;
				default :
					break; 	
			}
			return result;
		}
		
	}
}
