/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui;

import java.util.Arrays;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

/**
 * Composite widget containing a table used for displaying Snippet instances.
 * @author Don Sedota
 *
 */
public class SnippetsTableViewerPanel extends Composite {

	protected Table table;
	protected TableViewer tableViewer;
	protected SnippetList snippetList;	//The list of Snippets 
	protected TableInputListener tableInputListener;	//Listener that reacts to changes in the table input
	
	protected static final String NAME_COL = "name";
	protected static final String DISPLAY_NAME_COL = "displayName";
		
	protected String[] columnNames;
	
	/**
	 * Constructor.
	 * @param parent
	 * @param style
	 */
	public SnippetsTableViewerPanel(Composite parent, int style) {
		super(parent, style);
		initialize();
		setLayout(new GridLayout());
		createChildComponents();
		
	}
	
	
	protected void initialize() {
		snippetList = new SnippetList();
		columnNames = new String[] {NAME_COL, DISPLAY_NAME_COL};
	}
	
	/**
	 * Creates the table and table viewer.
	 *
	 */
	protected void createChildComponents() {
		
		createTable();
		createTableViewer();
		
		tableViewer.setContentProvider(new SnippetListContentProvider());
		tableViewer.setLabelProvider(new SnippetListLabelProvider());
		tableViewer.setInput(snippetList);
	
	}
		
	protected void createTable() {
		
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
				SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
		
		table = new Table(this, style);
						
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
				
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// 1st column 
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);		
		column.setText("Name");
		column.setWidth(100);
		
		// 2nd column 
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Display Name");
		column.setWidth(150);
				
	}
	
	protected void createTableViewer() {
		
		tableViewer = new TableViewer(table); 
		tableViewer.setUseHashlookup(true);
		
		tableViewer.setColumnProperties(columnNames);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[columnNames.length];

		// Column 1 : Name (text)
		TextCellEditor textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[0] = textEditor;

		// Column 2 : Display Name (text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[1] = textEditor;

		// Assign the cell editors to the viewer 
		tableViewer.setCellEditors(editors);
		// Set the cell modifier for the viewer
		tableViewer.setCellModifier(new SnippetCellModifier(this));
		
	}
	
	/**
	 * Returns the table's column names.
	 * @return
	 */
	public java.util.List getColumnNames() {
		return Arrays.asList(columnNames);
	}
	
	/**
	 * Returns the list of Snippets being displayed.
	 * @return
	 */
	public SnippetList getSnippetList() {
		return snippetList;
	}
	
	/**
	 * Returns the list of Snippets being displayed.
	 * @return
	 */
	public Snippet[] getSnippets() {
		Snippet[] snippets = new Snippet[snippetList.getSnippets().size()];
		snippets = (Snippet[])snippetList.getSnippets().toArray(snippets);
		return snippets;
	}
	
	/**
	 * Returns the currently selected Snippet in the table.
	 * @return
	 */
	public Snippet getSelectedSnippet() {
		Snippet snippet = (Snippet) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		return snippet;
	}
	
	/**
	 * Removes the currently selected Snippet from the table.
	 *
	 */
	public void removeSelectedSnippet() {
		Snippet snippet = (Snippet) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		if (snippet != null) {
			snippetList.removeSnippet(snippet);
		} 				
	}
	
	/**
	 * Creates and adds a new Snippet instance.
	 *
	 */
	public void addNewSnippet() {
		snippetList.addSnippet();
	}
	
	/**
	 * Adds the specified Snippet to the table.
	 * @param snippet
	 */
	public void addSnippet(Snippet snippet) {
		snippetList.addSnippet(snippet);
	}
	
	/**
	 * Sets the specified SnippetList as the table's input.
	 * @param sList
	 */
	public void setInput(SnippetList sList) {
		snippetList = sList;
		tableViewer.setInput(snippetList);
	}
	
	/**
	 * Adds to the list of listeners to be notified when the table selection changes.
	 * @param listener
	 */
	public void addTableSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
	}
	
	/**
	 * Adds to the list of listeners to be notified when the table's input changes.
	 * @param listener
	 */
	public void addTableInputListener(TableInputListener listener) {
		this.tableInputListener = listener;
	}
	
	/**
	 * The Snippet table viewer's content provider.
	 * @author Don Sedota
	 *
	 */
	private class SnippetListContentProvider implements IStructuredContentProvider, ISnippetListViewer {
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			if (newInput != null)
				((SnippetList) newInput).addChangeListener(this);
			if (oldInput != null)
				((SnippetList) oldInput).removeChangeListener(this);
		}

		public void dispose() {
			snippetList.removeChangeListener(this);
		}

		// Return the snippets as an array of Objects
		public Object[] getElements(Object parent) {
			return snippetList.getSnippets().toArray();
		}

		/* (non-Javadoc)
		 * @see ISnippetListViewer#addSnippet(Snippet)
		 */
		public void addSnippet(Snippet snippet) {
			tableViewer.add(snippet);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(snippetList);
		}

		/* (non-Javadoc)
		 * @see ISnippetListViewer#removeSnippet(Snippet)
		 */
		public void removeSnippet(Snippet snippet) {
			tableViewer.remove(snippet);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(snippetList);
		}

		/* (non-Javadoc)
		 * @see ISnippetListViewer#updateSnippet(Snippet)
		 */
		public void updateSnippet(Snippet snippet) {
			tableViewer.update(snippet, null);	
			if(tableInputListener != null)
				tableInputListener.tableInputModified(snippetList);
			
		}
	}
	
	/**
	 * The Snippet table viewer's LabelProvider.
	 * @author Don Sedota
	 *
	 */
	private class SnippetListLabelProvider extends LabelProvider implements ITableLabelProvider {

		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			String result = "";
			Snippet snippet = (Snippet) element;
			switch (columnIndex) {
				case 0:  
					result = snippet.getName();
					break;
				case 1 :
					result = snippet.getDisplayName();
					break;
				default :
					break; 	
			}
			return result;
		}
		
	}
}
