/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui;

import java.util.Arrays;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;


public class TemplatesTableViewerPanel extends Composite {

	protected Table table;
	protected TableViewer tableViewer;
	protected Button newTemplateButton;
	protected Button removeTemplateButton;
	protected FileTemplateList templateList;
	protected TableInputListener tableInputListener;
	
	protected static final String NAME_COL = "name";
	protected static final String LOCATION_COL = "location";
	protected static final String CONDITION_COL = "condition";
		
	protected String[] columnNames;
	
	public TemplatesTableViewerPanel(Composite parent, int style) {
		super(parent, style);
		initialize();
		setLayout(new GridLayout());
		createChildComponents();
		
	}
	
	protected void initialize() {
		templateList = new FileTemplateList();
		columnNames = new String[] {NAME_COL, LOCATION_COL, CONDITION_COL};
	}
	
	protected void createChildComponents() {
		
		createTable();
		createTableViewer();
		
		tableViewer.setContentProvider(new TemplateListContentProvider());
		tableViewer.setLabelProvider(new TemplateListLabelProvider());
				
		tableViewer.setInput(templateList);

		
	}
	
	protected void createTable() {
		
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
				SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
		
		table = new Table(this, style);
						
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
				
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// 1st column 
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);		
		column.setText("Name");
		column.setWidth(100);
		
		// 2nd column 
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Project Relative Location");
		column.setWidth(150);
						
		// 3rd column 
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Condition");
		column.setWidth(150);
				
		
	}
	
	protected void createTableViewer() {
		
		tableViewer = new TableViewer(table); 
		tableViewer.setUseHashlookup(true);
		
		tableViewer.setColumnProperties(columnNames);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[columnNames.length];

		// Column 1 : Name (text)
		TextCellEditor textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[0] = textEditor;

		// Column 2 : Location (text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[1] = textEditor;

		// Column 3 : Condition (Text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[2] = textEditor;

		// Assign the cell editors to the viewer 
		tableViewer.setCellEditors(editors);
		// Set the cell modifier for the viewer
		tableViewer.setCellModifier(new FileTemplateCellModifier(this));
		
	}
	
	public void addTableSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
	}
	
	public void addTableInputListener(TableInputListener listener) {
		this.tableInputListener = listener;
	}
	
	public FileTemplate getSelectedTemplate() {
		FileTemplate template = (FileTemplate) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		return template;
	}
	
	public java.util.List getColumnNames() {
		return Arrays.asList(columnNames);
	}
	
	public FileTemplateList getTemplateList() {
		return templateList;
	}
	
	public FileTemplate[] getTemplates() {
		FileTemplate[] templates = new FileTemplate[templateList.getTemplates().size()];
		templates = (FileTemplate[])templateList.getTemplates().toArray(templates);
		return templates;
	}
	
	public void removeSelectedTemplate() {
		FileTemplate template = (FileTemplate) ((IStructuredSelection) 
				tableViewer.getSelection()).getFirstElement();
		if (template != null) {
			templateList.removeTemplate(template);
		} 				
	}
	
	public void addNewTemplate() {
		templateList.addTemplate();
	}
	
	public void setInput(FileTemplateList templateList) {
		this.templateList = templateList;
		tableViewer.setInput(templateList);
	}
	
	private class TemplateListContentProvider implements IStructuredContentProvider, ITemplateListViewer {
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			if (newInput != null)
				((FileTemplateList) newInput).addChangeListener(this);
			if (oldInput != null)
				((FileTemplateList) oldInput).removeChangeListener(this);
		}

		public void dispose() {
			templateList.removeChangeListener(this);
		}

		// Return the templates as an array of Objects
		public Object[] getElements(Object parent) {
			return templateList.getTemplates().toArray();
		}

		/* (non-Javadoc)
		 * @see ITemplateListViewer#addVariable(FileTemplate)
		 */
		public void addTemplate(FileTemplate template) {
			tableViewer.add(template);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(templateList);
		}

		/* (non-Javadoc)
		 * @see ITemplateListViewer#removeVariable(FileTemplate)
		 */
		public void removeTemplate(FileTemplate template) {
			tableViewer.remove(template);		
			if(tableInputListener != null)
				tableInputListener.tableInputModified(templateList);
		}

		/* (non-Javadoc)
		 * @see ITemplateListViewer#updateVariable(FileTemplate)
		 */
		public void updateTemplate(FileTemplate template) {
			tableViewer.update(template, null);	
			if(tableInputListener != null)
				tableInputListener.tableInputModified(templateList);
			
		}
	}
	
	private class TemplateListLabelProvider extends LabelProvider implements ITableLabelProvider {

		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			String result = "";
			FileTemplate template = (FileTemplate) element;
			switch (columnIndex) {
				case 0:  
					result = template.getName();
					break;
				case 1 :
					result = template.getLocation();
					break;
				case 2 :
					result = template.getCondition();
					break;
				default :
					break; 	
			}
			return result;
		}
		
	}
}
