/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui;

import java.util.Arrays;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;


public class VariablesTableViewerPanel extends Composite {
	
	protected Table table;
	protected TableViewer tableViewer;
	protected Button newVariableButton;
	protected Button removeVariableButton;
	protected VariableList variableList;
	protected TableInputListener tableInputListener;
	
	protected static final String NAME_COL = "name";
	protected static final String DISPLAY_NAME_COL = "displayName";
	protected static final String TYPE_COL = "type";
	protected static final String DEFAULT_VALUE_COL = "defaultValue";
	protected static final String OPTIONS_COL = "options";
	protected static final String REQUIRED_COL = "required";
	
	protected String[] columnNames;
	
	protected final String[] requiredOptions = {"true", "false"};
	
	
	public VariablesTableViewerPanel(Composite parent, int style) {
		super(parent, style);
		initialize();
		setLayout(new GridLayout());
		createChildComponents();
		
	}
	
	protected void initialize() {
		variableList = new VariableList();
		//variableList.initialize();
		columnNames = new String[] {NAME_COL, DISPLAY_NAME_COL, TYPE_COL, DEFAULT_VALUE_COL, OPTIONS_COL, REQUIRED_COL};
	}
	
	protected void createChildComponents() {
		
		createTable();
		createTableViewer();
		
		tableViewer.setContentProvider(new VariableListContentProvider());
		tableViewer.setLabelProvider(new VariableListLabelProvider());
				
		tableViewer.setInput(variableList);

		
	}
		
	protected void createTable() {
		
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
				SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
		
		table = new Table(this, style);
						
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
				
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// 1st column 
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);		
		column.setText("Name");
		column.setWidth(100);
		
		// 2nd column 
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Display Name");
		column.setWidth(130);
						
		// 3rd column 
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Type");
		column.setWidth(100);
				
		// 4th column 
		column = new TableColumn(table, SWT.LEFT, 3);
		column.setText("Default Value");
		column.setWidth(100);
		
		// 5th column
		column = new TableColumn(table, SWT.LEFT, 4);
		column.setText("Options");
		column.setWidth(100);
		
		// 6th column
		column = new TableColumn(table, SWT.LEFT, 5);
		column.setText("Required");
		column.setWidth(70);
		
		
	}
	
	protected void createTableViewer() {
		
		tableViewer = new TableViewer(table); 
		tableViewer.setUseHashlookup(true);
		
		tableViewer.setColumnProperties(columnNames);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[columnNames.length];

		// Column 1 : Name (text)
		TextCellEditor textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[0] = textEditor;

		// Column 2 : Display Name (text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[1] = textEditor;

		// Column 3 : Type (Combo Box) 
		editors[2] = new ComboBoxCellEditor(table, variableList.getTypeOptions(), SWT.READ_ONLY);

		// Column 4 : Default Value (Text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[3] = textEditor;
		
		// Column 5 : Options (Text)
		textEditor = new TextCellEditor(table);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[4] = textEditor;
		
		// Column 6 : Required (Checkbox)
		editors[5] = new ComboBoxCellEditor(table, requiredOptions, SWT.READ_ONLY);
		
		// Assign the cell editors to the viewer 
		tableViewer.setCellEditors(editors);
		// Set the cell modifier for the viewer
		tableViewer.setCellModifier(new VariableCellModifier(this));
		
	}
	
	public java.util.List getColumnNames() {
		return Arrays.asList(columnNames);
	}
	
	public String[] getChoices(String property) {
		if (TYPE_COL.equals(property))
			return variableList.getTypeOptions();  
		else if(REQUIRED_COL.equals(property))
			return requiredOptions;
		return new String[]{};
	}
	
	public VariableList getVariableList() {
		return variableList;
	}
	
	public Variable[] getVariables() {
		Variable[] vars = new Variable[variableList.getVariables().size()];
		vars = (Variable[])variableList.getVariables().toArray(vars);
		return vars;
	}
	
	public Variable getSelectedVariable() {
		Variable var = (Variable) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		return var;
	}
	
	public void removeSelectedVariable() {
		Variable var = (Variable) ((IStructuredSelection) tableViewer.getSelection()).getFirstElement();
		if (var != null) {
			variableList.removeVariable(var);
		} 				
	}
	
	public void clear() {
		variableList.removeAll();
	}
	
	public void addNewVariable() {
		variableList.addVariable();
	}
	
	public void addVariable(Variable var) {
		variableList.addVariable(var);
	}
	
	public void setInput(VariableList varList) {
		variableList = varList;
		tableViewer.setInput(varList);
	}
	
	public void addTableSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
		
	}
	
	public void addTableInputListener(TableInputListener listener) {
		this.tableInputListener = listener;
	}
	
	private class VariableListContentProvider implements IStructuredContentProvider, IVariableListViewer {
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			if(oldInput != null && newInput!= null && newInput.equals(oldInput))
				return;
			if (newInput != null)
				((VariableList) newInput).addChangeListener(this);
			if (oldInput != null)
				((VariableList) oldInput).removeChangeListener(this);
			
		}

		public void dispose() {
			variableList.removeChangeListener(this);
		}

		// Return the variables as an array of Objects
		public Object[] getElements(Object parent) {
			return variableList.getVariables().toArray();
		}

		/* (non-Javadoc)
		 * @see IVariableListViewer#addVariable(Variable)
		 */
		public void addVariable(Variable var) {
			tableViewer.add(var);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(variableList);
		}

		/* (non-Javadoc)
		 * @see IVariableListViewer#removeVariable(Variable)
		 */
		public void removeVariable(Variable var) {
			tableViewer.remove(var);
			if(tableInputListener != null)
				tableInputListener.tableInputModified(variableList);
		}

		/* (non-Javadoc)
		 * @see IVariableListViewer#updateVariable(Variable)
		 */
		public void updateVariable(Variable var) {
			tableViewer.update(var, null);	
			if(tableInputListener != null)
				tableInputListener.tableInputModified(variableList);
			
		}
	}
	
	private class VariableListLabelProvider extends LabelProvider implements ITableLabelProvider {

		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			String result = "";
			Variable var = (Variable) element;
			switch (columnIndex) {
				case 0:  
					result = var.getName();
					break;
				case 1 :
					result = var.getDisplayName();
					break;
				case 2 :
					result = var.getType();
					break;
				case 3 :
					result = var.getDefaultValue();
					break;
				case 4 :
					result = var.getOptions();
					break;
				case 5 :
					result = var.isRequired()+"";
				default :
					break; 	
			}
			return result;
		}
		
	}
	
}
