/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui.wizards;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.atf.generator.model.ApplicationWizardInfo;
import org.eclipse.atf.generator.model.FileTemplateInfo;
import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.Utils;
import org.eclipse.atf.generator.model.VariableInfo;
import org.eclipse.atf.generator.model.VariableViewPageInfo;
import org.eclipse.atf.generator.ui.FileTemplate;
import org.eclipse.atf.generator.ui.FileTemplateList;
import org.eclipse.atf.generator.ui.PageTableViewerPanel;
import org.eclipse.atf.generator.ui.TableInputListener;
import org.eclipse.atf.generator.ui.TemplatesTableViewerPanel;
import org.eclipse.atf.generator.ui.Variable;
import org.eclipse.atf.generator.ui.VariableList;
import org.eclipse.atf.generator.ui.VariableRefsList;
import org.eclipse.atf.generator.ui.VariableRefsTableViewerPanel;
import org.eclipse.atf.generator.ui.VariablesTableViewerPanel;
import org.eclipse.atf.generator.ui.ViewPage;
import org.eclipse.atf.generator.ui.ViewPageList;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;


public class ApplicationWizardWizardPage extends WizardPage implements SelectionListener, ModifyListener, Listener {

	private TabFolder tabFolder;
	private TabItem variablesTabItem;
	private TabItem viewsTabItem;
	private TabItem templatesTabItem;
	private VariablesTableViewerPanel variablesTablePanel;
	private PageTableViewerPanel pagesTablePanel;
	private VariableRefsTableViewerPanel refsTablePanel;
	private TemplatesTableViewerPanel templatesTablePanel;
	private Hashtable pageVariablesHash = new Hashtable();
			
	private Button newVariableButton;
	private Button removeVariableButton;
	private Button newPageButton;
	private Button removePageButton;
	private Button newVariableRefButton;
	private Button removeVariableRefButton;
	private Button newTemplateButton;
	private Button removeTemplateButton;
	private Text templateContentText;
	private Text nameText;
	private Text descText;
	private Text classNameText;
	//private Text packageNameText;
	private String wizardName;
	private String description;
	private String className;
	private String packageName;
	
	boolean controlsInitted = false;
		
	private MrCleanToolingInfo dataModel;
	
	public ApplicationWizardWizardPage(MrCleanToolingInfo dataModel) {
		super("New Application Wizard");
		setTitle("Application Wizard generation"); //$NON-NLS-1$
	    setDescription("Specify information to create an application wizard");
		
	    ApplicationWizardInfo info = dataModel.getApplicationWizardInfo();
	    if(info == null) {
	    	info = new ApplicationWizardInfo();
	    	dataModel.setApplicationWizardInfo(info);
	    }
	    this.dataModel = dataModel;
	}
	
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setFont(parent.getFont());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
				
		createApplicationWizardControls(composite);
				
		setControl(composite);
		Dialog.applyDialogFont(composite);
		initializeControlContents();

	}
	
	public void initializeControlContents() {
		if(!controlsInitted) {
			
			createInitialVariables();
			createInitialViewVariableRefs();
			updateControlContents();						
						
			controlsInitted = true;
		}
	}
	
	protected void updateControlContents() {
		initializeWizardData();
		//if(packageName != null)
		//	packageNameText.setText(packageName);
		if(wizardName != null)			
			nameText.setText(wizardName);
		if(className != null)	
			classNameText.setText(className);
		if(description != null)	
			descText.setText(description);
		updateDataModel();
	}
	
	protected void initializeWizardData() {
		//if(packageName == null) {
			String uiProject = dataModel.getUiProjectName();
			if(uiProject != null && !uiProject.equals("")) {
				packageName = uiProject + ".wizard";
			}
		//}
		String techName = dataModel.getCamelCasedTechnologyName();
		if(techName != null && !techName.equals("")) {
			//if(wizardName == null)
				wizardName = Utils.capitalize(techName + " Application");
			//if(className == null)
				className = Utils.capitalize(techName + "ApplicationWizard");
			//if(description == null)
				description = "Creates new " + Utils.capitalize(techName) + " application";
		}
						
	}
	
	protected void createInitialVariables() {
		
		Variable var = new Variable("project");
		var.setDisplayName("Project");
		var.setType("Project");
		var.setEditable(false);
		var.setRequired(true);
		variablesTablePanel.addVariable(var);
		
		var = new Variable("appName");
		var.setDisplayName("Application Name");
		var.setEditable(false);
		var.setRequired(true);
		variablesTablePanel.addVariable(var);
		
		var = new Variable("appRoot");
		var.setDisplayName("Application Root");
		var.setDefaultValue("${appName}");
		var.setEditable(false);
		var.setRequired(true);
		variablesTablePanel.addVariable(var);
		
		var = new Variable("ajaxRoot");
		var.setDisplayName("AJAX Artifacts Root");
		var.setType("RelativePath");
		var.setEditable(false);
		var.setRequired(true);
		var.setOtherAttributes( "from" , "${_thisLocation}" );
		variablesTablePanel.addVariable(var);
	}
	
	protected void createInitialViewVariableRefs() {
		
		ViewPageList pageList = pagesTablePanel.getPageList();
		ArrayList pages = pageList.getPages();
		if(pages.size() >= 1) {
			Variable[] availableVariables = getAvailableVariables();
			VariableRefsList varList = new VariableRefsList(availableVariables);
			VariableList variableList = variablesTablePanel.getVariableList();
			ArrayList variables = variableList.getVariables();
			for(int i=0; i<variables.size(); i++) {
				Variable var = (Variable)variables.get(i);
				if(!var.isEditable())
					varList.addVariable(var);
			}
			
			pageVariablesHash.put(pages.get(0), varList);
			refsTablePanel.setInput(varList);
		}
		
	}
	
	protected void createApplicationWizardControls(Composite parent) {
		
		Composite topComposite = new Composite(parent, SWT.NONE);
		topComposite.setFont(parent.getFont());
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		topComposite.setLayout(layout);
		topComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label nameLabel = new Label(topComposite, SWT.NONE);
		nameLabel.setText("Name:");
		nameText = new Text(topComposite, SWT.BORDER);
		nameText.addModifyListener(this);
		nameText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel = new Label(topComposite, SWT.NONE);
		descLabel.setText("Description:");
		descText = new Text(topComposite, SWT.BORDER);
		descText.addModifyListener(this);
		descText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label classLabel = new Label(topComposite, SWT.NONE);
		classLabel.setText("Class:");
		classNameText = new Text(topComposite, SWT.BORDER);
		classNameText.addModifyListener(this);
		classNameText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		//Label packageLabel = new Label(topComposite, SWT.NONE);
		//packageLabel.setText("Package:");
		//packageNameText = new Text(topComposite, SWT.BORDER);
		//packageNameText.addModifyListener(this);
		//packageNameText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		tabFolder = new TabFolder(parent, SWT.TOP);
		tabFolder.setLayoutData(new GridData(GridData.FILL_BOTH));
		tabFolder.addSelectionListener(this);
				
		variablesTabItem = new TabItem(tabFolder, SWT.NONE);
		variablesTabItem.setText("Variables");
		viewsTabItem = new TabItem(tabFolder, SWT.NONE);
		viewsTabItem.setText("Views");
		
		templatesTabItem = new TabItem(tabFolder, SWT.NONE);
		templatesTabItem.setText("Templates");
		
		Composite variablesComposite = createVariablesComposite(tabFolder);
		Composite viewsComposite = createViewsComposite(tabFolder);
		Composite templatesComposite = createTemplatesComposite(tabFolder);
				
		variablesTabItem.setControl(variablesComposite);
		viewsTabItem.setControl(viewsComposite);
		templatesTabItem.setControl(templatesComposite);
		
	}
	
	protected Composite createVariablesComposite(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Composite labelComposite = new Composite(composite, SWT.NONE);
		labelComposite.setFont(parent.getFont());
		labelComposite.setLayout(new GridLayout());
		labelComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel = new Label(labelComposite, SWT.NONE);
		descLabel.setText("Define the variables which will represent the inputs in the generated application wizard.");
		
		Composite buttonComposite = new Composite(composite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite.setLayout(gridLayout);
		buttonComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newVariableButton = new Button(buttonComposite, SWT.PUSH);
		newVariableButton.setText("New Variable");
		newVariableButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		newVariableButton.addSelectionListener(this);
		
		removeVariableButton = new Button(buttonComposite, SWT.PUSH);
		removeVariableButton.setText("Remove Variable");
		removeVariableButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		removeVariableButton.setEnabled(false);
		removeVariableButton.addSelectionListener(this);
		
		variablesTablePanel = new VariablesTableViewerPanel(composite, SWT.NONE);
		variablesTablePanel.setLayoutData (new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH));
		variablesTablePanel.addTableSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				variableSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
			
		});
		
		variablesTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
				
		return composite;
	}
	
	protected Composite createTemplatesComposite(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Composite labelComposite = new Composite(composite, SWT.NONE);
		labelComposite.setFont(parent.getFont());
		labelComposite.setLayout(new GridLayout());
		labelComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel = new Label(labelComposite, SWT.NONE);
		descLabel.setText("Define the templates for the files to be generated by the application wizard.");
		
		Composite buttonComposite = new Composite(composite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite.setLayout(gridLayout);
		buttonComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newTemplateButton = new Button(buttonComposite, SWT.PUSH);
		newTemplateButton.setText("New Template");
		newTemplateButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		newTemplateButton.addSelectionListener(this);
		
		removeTemplateButton = new Button(buttonComposite, SWT.PUSH);
		removeTemplateButton.setText("Remove Template");
		removeTemplateButton.setEnabled(false);
		removeTemplateButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		removeTemplateButton.addSelectionListener(this);
		
		templatesTablePanel = new TemplatesTableViewerPanel(composite, SWT.NONE);
		templatesTablePanel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH));
		templatesTablePanel.addTableSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				templateSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
			
		});
		
		templatesTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
		
		Composite labelComposite2 = new Composite(composite, SWT.NONE);
		labelComposite2.setFont(parent.getFont());
		labelComposite2.setLayout(new GridLayout());
		labelComposite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label contentLabel = new Label(labelComposite2, SWT.NONE);
		contentLabel.setText("Enter template content below. You may reference predefined variables using the ${varName} notation.");
		
		templateContentText = new Text(composite, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
		templateContentText.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH));
		templateContentText.addModifyListener(this);
		templateContentText.setEnabled(false);
		
		return composite;
	}
	
	protected Composite createViewsComposite(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Composite labelComposite = new Composite(composite, SWT.NONE);
		labelComposite.setFont(parent.getFont());
		labelComposite.setLayout(new GridLayout());
		labelComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel = new Label(labelComposite, SWT.NONE);
		descLabel.setText("Define the view pages which will contain references to the previously defined variables.");
		
		Composite buttonComposite = new Composite(composite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite.setLayout(gridLayout);
		buttonComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newPageButton = new Button(buttonComposite, SWT.PUSH);
		newPageButton.setText("New Page");
		newPageButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		newPageButton.addSelectionListener(this);
		
		removePageButton = new Button(buttonComposite, SWT.PUSH);
		removePageButton.setText("Remove Page");
		removePageButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		removePageButton.setEnabled(false);
		removePageButton.addSelectionListener(this);
		
		pagesTablePanel = new PageTableViewerPanel(composite, SWT.NONE);
		GridData gData = new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		gData.heightHint = 100;
		pagesTablePanel.setLayoutData (gData);
		pagesTablePanel.addTableSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				pageSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
			
		});
		
		pagesTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
		
		Composite labelComposite2 = new Composite(composite, SWT.NONE);
		labelComposite2.setFont(parent.getFont());
		labelComposite2.setLayout(new GridLayout());
		labelComposite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel2 = new Label(labelComposite2, SWT.NONE);
		descLabel2.setText("Define the variables to show in the selected view page.");
		
		Composite buttonComposite2 = new Composite(composite, SWT.NONE);
		gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite2.setLayout(gridLayout);
		buttonComposite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newVariableRefButton = new Button(buttonComposite2, SWT.PUSH);
		newVariableRefButton.setText("New Variable Reference");
		newVariableRefButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		newVariableRefButton.addSelectionListener(this);
		
		removeVariableRefButton = new Button(buttonComposite2, SWT.PUSH);
		removeVariableRefButton.setText("Remove Variable Reference");
		removeVariableRefButton.setEnabled(false);
		removeVariableRefButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		removeVariableRefButton.addSelectionListener(this);
		
		refsTablePanel = new VariableRefsTableViewerPanel(composite, SWT.NONE, getAvailableVariables());
		gData = new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		gData.heightHint = 100;
		refsTablePanel.setLayoutData(gData);
		refsTablePanel.addTableSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				variableRefSelectionChanged();
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		refsTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
		
				
		return composite;
	}
	
	protected Variable[] getAvailableVariables() {
		ApplicationWizardInfo appWizardInfo = dataModel.getApplicationWizardInfo();
		if(appWizardInfo == null)
			return new Variable[0];
		VariableInfo[] varInfos = appWizardInfo.getVariableRefs();
		if(varInfos == null)
			return new Variable[0];
		Variable[] vars = new Variable[varInfos.length];
		for(int i=0; i<varInfos.length; i++) {
			Variable var = new Variable();
			var.setName(varInfos[i].getName());
			var.setDisplayName(varInfos[i].getDisplayName());
			var.setDefaultValue(varInfos[i].getDefaultValue());
			var.setType(varInfos[i].getType());
			vars[i] = var;
		}
		return vars;
	}
	
	protected void pageSelectionChanged() {
		
		ViewPage page = pagesTablePanel.getSelectedViewPage();
		showPageInfo(page);
		if(page != null && !page.isRequired())
			removePageButton.setEnabled(true);
		else
			removePageButton.setEnabled(false);
	}
	
	protected void variableRefSelectionChanged() {
		
		Variable var = refsTablePanel.getSelectedVariable();
		if(var != null && !var.isRequired())
			removeVariableRefButton.setEnabled(true);
		else
			removeVariableRefButton.setEnabled(false);
	}
	
	protected void variableSelectionChanged() {
		
		Variable var = variablesTablePanel.getSelectedVariable();
		if(var != null && !var.isRequired())
			removeVariableButton.setEnabled(true);
		else
			removeVariableButton.setEnabled(false);
	}
	
	protected void showPageInfo(ViewPage page) {
		VariableList varList = (VariableList)pageVariablesHash.get(page);
		Variable[] availableVariables = getAvailableVariables();
		if(varList == null) {
			varList = new VariableRefsList(availableVariables);
			pageVariablesHash.put(page, varList);
		}
		
		refsTablePanel.setInput(varList);
	}
	
	protected void templateSelectionChanged() {
		FileTemplate template = templatesTablePanel.getSelectedTemplate();
		showTemplateInfo(template);
		if(template != null)
			removeTemplateButton.setEnabled(true);
		else
			removeTemplateButton.setEnabled(false);
	}
	
	protected void showTemplateInfo(FileTemplate template) {
		if(template != null) {
			templateContentText.setEnabled(true);
			templateContentText.setText(template.getContent());
		}
	}
	
	public void setVisible(boolean visible) {
		if(visible)
			updateControlContents();
		super.setVisible(visible);
	}
	
	protected boolean validatePage() {
		if(!validateName()) {
			setErrorMessage("Valid name must be specified.");
			return false;
		} else if(!validateDescription()) {
			setErrorMessage("Valid description must be specified.");
			return false;
		} else if(!validateWizardClass()) {
			setErrorMessage("Valid class name must be specified.");
			return false;
		} else if(!validateVariables()) {
			return false;
		} else if(!validatePages()) {
			return false;
		} else if(!validateTemplates()) {
			return false;
		}
		setErrorMessage(null);
		return true;
	}
	
	protected boolean validateName() {
		if(wizardName == null || wizardName.equals(""))
			return false;
		return true;
	}
	
	protected boolean validateDescription() {
		if(description == null || description.equals(""))
			return false;
		return true;
	}
	
	protected boolean validateWizardClass() {
		if(className == null || className.equals(""))
			return false;
		return true;
	}
	
	protected boolean validateVariables() {
		if(variablesTablePanel == null)
			return true;
		Variable[] vars = variablesTablePanel.getVariables();
		ArrayList names = new ArrayList();
		for(int i=0; i<vars.length; i++) {
			if(vars[i].getName() == null || vars[i].getName().equals("")) {
				setErrorMessage("All variables must have a name defined.");
				return false;
			} else if(names.contains(vars[i].getName())) {
				setErrorMessage("Duplicate variable definition, \""+vars[i].getName()+"\".");
				return false;
			} 
			names.add(vars[i].getName());
			if(vars[i].getType() == null || vars[i].getType().equals("")) {
				setErrorMessage("Variable \""+vars[i].getName()+"\" does not have a type defined.");
				return false;
			} else if(vars[i].getDisplayName() == null || vars[i].getDisplayName().equals("")) {
				setErrorMessage("Variable \""+vars[i].getName()+"\" does not have a display name defined.");
				return false;
			}
		}
		return true;
	}
	
	protected boolean validatePages() {
		
		if(pagesTablePanel == null)
			return true;
		ViewPage[] pages = pagesTablePanel.getPages();
		for(int i=0; i<pages.length; i++) {
			if(pages[i].getName() == null || pages[i].getName().equals("")) {
				setErrorMessage("All view pages must have a name defined.");
				return false;
			} else if(pages[i].getTitle() == null || pages[i].getTitle().equals("")) {
				setErrorMessage("View page \""+pages[i].getName()+"\" does not have a title defined.");
				return false;
			} else if(pages[i].getDescription() == null || pages[i].getDescription().equals("")) {
				setErrorMessage("View page \""+pages[i].getName()+"\" does not have a description defined.");
				return false;
			}
			VariableList varList = (VariableList)pageVariablesHash.get(pages[i]);
			if(varList == null || varList.getVariables().size() == 0) {
				setErrorMessage("View page \""+pages[i].getName()+"\" must have at least one variable reference defined.");
				return false;
			} else {
				ArrayList varRefs = varList.getVariables();
				ArrayList varRefNames = new ArrayList();
				for(int j=0; j<varRefs.size(); j++) {
					Variable varRef = (Variable)varRefs.get(j);
					if(varRefNames.contains(varRef.getName())) {
						setErrorMessage("Redundant variable reference, \""+varRef.getName()+"\", defined for view page \""+pages[i].getName()+"\".");
						return false;
					}
					varRefNames.add(varRef.getName());
				}
			}
			
		}
		return true;
	}
	
	protected boolean validateTemplates() {
		
		if(templatesTablePanel == null)
			return true;
		FileTemplate[] templates = templatesTablePanel.getTemplates();
		for(int i=0; i<templates.length; i++) {
			if(templates[i].getName() == null || templates[i].getName().equals("")) {
				setErrorMessage("All templates must have a name.");
				return false;
			} 
			FileTemplate template = templatesTablePanel.getSelectedTemplate();
			if(template != null && template.getContent().equals("")) {
				setErrorMessage("Template \""+ template.getName()+"\" does not have any content.");
				return false;
			}
		}
		return true;
	}
	
	public void setPageComplete(boolean complete) {
        super.setPageComplete(complete);
        if(complete) {
        	updateDataModel();
		}
    }
	
	public void updateDataModel() {
		
		initializeWizardData();
		
		ApplicationWizardInfo wizardInfo = dataModel.getApplicationWizardInfo();
		wizardInfo.setName(wizardName);
		wizardInfo.setDescription(description);
		wizardInfo.setClassName(className);
		wizardInfo.setPackageName(packageName);
		
		updateVariablesModel();
		updateViewsModel();
		updateTemplatesModel();
	}
	
	protected void updateVariablesModel() {
		if(variablesTablePanel == null)
			return;
		VariableList variableList = variablesTablePanel.getVariableList();
		ArrayList variables = variableList.getVariables();
		if(variables.size() > 0) {
			ArrayList variableInfos = new ArrayList();
			for(int i=0; i<variables.size(); i++) {
				VariableInfo info = new VariableInfo();
				Variable var = (Variable)variables.get(i);
				info.setName(var.getName());
				info.setDisplayName(var.getDisplayName());
				info.setType(var.getType());
				info.setDefaultValue(var.getDefaultValue());
				info.setOptions(var.getOptions());
				info.setRequired(var.isRequired());
				info.setOtherAttributes( var.getOtherAttributes() );
				variableInfos.add(info);
			}
			VariableInfo[] vi = new VariableInfo[variableInfos.size()];
			vi = (VariableInfo[])variableInfos.toArray(vi);
			dataModel.getApplicationWizardInfo().setVariableRefs(vi);
		}
	}
	
	protected void updateViewsModel() {
		if(pagesTablePanel == null)
			return;
		ViewPageList pageList = pagesTablePanel.getPageList();
		ArrayList pages = pageList.getPages();
		if(pages.size() > 0) {
			ArrayList viewPageInfos = new ArrayList();
			for(int i=0; i<pages.size(); i++) {
				VariableViewPageInfo viewPageInfo = new VariableViewPageInfo();
				ViewPage page = (ViewPage)pages.get(i);
				viewPageInfo.setName(page.getName());
				viewPageInfo.setTitle(page.getTitle());
				viewPageInfo.setDescription(page.getDescription());
				VariableRefsList varList = (VariableRefsList)pageVariablesHash.get(page);
				if(varList != null) {
					ArrayList vars = varList.getVariables();
					ArrayList variableInfos = new ArrayList();
					for(int j=0; j<vars.size(); j++) {
						Variable var = (Variable)vars.get(j);
						VariableInfo varInfo = new VariableInfo();
						varInfo.setName(var.getName());
						varInfo.setDisplayName(var.getDisplayName());
						varInfo.setType(var.getType());
						varInfo.setDefaultValue(var.getDefaultValue());
						varInfo.setOtherAttributes( var.getOtherAttributes() );
						variableInfos.add(varInfo);
					}
					VariableInfo[] vi = new VariableInfo[variableInfos.size()];
					vi = (VariableInfo[])variableInfos.toArray(vi);
					viewPageInfo.setVariableRefs(vi);
				}
				viewPageInfos.add(viewPageInfo);
			}
			VariableViewPageInfo[] vvpi = new VariableViewPageInfo[viewPageInfos.size()];
			vvpi = (VariableViewPageInfo[])viewPageInfos.toArray(vvpi);
			dataModel.getApplicationWizardInfo().setViewPages(vvpi);
		}
	}
	
	protected void updateTemplatesModel() {
		if(templatesTablePanel == null)
			return;
		FileTemplateList templateList = templatesTablePanel.getTemplateList();
		ArrayList templates = templateList.getTemplates();
		if(templates.size() > 0) {
			ArrayList templateInfos = new ArrayList();
			for(int i=0; i<templates.size(); i++) {
				FileTemplateInfo templateInfo = new FileTemplateInfo();
				FileTemplate template = (FileTemplate)templates.get(i);
				templateInfo.setName(template.getName());
				templateInfo.setProjectRelativeLocation(template.getLocation());
				templateInfo.setTemplate(template.getContent());
				if(template.getCondition() != null && !template.getCondition().equals("")) {
					templateInfo.addCreationCondition(template.getCondition());
				}
				templateInfos.add(templateInfo);
			}
			FileTemplateInfo [] fti = new FileTemplateInfo[templateInfos.size()];
			fti = (FileTemplateInfo[])templateInfos.toArray(fti);
			dataModel.getApplicationWizardInfo().setFileTemplates(fti);
		}
	}
	
	public void widgetSelected(SelectionEvent e) {
		Widget w = e.widget;
		if(w.equals(newVariableButton)) {
			addNewVariable();
		} else if(w.equals(removeVariableButton)) {
			removeSelectedVariable();
		} else if(w.equals(newPageButton)) {
			addNewPage();
		} else if(w.equals(removePageButton)) {
			removeSelectedPage();
		} else if(w.equals(tabFolder)) {
			tabFolderSelectionChanged(e);
		} else if(w.equals(newVariableRefButton)) {
			addNewVariableRef();
		} else if(w.equals(removeVariableRefButton)) {
			removeSelectedVariableRef();
		} else if(w.equals(newTemplateButton)) {
			addNewTemplate();
		} else if(w.equals(removeTemplateButton)) {
			removeSelectedTemplate();
		}
		setPageComplete(validatePage());

	}
	
	protected void tabFolderSelectionChanged(SelectionEvent e) {
		updateDataModel();
		if(e.item.equals(viewsTabItem)) {
			refsTablePanel.setAvailableVariables(getAvailableVariables());
		}
	}
	
	protected void addNewVariable() {
		variablesTablePanel.addNewVariable();
	}
		
	protected void addNewPage() {
		pagesTablePanel.addNewPage();
	}
	
	protected void addNewVariableRef() {
		
		refsTablePanel.addNewVariable();
	}
	
	protected void addNewTemplate() {
		templatesTablePanel.addNewTemplate();
	}
	
	protected void removeSelectedVariable() {
		variablesTablePanel.removeSelectedVariable();
	}
	
	protected void removeSelectedPage() {
		ViewPage page = pagesTablePanel.getSelectedViewPage();
		//Remove page from table
		pagesTablePanel.removeSelectedPage();
		//Remove associated variables from hash
		pageVariablesHash.remove(page);
		//Clear the variable refs table
		refsTablePanel.clear();
	}
	
	protected void removeSelectedVariableRef() {
		Variable var = refsTablePanel.getSelectedVariable();
		//Remove var from the table
		refsTablePanel.removeSelectedVariable();
		//Remove the var from the list of vars associated with the selected page
		ViewPage selectedPage = pagesTablePanel.getSelectedViewPage();
		if(selectedPage != null) {
			VariableRefsList refsList = (VariableRefsList)pageVariablesHash.get(selectedPage);
			if(refsList != null) {
				refsList.removeVariable(var);
				pageVariablesHash.put(selectedPage, refsList);
			}
		}
	
		if(refsTablePanel.getSelectedVariable() == null )
			removeVariableRefButton.setEnabled(false);
	}
	
	protected void removeSelectedTemplate() {
		templatesTablePanel.removeSelectedTemplate();
		templateContentText.setText("");
		templateContentText.setEnabled(false);
		if(templatesTablePanel.getTemplates().length == 0)
			removeTemplateButton.setEnabled(false);
				
	}

	public void widgetDefaultSelected(SelectionEvent e) {}

	public void modifyText(ModifyEvent e) {
		Widget w = e.widget;
		if(w.equals(templateContentText)) {
			FileTemplate selected = templatesTablePanel.getSelectedTemplate();
			if(selected != null)
				selected.setContent(((Text)w).getText());
		} else if(w.equals(nameText)) {
			wizardName = nameText.getText();
		} else if(w.equals(descText)) {
			description = descText.getText();
		} else if(w.equals(classNameText)) {
			className = classNameText.getText();
		} 		
		//else if(w.equals(packageNameText)) {
		//	packageName = packageNameText.getText();
		//}
		
		setPageComplete(validatePage());

	}

	public void handleEvent(Event event) {
		setPageComplete(validatePage());
		
	}

	

}
