/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui.wizards;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.atf.generator.GeneratorPlugin;
import org.eclipse.atf.generator.action.CorePluginCreationOperation;
import org.eclipse.atf.generator.action.RuntimePluginCreationOperation;
import org.eclipse.atf.generator.action.UiPluginCreationOperation;
import org.eclipse.atf.generator.model.FacetInfo;
import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.NatureInfo;
import org.eclipse.atf.generator.model.RuntimeInfo;
import org.eclipse.atf.generator.model.Utils;
import org.eclipse.atf.generator.model.VariableInfo;
import org.eclipse.atf.natures.codegen.ApplicationWizardGenerator;
import org.eclipse.atf.natures.codegen.BuilderGenerator;
import org.eclipse.atf.natures.codegen.FacetInstallDelegateGenerator;
import org.eclipse.atf.natures.codegen.FacetUninstallDelegateGenerator;
import org.eclipse.atf.natures.codegen.NatureGenerator;
import org.eclipse.atf.natures.codegen.RuntimeHandlerGenerator;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.codegen.util.CodeGenUtil;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.pde.internal.ui.PDEPlugin;
import org.eclipse.pde.internal.ui.wizards.IProjectProvider;
import org.eclipse.pde.internal.ui.wizards.plugin.PluginFieldData;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;


public class NewATFToolingWizard extends Wizard implements INewWizard {

	private IWorkbench mWorkbench;
	private IStructuredSelection mSelection;
	private ApplicationWizardWizardPage applicationWizardPage;
	private SnippetsWizardPage snippetsWizardPage;
	private RuntimeArtifactsWizardPage runtimeArtifactsWizardPage;
	private MrCleanToolingInfo dataModel = new MrCleanToolingInfo();
	private static String DIALOG_SETTINGS_SECTION = "MrCleanToolingWizard";
	
	public NewATFToolingWizard() {
		super();
	    setNeedsProgressMonitor(true);
	    setDefaultPageImageDescriptor(GeneratorPlugin.getImageDescriptor("icons/pbwiz75x66.gif"));
	    setWindowTitle("AJAX Personality Builder"); //$NON-NLS-1$
	}
	
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		mWorkbench = workbench;
	    mSelection = selection;
	    IDialogSettings workbenchSettings = GeneratorPlugin.getDefault().getDialogSettings();
	    IDialogSettings wizardSettings = workbenchSettings.getSection(DIALOG_SETTINGS_SECTION); //$NON-NLS-1$
	    if (wizardSettings == null) 
	    	wizardSettings = workbenchSettings.addNewSection(DIALOG_SETTINGS_SECTION); //$NON-NLS-1$
	    setDialogSettings(wizardSettings);
	}
	
	/*
	 * @see Wizard#addPages
	 */
	public void addPages()
	{
	  super.addPages();
	  runtimeArtifactsWizardPage = new RuntimeArtifactsWizardPage(mWorkbench, mSelection, dataModel);
	  applicationWizardPage = new ApplicationWizardWizardPage(dataModel);
	  snippetsWizardPage = new SnippetsWizardPage(dataModel);
	  
	  addPage(runtimeArtifactsWizardPage);
	  addPage(applicationWizardPage);
	  addPage(snippetsWizardPage);
	  	  
	}
	
	/**
	   * Obtains the user input from the wizard pages, creates a <code>Config</code>
	   * object that contains all this information, and delegates the source code
	   * generation to a <code>JETGateway</code> object. Finally, the generated
	   * Java source code file is opened in an editor.
	   */
	  protected void finishPage(IProgressMonitor monitor) throws InterruptedException, CoreException
	  {
	    //Update the data model
		runtimeArtifactsWizardPage.updateDataModel();
		runtimeArtifactsWizardPage.storeDialogSettings();
	    applicationWizardPage.updateDataModel();
	    snippetsWizardPage.updateDataModel();
	    
	    dataModel.getNatureInfo().setContainingPluginId(dataModel.getCoreProjectName());
	    dataModel.getBuilderInfo().setContainingPluginId(dataModel.getCoreProjectName());
	    
	    //Create the plugin projects
	    createRuntimePluginProject(monitor);
		createCorePluginProject(monitor);
		createUIPluginProject(monitor);
	    	    
	  }
	  
	  private void createRuntimePluginProject(IProgressMonitor monitor) throws CoreException {
		  
		  IProjectProvider fProjectProvider = new IProjectProvider() {
				public String getProjectName() {
					return dataModel.getRuntimeProjectName();
				}
				public IProject getProject() {
					return ResourcesPlugin.getWorkspace().getRoot().getProject(dataModel.getRuntimeProjectName());
				}
				public IPath getLocationPath() {
					return ResourcesPlugin.getWorkspace().getRoot().getLocation();
				}
		  };
		  
		  PluginFieldData fPluginData = new PluginFieldData();
		  fPluginData.setClassname(dataModel.getRuntimeProjectName() + "." + dataModel.getRuntimeProjectPluginClass());
		  fPluginData.setDoGenerateClass(true);
		  fPluginData.setHasBundleStructure(true);
		  fPluginData.setId(dataModel.getRuntimeProjectName());
		  fPluginData.setLegacy(false);
		  fPluginData.setLibraryName("runtime.jar");
		  fPluginData.setName(Utils.capitalize(dataModel.getTechnologyName()) + "Runtime Plugin");
		  fPluginData.setOutputFolderName("bin");
		  fPluginData.setProvider("");
		  fPluginData.setRCPApplicationPlugin(false);
		  fPluginData.setSimple(false);
		  fPluginData.setSourceFolderName("src");
		  fPluginData.setTargetVersion("3.1");
		  fPluginData.setVersion("1.0.0");
		  fPluginData.setUIPlugin(true);
		  
		  try {
			  getContainer().run(false, true,
					  new RuntimePluginCreationOperation(fPluginData, fProjectProvider, dataModel, null));
		  } catch(InvocationTargetException e) {
			  PDEPlugin.logException(e);
		  } catch(InterruptedException ex) {
			  
		  }
		  
	  }
	  
	  private void createCorePluginProject(IProgressMonitor monitor) throws CoreException {
		  
		  IProjectProvider fProjectProvider = new IProjectProvider() {
				public String getProjectName() {
					return dataModel.getCoreProjectName();
				}
				public IProject getProject() {
					return ResourcesPlugin.getWorkspace().getRoot().getProject(dataModel.getCoreProjectName());
				}
				public IPath getLocationPath() {
					return ResourcesPlugin.getWorkspace().getRoot().getLocation();
				}
		  };
		  
		  if(dataModel.getRuntimeInfo() == null) {
			  NatureInfo natureInfo = dataModel.getNatureInfo();
			  RuntimeInfo runtimeInfo = new RuntimeInfo( dataModel.getCamelCasedTechnologyName() );
			  
			  dataModel.setRuntimeInfo(runtimeInfo);
		  }
		  
		  if(dataModel.getFacetInfo() == null) {
			  FacetInfo info = new FacetInfo(dataModel.getCamelCasedTechnologyName());
			  info.setPackageName(info.getPackageName());
			  dataModel.setFacetInfo(info);
	      }
		  
		  
		  PluginFieldData fPluginData = new PluginFieldData();
		  fPluginData.setClassname(dataModel.getCoreProjectName() + "." + dataModel.getCoreProjectPluginClass());
		  fPluginData.setDoGenerateClass(true);
		  fPluginData.setHasBundleStructure(true);
		  fPluginData.setId(dataModel.getCoreProjectName());
		  fPluginData.setLegacy(false);
		  fPluginData.setLibraryName("core.jar");
		  fPluginData.setName(Utils.capitalize(dataModel.getTechnologyName()) + "Core Plugin");
		  fPluginData.setOutputFolderName("bin");
		  fPluginData.setProvider("");
		  fPluginData.setRCPApplicationPlugin(false);
		  fPluginData.setSimple(false);
		  fPluginData.setSourceFolderName("src");
		  fPluginData.setTargetVersion("3.1");
		  fPluginData.setVersion("1.0.0");
		  fPluginData.setUIPlugin(true);
		  
		  try {
			  getContainer().run(false, true,
					  new CorePluginCreationOperation(fPluginData, fProjectProvider, dataModel, null));
		  } catch(InvocationTargetException e) {
			  PDEPlugin.logException(e);
		  } catch(InterruptedException ex) {
			  
		  }
		  
		  generateCoreClasses(monitor);
	  }
	  	  
	  private void createUIPluginProject(IProgressMonitor monitor) throws CoreException {
		  
		  IProjectProvider fProjectProvider = new IProjectProvider() {
				public String getProjectName() {
					return dataModel.getUiProjectName();
				}
				public IProject getProject() {
					return ResourcesPlugin.getWorkspace().getRoot().getProject(dataModel.getUiProjectName());
				}
				public IPath getLocationPath() {
					return ResourcesPlugin.getWorkspace().getRoot().getLocation();
					
				}
		  };
		  
		  //Set project variable filter classes if necessary
		  VariableInfo[] vars = dataModel.getApplicationWizardInfo().getVariableRefs();
		  for(int i=0; i<vars.length; i++) {
			  if(vars[i].getType().equals("Project")) {
				 String[] filterClasses = new String[2];
				 filterClasses[0] = "org.eclipse.atf.templates.variable.util.SimpleNameFilter";
				 filterClasses[1] = "org.eclipse.atf.ui.wizard.facet.FacetedProjectNatureFilter";
				 vars[i].setFilterClasses(filterClasses);
			  }
		  }
		  
		  PluginFieldData fPluginData = new PluginFieldData();
  		  
		  fPluginData.setClassname(dataModel.getUiProjectName() + "." + dataModel.getUiProjectPluginClass());
		  fPluginData.setDoGenerateClass(true);
		  fPluginData.setHasBundleStructure(true);
		  fPluginData.setId(dataModel.getUiProjectName());
		  fPluginData.setLegacy(false);
		  fPluginData.setLibraryName("ui.jar");
		  fPluginData.setName(Utils.capitalize(dataModel.getTechnologyName()) + " UI Plugin");
		  fPluginData.setOutputFolderName("bin");
		  fPluginData.setProvider("");
		  fPluginData.setRCPApplicationPlugin(false);
		  fPluginData.setSimple(false);
		  fPluginData.setSourceFolderName("src");
		  fPluginData.setTargetVersion("3.1");
		  fPluginData.setVersion("1.0.0");
		  fPluginData.setUIPlugin(true);
		  
		  try {
			  getContainer().run(false, true,
					  new UiPluginCreationOperation(fPluginData, fProjectProvider, dataModel, null));
		  } catch(InvocationTargetException e) {
			  PDEPlugin.logException(e);
		  } catch(InterruptedException ex) {
			  
		  }
		  
		  generateUIClasses(monitor);
	  }
	  	  
	  private void generateUIClasses(IProgressMonitor monitor) throws CoreException {
		  
		  generateApplicationWizardClasses(monitor);
		  		  
	  }
	  
	  private void generateCoreClasses(IProgressMonitor monitor) throws CoreException {
		  
		  generateRuntimeHandlerClass(monitor);
		  generateNatureClasses(monitor);
		  generateBuilderClasses(monitor);
		  generateFacetClasses(monitor);
	  }
	  
	  private void generateRuntimeHandlerClass(IProgressMonitor monitor) throws CoreException {
  
		// first create the file, all the necessary directories and get a handle
		String targetFolder = dataModel.getCoreProjectName() + "/src";
		String packageName = dataModel.getRuntimeInfo().getHandlerPackageName();
		String className = dataModel.getRuntimeInfo().getHandlerClassName()	+ ".java";

		IPath outputPath = new Path(targetFolder + "/"+ packageName.replace('.', '/'));
		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
		IPath localPath = null; // uses default
		IContainer container = CodeGenUtil.EclipseUtil.findOrCreateContainer(
					outputPath, true, localPath, subMonitor);
		
		if (container == null) {
			IStatus status = new Status(Status.ERROR,
					GeneratorPlugin.PLUGIN_ID, Status.OK,
					"Cound not find or create container for package "
							+ packageName + " in " + targetFolder, null);
			throw new CoreException(status);
		}
		  
		// make sure we can write to it
		IFile targetFile = container.getFile(new Path(className));
		if (targetFile.isReadOnly()) {
			targetFile.getResourceAttributes().setReadOnly(false);
		}

		// generate the contents
		RuntimeHandlerGenerator generator = RuntimeHandlerGenerator.create(null);
		String contents = generator.generate(dataModel);

		InputStream contentsIS = new ByteArrayInputStream( contents.getBytes() );
		  
		if (targetFile.exists())
		{
			targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
		}
		else
		{
			targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
		}
	  }
	  	  
	  private void generateNatureClasses(IProgressMonitor monitor) throws CoreException {
		//first create the file, all the necessary directories and get a handle
		String targetFolder = dataModel.getCoreProjectName() + "/src";
		String packageName = dataModel.getNatureInfo().getPackageName();
		String className = dataModel.getNatureInfo().getClassName()	+ ".java";
		
		IPath outputPath = new Path(targetFolder + "/"+ packageName.replace('.', '/'));
		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
		IPath localPath = null; // uses default
		IContainer container = CodeGenUtil.EclipseUtil.findOrCreateContainer(
					outputPath, true, localPath, subMonitor);
		
		if (container == null) {
			IStatus status = new Status(Status.ERROR,
					GeneratorPlugin.PLUGIN_ID, Status.OK,
					"Cound not find or create container for package "
						+ packageName + " in " + targetFolder, null);
			throw new CoreException(status);
		}
		  
		// make sure we can write to it
		IFile targetFile = container.getFile(new Path(className));
		if (targetFile.isReadOnly()) {
			targetFile.getResourceAttributes().setReadOnly(false);
		}
		
		// generate the contents
		NatureGenerator generator = NatureGenerator.create(null);
		String contents = generator.generate(dataModel.getNatureInfo());
		
		InputStream contentsIS = new ByteArrayInputStream( contents.getBytes() );
		  
		if (targetFile.exists())
		{
			targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
		}
		else
		{
			targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
		}		    	    
	  }

	  private void generateFacetClasses(IProgressMonitor monitor) throws CoreException {
		//first create the file, all the necessary directories and get a handle
		String targetFolder = dataModel.getCoreProjectName() + "/src";
		String packageName = dataModel.getFacetInfo().getPackageName();
		
		//generate the install delegate
		String className = dataModel.getFacetInfo().getInstallDelegateClass() + ".java";
		
		IPath outputPath = new Path(targetFolder + "/"+ packageName.replace('.', '/'));
		IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
		IPath localPath = null; // uses default
		IContainer container = CodeGenUtil.EclipseUtil.findOrCreateContainer(
					outputPath, true, localPath, subMonitor);
		
		if (container == null) {
			IStatus status = new Status(Status.ERROR,
					GeneratorPlugin.PLUGIN_ID, Status.OK,
					"Cound not find or create container for package "
						+ packageName + " in " + targetFolder, null);
			throw new CoreException(status);
		}
		  
		// make sure we can write to it
		IFile targetFile = container.getFile(new Path(className));
		if (targetFile.isReadOnly()) {
			targetFile.getResourceAttributes().setReadOnly(false);
		}
		
		// generate the contents
		FacetInstallDelegateGenerator generator = FacetInstallDelegateGenerator.create(null);
		String contents = generator.generate(dataModel);
		
		InputStream contentsIS = new ByteArrayInputStream( contents.getBytes() );
		  
		if (targetFile.exists())
		{
			targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
		}
		else
		{
			targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
		}
		
		//generate uninstall delegate
//		generate the install delegate
		className = dataModel.getFacetInfo().getUninstallDelegateClass() + ".java";		  
		// make sure we can write to it
		targetFile = container.getFile(new Path(className));
		if (targetFile.isReadOnly()) {
			targetFile.getResourceAttributes().setReadOnly(false);
		}
		
		// generate the contents
		FacetUninstallDelegateGenerator generator2 = FacetUninstallDelegateGenerator.create(null);
		contents = generator2.generate(dataModel);
		
		contentsIS = new ByteArrayInputStream( contents.getBytes() );
		  
		if (targetFile.exists())
		{
			targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
		}
		else
		{
			targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
		}		  		  
	  }
	  	  
	  private void generateBuilderClasses(IProgressMonitor monitor) throws CoreException{
		  
			// first create the file, all the necessary directories and get a handle
			String targetFolder = dataModel.getCoreProjectName() + "/src";
			String packageName = dataModel.getBuilderInfo().getPackageName();
			String className = dataModel.getBuilderInfo().getClassName() + ".java";

			IPath outputPath = new Path(targetFolder + "/"+ packageName.replace('.', '/'));
			IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
			IPath localPath = null; // uses default
			IContainer container = CodeGenUtil.EclipseUtil.findOrCreateContainer(
						outputPath, true, localPath, subMonitor);
			
			if (container == null) {
				IStatus status = new Status(Status.ERROR,
						GeneratorPlugin.PLUGIN_ID, Status.OK,
						"Cound not find or create container for package "
								+ packageName + " in " + targetFolder, null);
				throw new CoreException(status);
			}
			  
			// make sure we can write to it
			IFile targetFile = container.getFile(new Path(className));
			if (targetFile.isReadOnly()) {
				targetFile.getResourceAttributes().setReadOnly(false);
			}

			// generate the contents
			BuilderGenerator generator = BuilderGenerator.create(null);
			String contents = generator.generate(dataModel.getBuilderInfo());

			InputStream contentsIS = new ByteArrayInputStream( contents.getBytes() );
			  
			if (targetFile.exists())
			{
				targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
			}
			else
			{
				targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
			}	  
	  }
	  
	  private void generateApplicationWizardClasses(IProgressMonitor monitor) throws CoreException {
		  
			// first create the file, all the necessary directories and get a handle
			String targetFolder = dataModel.getUiProjectName() + "/src";
			String packageName = dataModel.getApplicationWizardInfo().getPackageName();
			String className = dataModel.getApplicationWizardInfo().getClassName() + ".java";

			IPath outputPath = new Path(targetFolder + "/"+ packageName.replace('.', '/'));
			IProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
			IPath localPath = null; // uses default
			IContainer container = CodeGenUtil.EclipseUtil.findOrCreateContainer(
						outputPath, true, localPath, subMonitor);
			
			if (container == null) {
				IStatus status = new Status(Status.ERROR,
						GeneratorPlugin.PLUGIN_ID, Status.OK,
						"Cound not find or create container for package "
								+ packageName + " in " + targetFolder, null);
				throw new CoreException(status);
			}
			  
			// make sure we can write to it
			IFile targetFile = container.getFile(new Path(className));
			if (targetFile.isReadOnly()) {
				targetFile.getResourceAttributes().setReadOnly(false);
			}

			// generate the contents
			ApplicationWizardGenerator generator = ApplicationWizardGenerator.create(null);
			String contents = generator.generate(dataModel);

			InputStream contentsIS = new ByteArrayInputStream( contents.getBytes() );
			  
			if (targetFile.exists())
			{
				targetFile.setContents(contentsIS, true, true, new SubProgressMonitor(monitor, 1));
			}
			else
			{
				targetFile.create(contentsIS, true, new SubProgressMonitor(monitor, 1));
			}	  
	  }
	  	 	  
	  /*
	   * @see Wizard#performFinish
	   */
	  public boolean performFinish()
	  {
	    IWorkspaceRunnable op = new IWorkspaceRunnable()
	      {
	        public void run(IProgressMonitor monitor) throws CoreException, OperationCanceledException
	        {
	          try
	          {
	            finishPage(monitor);
	          }
	          catch (InterruptedException e)
	          {
	            throw new OperationCanceledException(e.getMessage());
	          }
	        }
	      };

	    try
	    {
	      getContainer().run(false, true, new WorkbenchRunnableAdapter(op));
	    }
	    catch (InvocationTargetException e)
	    {
	      handleFinishException(getShell(), e);
	      return false;
	    }
	    catch (InterruptedException e)
	    {
	      return false;
	    }
	    return true;
	  }
	  
	  /**
	   * An <code>IRunnableWithProgress</code> that adapts
	   * <code>IWorkspaceRunnable</code> so that it can be executed inside
	   * <code>IRunnableContext</code>.<code>OperationCanceledException</code>
	   * thrown by the apapted runnabled are caught and rethrown as a
	   * <code>InterruptedException</code>.
	   */
	  private static class WorkbenchRunnableAdapter implements IRunnableWithProgress
	  {

	    private IWorkspaceRunnable fWorkspaceRunnable;

	    public WorkbenchRunnableAdapter(IWorkspaceRunnable runnable)
	    {
	      fWorkspaceRunnable = runnable;
	    }

	    /*
	     * @see IRunnableWithProgress#run(IProgressMonitor)
	     */
	    public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException
	    {
	      try
	      {
	        JavaCore.run(fWorkspaceRunnable, monitor);
	      }
	      catch (OperationCanceledException e)
	      {
	        throw new InterruptedException(e.getMessage());
	      }
	      catch (CoreException e)
	      {
	        throw new InvocationTargetException(e);
	      }
	    }
	  }

	  protected void handleFinishException(Shell shell, InvocationTargetException e)
	  {
	    String exceptionMessage = "Error when generating the Typesafe Enumeration Class.";
	    if(e.getCause() != null && e.getCause().getMessage() != null)
	    {
	      exceptionMessage = e.getCause().getMessage();
	    }
	    else if(e.getMessage() != null)
	    {
	    	exceptionMessage = e.getMessage();
	    }
	    
	    IStatus status = new Status(IStatus.ERROR, GeneratorPlugin.getPluginId(), IStatus.ERROR, exceptionMessage, e);
	    GeneratorPlugin.log(status);

	    String title = WizardMessages.getString("Wizard.op_error.title"); //$NON-NLS-1$
	    String message = WizardMessages.getString("Wizard.op_error.message"); //$NON-NLS-1$
	    ErrorDialog.openError(shell, title, message, status);
	  }
	  
}
